/******************************************************************************************
    Copyright (C) 1997-2019 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in the
    Software without restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the
    following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/
#include "spc_typ.h"

#define MAX_angleDHA 90

double   spc_typ::ProjectOnPlane(atm_typ H, atm_typ X, atm_typ C0, atm_typ C1)
// For atom H above a plane P defined by atoms X, C0 and C1, return the
// distance of the Length of the projection of H onto that plane from X.
{
        // vec_typ      U,A,P,V1,V2;

        vec_typ V1 = MkVector(AtomX(C0)-AtomX(X),AtomY(C0)-AtomY(X),AtomZ(C0)-AtomZ(X));
        vec_typ V2 = MkVector(AtomX(C1)-AtomX(X),AtomY(C1)-AtomY(X),AtomZ(C1)-AtomZ(X));
        vec_typ Vd = GetVectorProduct(V1,V2);
        vec_typ VH = MkVector(AtomX(H),AtomY(H),AtomZ(H));
        vec_typ VX = MkVector(AtomX(X),AtomY(X),AtomZ(X));
        vec_typ Vp = GetVectorProjectionOntoPlane(Vd,VX,VH);
#if 0
        atm_typ Virtual =VirtualAtom(Vp->x + AtomX(X),Vp->y + AtomY(X),Vp->z + AtomZ(X));
        // PutAtom(stderr,Virtual);
        // PutAtom(stderr,X);
#endif
        double d= sqrt(GetVectorLength(Vp));
        delete Vp; delete V1; delete V2; delete Vd; delete VX; delete VH;
        return d;
}

void	spc_typ::DebugCH_Pi(atm_typ Donor,atm_typ Accept,double d_HA,double d_DA,
		double angle_DHA,Int4 num, a_type AB)
{
	Int4 resA = ResAtom(Accept);
	Int4 resD = ResAtom(Donor);
	char	chnA=AtomChain(Accept),chnD=AtomChain(Donor);
	char  na,aA,aD=GetResidueAtom(Donor, AB, 0, &na);
	aA=GetResidueAtom(Accept, AB, 0, &na);
	fprintf(stderr, " CH-pi: %c%d%c vs %c%d%c; d_HA=%.3lf; d_DA=%.3lf; angle_DHA=%.1lf;",
		AlphaChar(aD,AB),resD,chnD,AlphaChar(aA,AB),resA,chnA,d_HA,d_DA,angle_DHA);
	fprintf(stderr," (%d hits)\n",num);
}

BooLean	spc_typ::OkayPiParameters(double  d_HpX,double d_DV,double angle_DHV)
{
	if(d_HpX <= 1.2 && d_DV <= 4.5 && angle_DHV >= 110){
		if(0) fprintf(stderr,"   --> d_HpX = %.2lf; d_DV=%.2lf; angle=%.1lf %c\n",
		   d_HpX,d_DV,angle_DHV,'A');
		return TRUE;
	} return FALSE;
}

#define MAX_NUM_ATOMS_FOR_VIRTUAL 30

Int4    spc_typ::FindOtherPiHbonds(FILE *fp,res_typ ResD,res_typ ResA,Int4 C,Int4 C2,
          float dmax, char color, pdb_typ P, double &D_HpX, double &D_DV,
	  double &Angle_DHV, char &Typ, char Mode)
// find potential aromatic pi-H interactions.
{
	Int4	n,a,d,r,r2,res,res2,numhits;
	atm_typ	H,Donor,Accept,atom[MAX_NUM_ATOMS_FOR_VIRTUAL + 3];
	double	d_HA,d_DA,angle_DHA,Ave_d_DA,Ave_d_HA,minDist=BigDist;
	BooLean	na;
	FILE	*efp=P->fp_stderr;
	a_type	AB=AminoAcidAlphabetPDB(P);
	BooLean	comments=FALSE;
	
	if(C > P->nchains || C<1 || C2 >P->nchains || C2<1){
		pdb_error("FindOtherPiHbondsPDB( ): input error"); 
	}
	if(efp == stderr) comments=TRUE; else comments=FALSE;
	comments=TRUE; // waiting to debug this option;
	if(ResD == ResA) return 0;
	char    aa = GetResidueAtom(AtomResidue(1,ResA),AB,P->nA,&na);
	aa = AlphaChar(aa,AB);
#if 1	// New 10/15/02: afn.
	char    aa2 = GetResidueAtom(AtomResidue(1,ResD),AB,P->nA,&na);
	aa2 = AlphaChar(aa2,AB);
#endif
	if(na || !strchr("NQDER",aa)) return 0;
// !(aa == 'N' || aa == 'Q' || aa == 'D' || aa == 'E' || aa == 'R')) return 0;
	atm_typ	Near,Far;
	double	d_near,d_far,angle_HNF;
	for(n=0,d = 1; d <= ResidueAtomNumber(ResD); d++){
	     if(!ResidueAtomHydrogens(d,ResD)) continue;
	     Donor = AtomResidue(d,ResD);
// PutAtom(stderr,Donor);
	     res=ResAtom(Donor);
	     if(P->seq[C]) r = P->seq[C][res]; else r = 0;
	     for(Int4 h=1; (H=ResidueAtomHydrogen(d,h,ResD)); h++){
		Ave_d_DA=Ave_d_HA=0.0;
// PutAtom(stderr,H);
		d_near=1000.0; d_far=0.0;
		Int4 numNO=0;
		atm_typ atomNO[MAX_NUM_ATOMS_FOR_VIRTUAL + 3];
	        for(numhits=0,a = 1; a <= ResidueAtomNumber(ResA); a++){
	     	  Accept = AtomResidue(a,ResA);
		  if(HydrogenAtom(Accept)) continue;
		  if(!SideAtom(Accept)) continue;
		  if(IsAtom(" CB ",Accept)) continue;
		  if(strchr("EQR",aa) && IsAtom(" CG ",Accept)) continue;
		  if((aa == 'R') && IsAtom(" CD ",Accept)) continue;
		  res2 = ResAtom(Accept);
		  if(C==C2 && abs(res2 - res) < 1) continue;
		  if(P->seq[C2]) r2 = P->seq[C2][res2]; else r2 = 0;
		  d_HA = DistanceAtoms(H,Accept);
		  angle_DHA = CalcAngle(Donor,H,Accept);
		  d_DA = DistanceAtoms(Donor,Accept);
		  if(d_HA <= dmax && d_DA > d_HA) 
		  {
// PutAtom(stderr,Accept);
		    Ave_d_DA += d_DA; Ave_d_HA += d_HA;
		    if(d_near > d_HA){ d_near = d_HA; Near=Accept; }
		    if(d_far < d_HA){ d_far = d_HA; Far=Accept; }
		    // aveX += ...
		    numhits++;	// heavy atom hit;
		    if(numhits < MAX_NUM_ATOMS_FOR_VIRTUAL) atom[numhits] = Accept;
		    else {
			for(Int4 x=1; x <= numhits; x++) PutAtom(stderr,atom[x]);
			print_error("Input error: too many overlapping atoms in coordinate file");
		    } 
		    if(!CarbonAtom(Accept)){
		      numNO++;	// oxygen or nitrogen hit;
		      if(numNO < MAX_NUM_ATOMS_FOR_VIRTUAL) atomNO[numNO] = Accept;
		      else {
			for(Int4 x=1; x <= numNO; x++) PutAtom(stderr,atomNO[x]);
			print_error("Input error: too many overlapping atoms in coordinate file");
		      } 
		    }
		  }
		}
		if(numNO < 2) continue;
		Int4 z;
		atm_typ Virtual=0;
		double	d_DV,d_HV,angle_DHV,d_HpX;
		if(numhits > 1){
		   angle_HNF = CalcAngle(H,Near,Far);
		   Ave_d_DA = Ave_d_DA/(double) numhits;
		   Ave_d_HA = Ave_d_HA/(double) numhits;
		}
		if(numhits > 2){
		   Virtual = AverageAtom(numhits,atom);
		   d_DV=DistanceAtoms(Donor,Virtual);
		   d_HV=DistanceAtoms(H,Virtual);
		   angle_DHV = CalcAngle(Donor,H,Virtual);
		   d_HpX = ProjectOnPlane(H,Virtual,atomNO[1],atomNO[2]);
	// fprintf(stderr,"d_HpX = %.2f\n",d_HpX);
#if 1	// AFN: 9-30-2016; for SIPRES program...
		  minDist=MINIMUM(double,d_HV,minDist);
#endif
		  if(D_HpX > d_HpX) D_HpX= d_HpX;
		  if(D_DV > d_DV) D_DV = d_DV;
		  if(Angle_DHV > angle_DHV) Angle_DHV = angle_DHV;
		}
		// < 90 --> within plane.
		if((numhits >= 3 && (aa == 'N' || aa == 'Q' || aa == 'D' || aa == 'E') ||
			numhits >= 4 && (aa == 'R')) 
			&& d_HpX <= 1.2 && d_DV <= 4.5 && angle_DHV >= 120){
		    atm_typ	A,B;
		    Int4 rA,rB;
		    unsigned char rcA;

#if 0
		    if(fp){
			fprintf(fp,"found H-bond (%d): (%.2f:%.2f A).\n",numhits,
				Ave_d_DA,Ave_d_HA);
		    	fprintf(fp,"Virtual:\n"); PutAtom(fp,Virtual);
		    	fprintf(fp,"Donor:\n"); PutAtom(fp,Donor);
		    	fprintf(fp,"Hydrogen:\n"); PutAtom(fp,H);
		   	fprintf(fp,"Accept:\n");
		        for(z=1; z <= numhits && z < 10; z++) PutAtom(fp,atom[z]);
		    }
#endif
#if 1			// adding more information... afn: 5/5/2022.
		    if(OxygenAtom(Donor)) Typ='O';
		    else if(NitrogenAtom(Donor)) Typ='N';
		    else if(CarbonAtom(Donor)) Typ='C';
		    else if(SulfurAtom(Donor)) Typ='S';
		    else Typ='X';
#endif

		    A=Donor;B=Accept; rcA=r; 
		    rcA=r;
		    rA=ResAtom(A); rB=ResAtom(B);
#if 1	// fix negative position problem: AFN 8_26_2015.
	if(rA < 0){ if(Virtual) NilAtom(Virtual); continue; }
#endif
		    char str[9],*str_ptr = AtomName(A);
		    if(isspace(str_ptr[0])) str_ptr++;
		    for(z=0; str_ptr[z] && !isspace(str_ptr[z]); z++){
				if(isalpha(str_ptr[z])) str[z]=tolower(str_ptr[z]);
				else str[z]=str_ptr[z];
		    } str[z]=0;
#if 1	// ***************** Find hydrogen too.. **********************************
		    Int4 zz;
		    str_ptr = AtomName(H);
		    if(isspace(str_ptr[0])) str_ptr++;
		    for(str[z]='-',z++,zz=0; str_ptr[zz] && !isspace(str_ptr[zz]); zz++,z++){
				if(isalpha(str_ptr[zz])) str[z]=tolower(str_ptr[zz]);
				else str[z]=str_ptr[zz];
		    } str[z]=0;
#endif	//*************************************************************************
		    if(fp){
#if 1	// New format for mstarc....AFN:4/24/2018.
		   if(Mode){
		      if(IsWaterAtom(A)){ if(Virtual) NilAtom(Virtual); continue; }
		      if(AtomChain(A) != ' '){ fprintf(fp,"%c%d:%c",AlphaChar(rcA,AB),rA,AtomChain(A)); }
		      else fprintf(fp,"%c%d",AlphaChar(rcA,AB),rA);
		      if(strcmp("o",str)==0) strcpy(str,"c=o");
		      fprintf(fp,"_%s",str);
		      if(AtomChain(B) != ' '){
			   fprintf(fp,"\t%c%d:%c_pi\t%.2f\t%.2f\t%.1f\n",aa,rB,AtomChain(B),d_DV,d_HV,angle_DHA);
		      } else fprintf(fp, "\t%c%d_pi\t%.2f\t%.2f\t%.1f\n",aa,rB,d_DV,d_HV,angle_DHA);
			 
                    } else 
#endif
		    {
		     if(IsWaterAtom(A)){ fprintf(fp,"#HOH%d\t// to ",rA); }
		     else {
			 if(IsHeteroAtom(A)) fprintf(fp,"// ");
			 if(AtomChain(A) != ' '){ fprintf(fp,"%c%d:%c",AlphaChar(rcA,AB),rA,AtomChain(A)); }
			 else fprintf(fp,"%c%d",AlphaChar(rcA,AB),rA);
			 if(strcmp("o",str)==0) strcpy(str,"c=o");
			 if(comments) fprintf(fp,"_%s.X\t// XH-pi to ",str);
			 else fprintf(fp,"_%s.X\n",str);
		     }
		     if(comments){
		        if(AtomChain(B) != ' '){
			   fprintf(fp,"\t// to %c%d:%c (%d);d=%.2f(%.2f)A; HNF = %.1f,DHA = %.1f;DHV = %.f.\n",
				aa,rB,AtomChain(B),numhits, d_DV,d_HV,angle_HNF,angle_DHA,angle_DHV);
		        } else fprintf(fp, "\t// to %c%d (%d);d=%.2f(%.2f) A; HNF = %.1f,DHA = %.1f;DHV = %.f.\n",
				aa,rB,numhits,d_DV,d_HV,angle_HNF,angle_DHA,angle_DHV);
		      } else fprintf(fp,"\n");
		     }
		    }
		    n++;
		} else if(numhits >= 3){
		    // if(Virtual) NilAtom(Virtual);
#if 0
		    fprintf(stderr,"Donor:\n"); PutAtom(stderr,Donor);
		    fprintf(stderr,"Hydrogen:\n"); PutAtom(stderr,H);
		    fprintf(stderr,"Accept:\n");
		    for(z=1; z <= numhits && z < 10; z++) PutAtom(stderr,atom[z]);
#endif
#if 0
		    if(numhits > 2){	// PROBLEM WITH str!!
			fprintf(fp,
			 "// %c%d(%c%d_%s): near hit (%d)! (%.2f:%.2f A); d_HpX = %.2f\n",
				aa,res2,aa2,res,str,numhits,Ave_d_DA,Ave_d_HA,d_HpX);
		    }
#else
		    if(Mode==0 && numhits > 2){
			if(fp) fprintf(fp,
			  "# %c%d.W // XH-pi near hit to %c%d (%d)! (%.2f:%.2f A); d_HpX=%.2f\n",
				aa,res2,aa2,res,numhits,Ave_d_DA,Ave_d_HA,d_HpX);
		    }
#endif
		} if(Virtual) NilAtom(Virtual);
	     }
	}
	return n;
}


Int4    spc_typ::FindAromaticHbonds(FILE *fp, res_typ ResD, res_typ ResA,Int4 C, Int4 C2,
		float dmax, char color, pdb_typ P, double &bst_HpX,double &bst_DV,
		double &bst_DHV, char &Typ,char Mode)
// find potential aromatic pi-H interactions.
{
	Int4	n,a,d,r,r2,resA,resD,numhits;
	atm_typ	H,Donor,Accept,atom[20];
	double	d_HA,d_DA,angle_DHA,Ave_d_DA,Ave_d_HA;
	BooLean	na;
	a_type	AB=AminoAcidAlphabetPDB(P);
	FILE	*efp=P->fp_stderr;
	BooLean	comments=FALSE;
	
	if(C > nChainsPDB(P) || C<1 || C2 > nChainsPDB(P) || C2<1){
		pdb_error("FindAromaticHbondsPDB( ): input error"); 
	} Typ=0;
	if(efp == stderr) comments=TRUE; else comments=FALSE;
	comments=TRUE; // waiting to debug this option;
	if(dmax == 0.0) dmax = 4.5; //  for aromatic CH-pi interactions
	if(ResD == ResA) return 0;
	char    aa = GetResidueAtom(AtomResidue(1,ResA),AB,P->nA,&na);
	aa = AlphaChar(aa,AB);
	char    aa2 = GetResidueAtom(AtomResidue(1,ResD),AB,P->nA,&na);
	aa2 = AlphaChar(aa2,AB);
	if(na || strchr("HFYW",aa) == NULL) return 0;
	atm_typ	Near,Far;
	double	d_near,d_far,angle_HNF;
	atm_typ	atom6[10];
	Int4	x;
	for(n=0,d = 1; d <= ResidueAtomNumber(ResD); d++){
	     if(!ResidueAtomHydrogens(d,ResD)) continue;
	     Donor = AtomResidue(d,ResD);
//PutAtom(stderr,Donor);
	     if(mode=='s' && SideAtom(Donor)) continue;
	     resD=ResAtom(Donor);
	     if(P->maxres[C] < resD) r = 0;
	     else {
	       if(P->seq[C]) r = P->seq[C][resD];  else r = 0;
	     }
	     for(Int4 h=1; (H=ResidueAtomHydrogen(d,h,ResD)); h++){
		Ave_d_DA=Ave_d_HA=0.0;
//PutAtom(stderr,H);
		d_near=1000.0; d_far=0.0;
		mh_type	mH=Mheap(10,3);
	        for(numhits=0,a = 1; a <= ResidueAtomNumber(ResA); a++){
	     	  Accept = AtomResidue(a,ResA);
		  if(HydrogenAtom(Accept)) continue;
		  if(!SideAtom(Accept)) continue;
		  if(IsAtom(" CB ",Accept) || IsAtom(" OH ",Accept)) continue;
		  resA = ResAtom(Accept);
		  if(C==C2 && abs(resA - resD) < 1) continue;
		  if(P->seq[C2]) r2 = P->seq[C2][resA];  
		  else r2 = 0;
		  d_HA = DistanceAtoms(H,Accept);
		  if(d_HA > dmax) continue;
		  d_DA = DistanceAtoms(Donor,Accept);
		  if(d_DA <= d_HA) continue;
		  angle_DHA = CalcAngle(Donor,H,Accept);
		  // if(d_HA <= 3.5 && d_DA > d_HA && angle_DHA >= 90)
		  // if(d_HA <= 3.5)
		  // if(d_HA <= dmax)
		  // if(d_HA <= dmax && d_DA > d_HA && angle_DHA >= 90)
#if 0	//===================== DEBUG =====================
if(resA==352 && resD==359)	//  && NitrogenAtom(Donor))
{ this->DebugCH_Pi(Donor,Accept,d_HA,d_DA,angle_DHA,numhits,AB); }
#endif	//================================================
		  if(angle_DHA >= 90) {
// PutAtom(stderr,Accept);
		    Ave_d_DA += d_DA;
		    Ave_d_HA += d_HA;
		    if(d_near > d_HA){ d_near = d_HA; Near=Accept; }
		    if(d_far < d_HA){ d_far = d_HA; Far=Accept; }
		    // aveX += ...
		    numhits++;	// heavy atom hit;
		    if(numhits < 10) atom[numhits] = Accept;
		    x=InsertMheap((keytyp)d_HA,mH);	// i = 
		    if(x != 0) atom6[x]=Accept;
		  }
		}
		Int4 z;
		atm_typ Virtual=0;
		double	d_DV=0.0,d_HV=0.0,angle_DHV=0.0,d_HpX=0.0;
		if(numhits > 1){
		   angle_HNF = CalcAngle(H,Near,Far);
		   Ave_d_DA = Ave_d_DA/(double) numhits;
		   Ave_d_HA = Ave_d_HA/(double) numhits;
		}
		if(numhits > 4){
#if 1
		  // Add dheap or mheap of size six.
		  // use to compute centroid atom from both five and six member rings
		  // for trp. Currently missing some CH-pi bonds to five member ring...
		  // This will also allow us to use a larger value for dmax.
		  numhits = ItemsInMheap(mH);
		  for(a=0; !EmptyMheap(mH); ){
			x=DelMinMheap(mH); a++; atom[a]=atom6[x]; 
			// if(resA==350){ fprintf(stderr,"%d: ",a); PutAtom(stderr,atom[a]); }
		  } // a++; atom[a]=0;
#endif
		   Virtual = AverageAtom(numhits,atom);
		   d_DV=DistanceAtoms(Donor,Virtual);
		   d_HV=DistanceAtoms(H,Virtual);
		   angle_DHV = CalcAngle(Donor,H,Virtual);
		   d_HpX = ProjectOnPlane(H,Virtual,atom[1],atom[2]);
		   NilAtom(Virtual);
		} // < 90 --> within plane.
		NilMheap(mH);
#if 0	//====================== DEBUG... ========================
	else {
		  Virtual = AverageAtom(numhits,atom);
		  d_DV=DistanceAtoms(Donor,Virtual);
		  d_HV=DistanceAtoms(H,Virtual);
		  angle_DHV = CalcAngle(Donor,H,Virtual);
		  if(resA==97 && resD==10 && NitrogenAtom(Donor)){
			char  na,aA,aD=GetResidueAtom(Donor, AB, 0, &na);
			aA=GetResidueAtom(Accept, AB, 0, &na);
			fprintf(stderr," Aro..H-N- ===> %c%d vs %c%d\n",
			  AlphaChar(aD,AB),resD,AlphaChar(aA,AB),resA);
			fprintf(stderr,"numhits=%d\n",numhits);
			fprintf(stderr,
			  " d_HpX=%lf; d_DV=%lf; d_HV=%lf; angle_DHV=%lf\n",
			   d_HpX,d_DV,d_HV,angle_DHV);
		  }
	}
#endif	//========================================================
		if(((numhits >= 5 && (aa == 'H' || aa == 'W')) ||
			(numhits >= 6 && (aa == 'F' || aa == 'Y' || aa == 'W'))) 
			&& d_HpX <= 1.2 && d_DV <= 4.5 && angle_DHV >= 110){
#if 0
	fprintf(stderr,"numhits=%d\n",numhits);
	fprintf(stderr, " d_HpX=%lf; d_DV=%lf; d_HV=%lf; angle_DHV=%lf\n",
		d_HpX,d_DV,d_HV,angle_DHV);
#endif
		    Int4 rD=ResAtom(Donor),rA=ResAtom(Accept);
		    if(rA < 0 || rD < 0) continue;
		    if(d_HpX < bst_HpX){ 
			bst_HpX=d_HpX; bst_DHV=angle_DHV; bst_DV=d_DV; 
			if(OxygenAtom(Donor)) Typ='O';
			else if(NitrogenAtom(Donor)) Typ='N';
			else if(CarbonAtom(Donor)) Typ='C';
			else if(SulfurAtom(Donor)) Typ='S';
			else Typ='X';
#if 0	//===================== DEBUG =====================
if(resA==350 && resD==88){	//  && NitrogenAtom(Donor)){
	fprintf(stderr,"-%cH: ",Typ);
	this->DebugCH_Pi(Donor,Accept,d_HpX,d_DV,angle_DHV,numhits,AB);
	PutAtom(stderr,Donor);
}
#endif	//================================================
		    } n++; 
		}
	     }
	} return n;
}

Int4	spc_typ::RunAllCH_Pi(FILE *ofp)
{
	Int4	N,NN,i,j,k,x,p,MinRes,MaxRes,nI,nJ;
	FILE	*efp=0;
	Int4	nChn,MaxLen=0;

	assert(mode != 'D');
	for(i=0 ; Chain[i]; ) i++; nChn=i; 
	if(pmlfp){ this->PutHeaderPyMOL(); }
	// 1. open up one of the structures for reference.
	fprintf(stderr,"CH-pi interactions done.\n"); // exit(1);
	pdb_typ	pdb=psd->RtnPDB(1); 
	Int4	nPDB=psd->NumPDB();
	MinRes=999999; MaxRes=0;
	for(i=0; Chain[i]; i++){  // find start and end common to all chains.
	        j=GetChainNumberPDB(pdb,Chain[i]);
		if((x=MinResPDB(j,pdb)) > 0){ MinRes=MINIMUM(Int4,MinRes,x); }
		if(i==0) MaxRes=MAXIMUM(Int4,MaxRes,MaxResPDB(j,pdb));
		else MaxRes=MINIMUM(Int4,MaxRes,MaxResPDB(j,pdb));
		Int4 Len=MaxRes-MinRes+1;
		MaxLen = MAXIMUM(Int4,Len,MaxLen);
	} // Only need to look at one; they are all the same.
	NumAroAro=NumAroAroPi=0;

//fprintf(stderr,"DEBUG 1\n");
	// if(sis==0 && mode != 'B') sis = new sis_typ(Chain,MaxLen);
	if(sis==0 && mode != 'B') sis = new sis_typ(Chain,MaxRes);
	if(vsifp){
	  fprintf(vsifp,"~$=1.\n");
	  for(i=1; i <= nChn; i++){
	     fprintf(vsifp,"File%d=%s:%c  //\n",i,psd->RtnPath(1),Chain[i-1]);
	  } fprintf(vsifp,"\n1-10000.W15\n\n");
	  fprintf(vsifp,"%d-%d.Y80\n",MinRes,MaxRes);
	}
    {
	i=GetChainNumberPDB(pdb,Chain[0]);
	e_type Sq=GetPDBSeq(i,pdb);
	a_type AB=AminoAcidAlphabetPDB(pdb); StrtMtrx=MinRes; EndMtrx=MaxRes;
// PutSeq(stderr,Sq,AB);
	// set_typ useSB=0;
	if(mode=='S' && PlotGrph){	// Create contact matrix...
	  this->CreateMtrx(pdb);
	} // else { useSB=MakeSet(MaxRes+4); FillSet(useSB); }
	Int4 NumVSI=0;
	char setI=0,setJ=0;
	for(nI=MinRes; nI < MaxRes; nI++) {
	   if(PtrnSet && !MemberSet(nI,PtrnSet[0])){
	      if(!show_all && TopDC_Set && !MemberSet(nI,TopDC_Set)) continue;
	   }
	   if(PtrnSet){
		for(Int4 ii=1; PtrnSet[ii]; ii++){
		   if(MemberSet(nI,PtrnSet[ii])) setI=ii; 
		}
	   }
	   // if(!MemberSet(nI,useSB)) continue;
	   // for(nJ=MinRes; nJ <= MaxRes; nJ++)
	   for(nJ=nI+1; nJ <= MaxRes; nJ++) {
	     if(PtrnSet && !MemberSet(nJ,PtrnSet[0])){
	        if(!show_all && TopDC_Set && !MemberSet(nJ,TopDC_Set)) continue;
	     }
	     if(PtrnSet){
		for(Int4 jj=1; PtrnSet[jj]; jj++){
		   if(MemberSet(nJ,PtrnSet[jj])) setJ=jj; 
		}
	     }
	     // if(!MemberSet(nJ,useSB)) continue;
	     if(abs(nI-nJ) < MinSpacing) continue;
	     if(mode=='s' && nJ < nI) continue;  // only need to check one way.
	     Int4	cI,cJ;
	     float	HA_dmax=1.2,dmax=4.5; 
	     double	dIJ,dJI,dII,cut=2.5,cutX=3.0,Cut=0.01;
	     double	d=BigDist,dd,D=BigDist,DD=BigDist,Dd=BigDist;
#if 1
	     cI=GetChainNumberPDB(pdb,Chain[0]);
	     char rI=psd->GetResChar(1,cI,nI),rJ=psd->GetResChar(1,cI,nJ);
	     if(rI==0 || rJ==0) continue;
	     if(this->InvalidHbondPair(rI,rJ,'P')) continue; // make sure both are aromatic.
#endif
	     for(i=0; Chain[i]; i++){
	       double	td=BigDist,tD=BigDist,cd=BigDist,cD=BigDist;
	       // find residues relatively near each other.
	       cI=GetChainNumberPDB(pdb,Chain[i]);
	       j=i+1;
	       if(Chain[j]==0) j=0;
	       cJ=GetChainNumberPDB(pdb,Chain[j]);
	       dIJ=DistancePDB(nI,"CA",cI,nJ,"CA",cJ,pdb);
	       dJI=DistancePDB(nI,"CA",cJ,nJ,"CA",cI,pdb);
	       dII=DistancePDB(nI,"CA",cI,nJ,"CA",cI,pdb);
	       if(0) fprintf(stderr,
			"%d -> %d%c vs %d%c: dIJ=%lf; dJI=%lf; dII=%lf\n",
		  	N,nI,Chain[i],nJ,Chain[j],dIJ,dJI,dII);
	       if(dIJ > 30.0 && dJI > 30.0 && dII > 30.0){ continue; }
	       char Typ=0;
	       for(k=0,p=1; k <= 1; k++,p=nPDB){ 
		 if(mode != 'B' && nChn > 1){	// trans...???
	           td=this->GetCH_PiFast(efp,nI,nJ,HA_dmax,cI,cJ,p,Typ);
	           tD=this->GetCH_PiFast(efp,nI,nJ,HA_dmax,cJ,cI,p,Typ);
		 } 
	         cd=this->GetCH_PiFast(efp,nI,nJ,HA_dmax,cI,cI,p,Typ);
	         cD=this->GetCH_PiFast(efp,nJ,nI,HA_dmax,cI,cI,p,Typ);
		 
	         if(td > 0.0) d=MINIMUM(double,d,td);	// trans
	         if(tD > 0.0) D=MINIMUM(double,D,tD);	// trans
	         if(cd > 0.0) DD=MINIMUM(double,DD,cd);	// cis
	         if(cD > 0.0) Dd=MINIMUM(double,Dd,cD);	// cis
		 dd=td; dd=MINIMUM(double,tD,dd);
		 dd=MINIMUM(double,cd,dd); dd=MINIMUM(double,cD,dd);
		 if(verbose && dd < cutX) fprintf(stderr,
		   "%d -> %d%c vs %d%c: td=%lf; tD=%lf; cd=%lf; cD=%lf; dd=%lg\n",
		      p,nI,Chain[i],nJ,Chain[j],td,tD,cd,cd,dd);
	       }
	     }
	     if((d < cutX && d > Cut) || (D < cutX && D > Cut) ||
		(DD < cutX && DD > Cut) || (Dd < cutX && Dd > Cut)) {
		char Str[150],str[10],StrI[100],StrJ[100],strTmp[20];
		Int4 os=OffSetSeq(Sq);
		char rI=AlphaChar(ResSeq(nI-os,Sq),AB);
		char rJ=AlphaChar(ResSeq(nJ-os,Sq),AB);
		if(mode=='s') sprintf(str,"sc2bb");
		else if(mode=='S') sprintf(str,"sc2sc"); else sprintf(str,"bb2bb");
		char cII=0,cJJ=0;
		if(0 && mode=='B'){
		   sprintf(Str,"%c%d vs %c%d: %.2lf %.2lf (%s)\n",
				rI,nI,rJ,nJ,DD,Dd,str);
		} else if(PtrnSet){	// using DARC patterns as input
		   cII=SbGrpCode[setI]; cJJ=SbGrpCode[setJ];
		   // if(!isalpha(cII)) cII='-'; if(!isalpha(cJJ)) cJJ='-';
		   if(d < BigDist){
		     d=MINIMUM(double,d,D); DD=MINIMUM(double,Dd,DD);
		     if(DD < BigDist) sprintf(strTmp,"%.2lf",DD); else sprintf(strTmp," - ");
		     sprintf(Str,
			"%c%d(%c) vs %c%d(%c): %s %.2lf (%s)\n",
				rI,nI,cII,rJ,nJ,cJJ,strTmp,d,str);
		   } else {
		     DD=MINIMUM(double,Dd,DD);
		     if(DD < BigDist) sprintf(strTmp,"%.2lf",DD); else sprintf(strTmp," - ");
		     sprintf(Str,"%c%d(%c) vs %c%d(%c): %s (%s)\n",
				rI,nI,cII,rJ,nJ,cJJ,strTmp,str);
		   }
		} else {
		   if(d < BigDist){
		     d=MINIMUM(double,d,D); DD=MINIMUM(double,Dd,DD);
		     if(DD < BigDist) sprintf(strTmp,"%.2lf",DD); else sprintf(strTmp," - ");
		     sprintf(Str,"%c%d vs %c%d: %s %.2lf (%s)\n",
				rI,nI,rJ,nJ,strTmp,d,str);
		   } else {
		     DD=MINIMUM(double,Dd,DD);
#if 0
		     if(DD < BigDist) sprintf(strTmp,"%.2lf %c",DD,Angstroms);
#else
		     if(DD < BigDist) sprintf(strTmp,"%.2lf ",DD);
#endif
		     else sprintf(strTmp," - ");
		     sprintf(Str,"%c%d vs %c%d: %s (%s)\n",
				rI,nI,rJ,nJ,strTmp,str);
		   }
		} this->GetSeqContext(StrI,nI-os-1,3,Sq,AB); 
		this->GetSeqContext(StrJ,nJ-os-1,3,Sq,AB);
		// fprintf(stderr,"%s\n%s\n%s\nnI=%d; nJ=%d; os=%d\n",Str,StrI,StrJ,nI,nJ,os);
		// if(verbose){ fprintf(stderr,"%s\n",Str); }
		if(d <= cutX){		// trans: I vs J
		   d=this->RunOneCH_PiFast(nI,nJ,TRUE,cut,ofp,Str,StrI,StrJ);
		}
		if(D <= cutX){		// trans: J vs I
		   D=this->RunOneCH_PiFast(nJ,nI,TRUE,cut,ofp,Str,StrI,StrJ);
		}
		if(DD <= cutX){		// cis: I vs J
		   Dd=DD=this->RunOneCH_PiFast(nI,nJ,FALSE,cut,ofp,Str,StrI,StrJ);
		} else if(Dd <= cutX){ 	// cis: J vs I.
		   DD=Dd=this->RunOneCH_PiFast(nJ,nI,FALSE,cut,ofp,Str,StrI,StrJ);
		}
	        if(d < cut || D < cut || DD < cut || Dd < cut){
		  // PutSeqRegion(ofp,nI-os-1,3,Sq,AB); fprintf(ofp,"\n");
		  // PutSeqRegion(ofp,nJ-os-1,3,Sq,AB); fprintf(ofp,"\n\n"); fflush(ofp);
		  if(pmlfp){
		   Int4 ri=AlphaCode(rI,AB),rj=AlphaCode(rJ,AB);
		   this->PutResPairPyMOL(ri,rI,nI,rj,rJ,nJ);
		  }
	          if(vsifp){
		    if(NumVSI > 0) fprintf(vsifp,","); NumVSI++;
		    if(PtrnSet){
		       fprintf(vsifp,"%c%d.%c,%c%d.%c",rI,nI,cII,rJ,nJ,cJJ);
		    } else {
		       fprintf(vsifp,"%c%d.R,%c%d.R",rI,nI,rJ,nJ);
		    }
		    fflush(vsifp);
		  }
		} // end if(d < cut || D < cut || DD < cut || Dd < cut)
	     } // end if((d < cutX && d > Cut) || (D < cutX && D > Cut)...
	  } // end for(nJ=nI+1; nJ <= MaxRes; nJ++)
	} // end for(nI=MinRes; nI <= MaxRes; nI++)
	// if(useSB) NilSet(useSB); 
	NilSeq(Sq); 
    }
	if(vsifp){ fprintf(vsifp,"\n\n"); }
	if(pmlfp) this->PutTailPyMOL();
	return 0;
}

double	spc_typ::RunOneCH_PiFast(Int4 nI,Int4 nJ, BooLean trans,double cut,
		FILE *outfp,char *Msg,char *StrI, char *StrJ)
// Compare residue nI and nJ for interactions in both cis and trans.
{
	double	dd,Dd,*Sum,**Value; 
	float	HA_dmax=2.5,dmax=3.6; 
	FILE	*efp=0,*ofp=0;
	Int4	i,j,z,N,cI,cJ,nChn,nPDB,Mid,*NN;
	assert(mode != 'D'); assert(psd != 0);
	nPDB=psd->NumPDB(); Mid=nPDB/2;
	tri_typ	*tri=new tri_typ(Chain,nPDB);


	for(nChn=0 ; Chain[nChn]; ) nChn++; 
        NEWP(Value,10,double);
        char    **Typ; NEWP(Typ,10,char);
        for(i=0; i <=3; i++){ NEW(Value[i],nChn+3,double); NEW(Typ[i],nChn+3,char); }
	NEW(Sum, nChn+3,double); NEW(NN, nChn +3, Int4);

	if(0 && cut > 0.0) ofp=tmpfile(); 
	if(verbose) efp=stderr;

	//====== loop over all pdb files... ======
	for(N=1; N <= nPDB; N++){
	  pdb_typ pdb = psd->RtnPDB(N);
	  if(trans) j=1; else j=0;
	  for(i=0 ; Chain[i]; i++,j++){
	   if(Chain[j]==0) j=0;
	   cI=GetChainNumberPDB(pdb,Chain[i]);
	   cJ=GetChainNumberPDB(pdb,Chain[j]);
	   if(ofp){
	     if(N==1 && i==0){
	       fprintf(ofp,"index\t");
	       for(Int4 x=0,y=1; Chain[x];y++,x++){
		if(Chain[y]==0) y=0; 
		if(trans) fprintf(ofp,"%c%c",Chain[x],Chain[y]);
		else fprintf(ofp,"%c",Chain[x]);
		if(y != 0) fprintf(ofp,"\t");
	       } fprintf(ofp,"\n"); 
	     }
	     if(i==0) fprintf(ofp,"%d\t",N);
	   }
	   if(cI == 0 || !IsProteinPDB(cI,pdb)){
		if(ofp){ fprintf(ofp,"-"); if(Chain[i+1]) fprintf(ofp,"\t"); }
		continue;
	   } if(cJ == 0 || !IsProteinPDB(cJ,pdb)){
		if(ofp){ fprintf(ofp,"-"); if(Chain[i+1]) fprintf(ofp,"\t"); }
		continue;
	   }
	   if(efp) fprintf(efp,"============ %d%c:%d%c ==================\n",
					nI,Chain[i],nJ,Chain[j]);
	   char typ=0;
	   dd=this->GetCH_PiFast(efp,nI,nJ,HA_dmax,cI,cJ,N,typ);
	   if(ofp){
	      if(dd >= BigDist){
	        fprintf(ofp,"...."); fflush(ofp); 
	        // fprintf(ofp,"%.1lf",dd);
	      } else fprintf(ofp,"%.2lf",dd);
	   }
	   tri->AddValue(N,dd,i,typ);

#if 0
	   if(i==0){
	     if(N==1) NEW(Value[0],nChn+3,double);
	     else if(N==Mid) NEW(Value[1],nChn+3,double);
	     else if(N==nPDB) NEW(Value[2],nChn+3,double);
	   }
	   if(N==1) Value[0][i]=dd;
	   else if(N==Mid) Value[1][i]=dd;
	   else if(N==nPDB) Value[2][i]=dd;
#else
           if(N==1){ Value[0][i]=dd; Typ[0][i]=typ; }
           else if(N==Mid){ Value[1][i]=dd; Typ[1][i]=typ; }
           else if(N==nPDB){ Value[2][i]=dd; Typ[2][i]=typ; }
#endif
	
	   if(ofp && Chain[i+1]) fprintf(ofp,"\t"); 
	   if(dd < this->UnDefHBond){ Sum[i] += dd; NN[i]++; }
	  } if(ofp) fprintf(ofp,"\n"); 
	  // tri->Put(stderr,N);
	}
	// NEW(Value[3],nChn +3, double); Allocated above...
	double	rtn=DBL_MAX;
	char	nhits=0;
// ofp=stdout;
	if(1 || efp && nChn > 0){
	   if(ofp) fprintf(ofp,"avg.\t");
	   for(Int4 x=0; Chain[x];x++){
	     dd=Sum[x]/(double) NN[x]; 
	     if(ofp) fprintf(ofp,"%.2lf",dd);
	     Value[3][x]=dd;
	     if(ofp && Chain[x+1]) fprintf(ofp,"\t");
	     rtn=MINIMUM(double,rtn,dd);
	     if(PlotGrph && Mtrx){
		if(dd <= 2.5){
		   Mtrx[nI][nJ]++; Mtrx[nJ][nI]++; nhits++;
		}
	     }
	     if(dd <= 2.5){
		Int4 y,r;
		if(trans){	// x = Ct & y = Rt or x = Lt & y = Ct
	   	   if(Chain[x+1]==0) y=0; else y=x+1;
		   if(sis){
		     sis->AddSite(Ct,x,nI); sis->AddSite(Rt,x,nJ);
		     sis->AddSite(Lt,y,nI); sis->AddSite(Ct,y,nJ);
		   }
		} else {	// x = Ct & x = Ct
		   if(sis){ sis->AddSite(Ct,x,nI); sis->AddSite(Ct,x,nJ); }
		}
	     }
	   } if(ofp) fprintf(ofp,"\n"); 
	} free(Sum); free(NN);
#if 1
	if(nhits > 0 && PlotGrph && Mtrx){
		if(mode=='S'){ AddSet(nI,IsSC[nJ]); AddSet(nJ,IsSC[nI]); }
		else if(mode=='s'){ AddSet(nI,IsBB[nJ]); AddSet(nJ,IsBB[nI]); }
		Mtrx[nI][0]++; Mtrx[nJ][0]++; 
		JoinWdgraph(nI,nJ,nChn-nhits,wdg);
		JoinWdgraph(nJ,nI,nChn-nhits,wdg);
	}
#endif
	if(ofp && cut > 0){
	   if(rtn <= cut){ 
		if(Msg) fprintf(outfp,"%s",Msg);
		char c;
		if(ofp){
		    rewind(ofp); 
		    while((c=fgetc(ofp)) != EOF) fprintf(outfp,"%c",c); 
		}
	   }
	} if(ofp) fclose(ofp);
	char cmd[200],str[10],CT='C';
	if(trans) CT='T';
	if(mode == 'S') sprintf(str,"sc2sc");
	else if(mode == 's') sprintf(str,"sc2bb");
	else if(mode == 'B') sprintf(str,"bb2bb");
	else print_error("This should not happen");
	sprintf(cmd,"sparc %s %s %s %d:%d%c\n",str,argv[2],argv[3],nI,nJ,CT);
	keytyp key=0;
	char   **msgF,**msgS,**msgB;
	assert(mode != 'B');
#if 0
	msgF=tri->RtnResults(key,Msg,cmd,trans,Value,'F');
	if(msgF){
	   if(rphWF==0) rphWF = new rph_typ(rpheap_size,nChn);
	   for(z=0; msgF[z]; ) z++;
	   msgF[z] = AllocString(StrI); z++; msgF[z] = AllocString(StrJ);
	   rphWF->Insert(key,msgF);
	}
#elif 0
	// msgS=tri->RtnResults(key,Msg,cmd,trans,Value,'P');
#endif
	msgS=tri->RtnResults(key,Msg,cmd,trans,Value,'P',Typ);
	if(msgS){
#if 0
	   for(z=0; msgS[z]; z++) fprintf(stderr,"%s",msgS[z]);
	   fprintf(stderr,"\n");
#endif
	   for(z=0; msgS[z]; ) z++;
	   msgS[z] = AllocString(StrI); z++; msgS[z] = AllocString(StrJ);
	   if(rphWS==0) rphWS = new rph_typ(rpheap_size,nChn);
	   rphWS->Insert(-key,msgS);
	}
        for(i=0; i<=3; i++) if(Value[i]) free(Value[i]); free(Value);
        for(i=0; i<=3; i++) if(Typ[i]) free(Typ[i]); free(Typ);
	delete tri; tri=0;
	return rtn;
}

double	spc_typ::GetCH_PiFast(FILE *fptr, Int4 resI,Int4 resJ, float dmax,
		Int4 cI, Int4 cJ, Int4 X, char &Typ)
// Move this routine to libpdb.a at some point:
{
	if(cI==0 || cJ==0){ print_error("chain not found!"); }
	assert(psd != 0);
	pdb_typ P=psd->RtnPDB(X);
	res_typ ResI=psd->RtnResI(X,cI,resI);
	res_typ ResJ=psd->RtnResI(X,cJ,resJ);
	double dd=this->BigDist;
	if(ResI == 0  || ResJ == 0) return dd;
        Int4    aI,aJ,rI,rJ,rtn;
        char    color='R',Mode='X',Type=' ',bst;
	double	d_HpX=BigDist,d_DV=BigDist,angle_DHV=0; 
	double	default_dmax=dmax; dmax=4.5;

        rtn=this->FindAromaticHbonds(fptr,ResI,ResJ,cI,cJ,dmax,color,P,d_HpX,
			d_DV,angle_DHV,Type,Mode);
	if(this->OkayPiParameters(d_HpX,d_DV,angle_DHV) &&  dd > d_HpX){
		Typ=Type; dd=d_HpX;
#if 0
if(rtn> 0 && d_HpX < 50.0 && resJ==350 && resI==88) 
	fprintf(stderr,"%da: -%cH = %.2lf;%.2lf;%.2lf;dd=%.2lf\n",
		X,Type,d_HpX,d_DV,angle_DHV,dd);
#endif
	}

        rtn=this->FindAromaticHbonds(fptr,ResJ,ResI,cJ,cI,dmax,color,P,d_HpX,
		d_DV,angle_DHV,Type,Mode);
	if(this->OkayPiParameters(d_HpX,d_DV,angle_DHV) && dd > d_HpX){
		Typ=Type; dd=d_HpX;
#if 0
if(rtn > 0 && d_HpX < 50.0 && resI==350 && resJ==88) 
	fprintf(stderr,"%db: -%cH = %.2lf;%.2lf;%.2lf;dd=%.2lf\n",
		X,Type,d_HpX,d_DV,angle_DHV,dd);
#endif
	}

        this->FindOtherPiHbonds(fptr,ResI,ResJ,cI,cJ,dmax,color,P,d_HpX,
		d_DV,angle_DHV,Type,Mode);
	if(this->OkayPiParameters(d_HpX,d_DV,angle_DHV) && dd > d_HpX){
		Typ=Type; dd=d_HpX;
	}

        this->FindOtherPiHbonds(fptr,ResJ,ResI,cJ,cI,dmax,color,P,d_HpX,
		d_DV,angle_DHV,Type,Mode);
	if(this->OkayPiParameters(d_HpX,d_DV,angle_DHV) && dd > d_HpX){
		Typ=Type; dd=d_HpX;
	} dmax=default_dmax;
        return dd;
} 

