/******************************************************************************************
    Copyright (C) 1997-2019 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in the
    Software without restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the
    following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/
#include "spc_typ.h"

#define MAX_angleDHA 90

double	spc_typ::Normal2NormalAngle(atm_typ P, atm_typ Q, atm_typ R, atm_typ S)
{
	atm_typ	O,PmQ,RmS;
	double	dd;
	float	x,y,z;
	x=AtomX(P)-AtomX(Q); y=AtomY(P)-AtomY(Q); z=AtomZ(P)-AtomZ(Q);
	PmQ=VirtualAtom(x,y,z);
	x=AtomX(R)-AtomX(S); y=AtomY(R)-AtomY(S); z=AtomZ(R)-AtomZ(S);
	RmS=VirtualAtom(x,y,z); O=VirtualAtom(0.0,0.0,0.0);
	dd=CalcAngle(PmQ,O,RmS);
	// PutAtom(stderr,O); PutAtom(stderr,PmQ); PutAtom(stderr,RmS);
	NilAtom(PmQ);
	NilAtom(RmS);
	NilAtom(O);
	assert(dd >= 0.0 && dd <= 180.0); 
	if(dd > 90.0) dd = 180.0 - dd;
	return dd;
}

Int4    spc_typ::AromaticRingAtoms(aai_typ X,char c)
{
	if(c=='A') return this->AromaticRingAtoms(X->ResA,X->RngA,
                   X->hydA, X->cenA,X->AB,X->nAB,X->RngAW,X->hydAW);
	if(c=='B') return this->AromaticRingAtoms(X->ResB,X->RngB,
                   X->hydB, X->cenB,X->AB,X->nAB,X->RngBW,X->hydBW);
	else print_error("AromaticRingAtoms() input error");
}

Int4	spc_typ::AromaticRingAtoms(res_typ ResA,atm_typ rtn[10],atm_typ hyd[10],
		atm_typ vtl[4],a_type AB,a_type nAB,atm_typ Rtn[10],atm_typ Hyd[10])
{
	Int4	n,d,N=0,nH=0,nh=0,i,j,h;
	BooLean	na,found,Found;
	atm_typ	atm,vrtl,H;
	char    *name,aa=GetResidueAtom(AtomResidue(1,ResA),AB,nAB,&na);
	aa=AlphaChar(aa,AB);
	if(na || strchr("HFYW",aa) == NULL) return 0;

	for(i=0; i < 10; i++) hyd[i]=rtn[i]=Rtn[i]=Hyd[i]=0;
	for(i=0; i < 4; i++) vtl[i]=0;
	for(n=0,d = 1; d <= ResidueAtomNumber(ResA); d++){
           atm = AtomResidue(d,ResA); name=AtomName(atm); 
	   Found=found=FALSE;   // fprintf(stderr,"name='%s'\n",name);
	   switch (aa){ 
		case 'H': { 
		  for(j=0; this->RingH5[j]; j++){
		     if(strcmp(name,RingH5[j])==0) { found=TRUE; break; }
		  }
	        } break;
		case 'Y': 
		case 'F': { 
		  for(j=0; RingFY6[j]; j++){
		     if(strcmp(name,RingFY6[j])==0) { found=TRUE; break; }
		  }
	        } break;
		case 'W': { 
		  for(j=0; RingW6[j]; j++){
		     if(strcmp(name,RingW6[j])==0) { found=TRUE; break; }
		  }
		  for(j=0; RingW5[j]; j++){
		     if(strcmp(name,RingW5[j])==0) { Found=TRUE; break; }
		  }
	        } break;
		default:
		  break;
	   }
	   if(found){
	   	if(ResidueAtomHydrogens(d,ResA)){
	   	   for(Int4 h=1; (H=ResidueAtomHydrogen(d,h,ResA)); h++){
	                nh++; hyd[nh]=H; 
		   }
	   	} N++; rtn[N]=atm;
	   }
	   if(Found){
	   	if(ResidueAtomHydrogens(d,ResA)){
	   	   for(Int4 h=1; (H=ResidueAtomHydrogen(d,h,ResA)); h++){
	                nH++; Hyd[nH]=H; 
		   }
	   	} n++; Rtn[n]=atm;
	   }
	} N=this->VirtualAromatic(ResA,vtl,AB,nAB);
	return N;
}

Int4	spc_typ::VirtualAromatic(res_typ ResA,atm_typ vtl[4],a_type AB,a_type nAB)
{
	Int4	n,d,N=0,i,j;
	BooLean	na;
	char	*name;
	atm_typ	atm,vrtl,atom[10];
	char    aa = GetResidueAtom(AtomResidue(1,ResA),AB,nAB,&na);
	aa = AlphaChar(aa,AB);
	for(i=0; i < 4; i++) vtl[i]=0;
	Int4 rA=ResidueID(ResA);  // DEBUG

	for(i=0; i < 3; i++) vtl[i]=0;
	switch (aa){
	  case 'H': { 
	      for(n=0,i=1; i <= ResidueAtomNumber(ResA); i++){
	        atm =AtomResidue(i,ResA); name = AtomName(atm);
		// fprintf(stderr,"name='%s'\n",name);
		for(j=0; RingH5[j]; j++){
		  if(strcmp(name,RingH5[j]) == 0){ 
			n++; atom[n]=atm; break;
		  }
		}
	      } assert(n == 5);
	      vrtl = AverageAtom(n,atom); 
	      N++; vtl[N]=vrtl; 
	   } break;
	  case 'Y':
	  case 'F': {
	      for(n=0,i=1; i <= ResidueAtomNumber(ResA); i++){
	        atm =AtomResidue(i,ResA); name = AtomName(atm);
		// fprintf(stderr,"name='%s'\n",name);
		for(j=0; RingFY6[j]; j++){
		  if(strcmp(name,RingFY6[j]) == 0){ 
			n++; atom[n]=atm; break;
		  }
		}
	      } assert(n == 6);
	      vrtl = AverageAtom(n,atom); 
	      N++; vtl[N]=vrtl; 
	   } break;
	  case 'W': {
	      for(n=0,i=1; i <= ResidueAtomNumber(ResA); i++){
	        atm =AtomResidue(i,ResA); name = AtomName(atm);
		for(j=0; RingW6[j]; j++){
		  if(strcmp(name,RingW6[j]) == 0){ 
			n++; atom[n]=atm; break;
		  }
		}
	      }
	      assert(n == 6);
	      vrtl = AverageAtom(n,atom); 
	      N++; vtl[N]=vrtl;
	      for(n=0,i=1; i <= ResidueAtomNumber(ResA); i++){
	        atm =AtomResidue(i,ResA); name = AtomName(atm);
		// fprintf(stderr,"name='%s'\n",name);
		for(j=0; RingW5[j]; j++){
		  if(strcmp(name,RingW5[j]) == 0){ 
			n++; atom[n]=atm; break;
		  }
		}
	      } assert(n == 5);
	      vrtl = AverageAtom(n,atom); 
	      N++; vtl[N]=vrtl; 
	   } break;
	  default:
	   break;
	} return N;
}

atm_typ	*spc_typ::PerpendicularAtom(atm_typ P, atm_typ Q,atm_typ R)
/********************************************************
Return an atom located perpendicular to the plane formed by points P,Q and R
and placed directly above atom P.
 ********************************************************/
#if 0	// DEBUG...Example from: https://www.youtube.com/watch?v=rL9UXzZYYo4
	atm_typ P=VirtualAtom(2,1,4,'P');
	atm_typ Q=VirtualAtom(4,-2,7,'Q');
	atm_typ R=VirtualAtom(5,3,-2,'R');
	PutAtom(stdout,P); PutAtom(stdout,Q); PutAtom(stdout,R); 
	atm_typ *NN= this->PerpendicularAtom(P,Q,R);
	PutAtom(stdout,NN[1]); PutAtom(stdout,NN[2]); 
	exit(1);	
#endif
{
	// equation of the plane through the points P, Q, R.
	double	dd,x,y,z,dx,dy,dz,absAB,absAC;
	double di,dj,dk;	// determinants of i, j, and k.
	Int4	i,j,k;
	atm_typ A,B,N,*V; NEW(V,4,atm_typ);
	// A = PQ, B = PR vectors
	A = VirtualAtom(AtomX(Q)-AtomX(P),AtomY(Q)-AtomY(P),AtomZ(Q)-AtomZ(P),'A');
	B = VirtualAtom(AtomX(R)-AtomX(P),AtomY(R)-AtomY(P),AtomZ(R)-AtomZ(P),'B');
	if(0){ PutAtom(stdout,A); PutAtom(stdout,B); }
#if 0
	double n[4][4];

	// A X B (crossproduct) = n
	n[1][1]=n[1][2]=n[1][3]=1.0;	                      // | i  j  k  |
	n[2][1]=AtomX(A); n[2][2]=AtomY(A); n[2][3]=AtomZ(A); // | Ax Ay Az |
	n[3][1]=AtomX(B); n[3][2]=AtomY(B); n[3][3]=AtomZ(B); // | Bx By Bz |
	// find determinants of the cross product.
	di = n[2][2]*n[3][3] - n[3][2]*n[2][3];
	dj = -1.0*(n[2][1]*n[3][3] - n[3][1]*n[2][3]); 
	dk = n[2][1]*n[3][2] - n[3][1]*n[2][2]; 
#else
	di=AtomY(A)*AtomZ(B) - AtomY(B)*AtomZ(A);
	dj=-1.0*(AtomX(A)*AtomZ(B) - AtomX(B)*AtomZ(A));
	dk=AtomX(A)*AtomY(B) - AtomX(B)*AtomY(A);
#endif
	NilAtom(A); NilAtom(B);
	N=VirtualAtom(di,dj,dk,'N');
	if(0) PutAtom(stdout,N); NilAtom(N);
	// Normalize N.
	double len=sqrt(di*di + dj*dj + dk*dk);
	N=VirtualAtom(di/len,dj/len,dk/len,'n');
	if(0) PutAtom(stdout,N); 

	// Equation for the plane:
	// di*(x-AtomX(P)) + dj*(y-AtomY(P)) + dk*(z-AtomZ(P))=0;
	// or di*x + dj*y + dk*z= K;
	double X,Y,Z,K = di*AtomX(P) + dj*AtomY(P) + dk*AtomZ(P);
// fprintf(stderr,"K=%.2lf\n",K);
	// V=VirtualAtom(di,dj,dk);
	V[1]=VirtualAtom(AtomX(N)+AtomX(P),AtomY(N)+AtomY(P),AtomZ(N)+AtomZ(P),'V');
	V[2]=VirtualAtom(AtomX(P)-AtomX(N),AtomY(P)-AtomY(N),AtomZ(P)-AtomZ(N),'V');
	NilAtom(N);
#if 0
	PutAtom(stdout,V[1]); PutAtom(stdout,V[2]); 
	for(i=1,x=AtomX(P); i <= 3; i++,x+=1.5){
	    y=AtomY(P);
	    z = (K - di*x - dj*y)/dk;
	    if(i==1 && j==1 && k==1) N=VirtualAtom(x,y,z,'Z');
	    else N=VirtualAtom(x,y,z);
	    PutAtom(stdout,N); NilAtom(N);
	}
#endif
	return V;
}

BooLean spc_typ::PrintAromaticRing(FILE *fp,aai_typ X,atm_typ Nrm, char c,set_typ WasPrint)
{
	Int4	i,j,s;
	atm_typ	*Rng,*hyd,cen;
	BooLean	IsW=FALSE;
	if(c == 'A'){
	   if(X->aaA == 'W') IsW=TRUE; 
	   s=ResidueID(X->ResA); 
	   Rng=X->RngA; hyd=X->hydA; cen=X->cenA[1];
	} else if(c == 'B'){
	   if(X->aaB == 'W') IsW=TRUE;
	   s=ResidueID(X->ResB); 
	   Rng=X->RngB; hyd=X->hydB; cen=X->cenB[1];
	} else print_error("PrintAromaticRings() input error");
// fprintf(stderr,"s=%d\n",s);
	if(MemberSet(s,WasPrint)) return FALSE;
	for(j=1; Rng[j]; j++){ PutAtom(fp,Rng[j]); }
	for(j=1; hyd[j]; j++){ PutAtom(fp,hyd[j]); }
	PutAtom(fp,cen);
	PutAtom(fp,Nrm); 
#if 1
	if(IsW){	// skips last two overlapping ring atoms...
	  if(c == 'A'){ Rng=X->RngAW; hyd=X->hydAW; cen=X->cenA[2]; }
	  else { Rng=X->RngBW; hyd=X->hydBW; cen=X->cenB[2]; } 
	  for(j=1; Rng[j] && j <= 3; j++){ PutAtom(fp,Rng[j]); }
	  for(j=1; hyd[j]; j++){ PutAtom(fp,hyd[j]); }
	  PutAtom(fp,cen);
	  atm_typ *atmN=this->PerpendicularAtom(cen,Rng[1],Rng[2]);
	  if(atmN){ 
	    double d=DistanceAtoms(cen,atmN[1]),dd=DistanceAtoms(cen,atmN[2]);
	    if(d < dd) PutAtom(fp,atmN[1]); else PutAtom(fp,atmN[2]);
	    NilAtom(atmN[1]); NilAtom(atmN[2]); 
	    free(atmN); 
	  }
	}
#endif
	AddSet(s,WasPrint);
	return TRUE;
}

double spc_typ::FindAroAro(aai_typ X,BooLean UseAW5, BooLean UseBW5)
// see McGaughey,Gagne, Rapp. 1998. JBC 273:15458-15463.
{
	Int4	i,j;
	Int4	sA=ResidueID(X->ResA),sB=ResidueID(X->ResB);
	double	d,dd=this->BigDist,gamma,theta,delta;
	double	Rcen,RcenMax=7.2,Rclo,RcloMax=4.5; 
	atm_typ	*RngA=X->RngA,*RngB=X->RngB;
	atm_typ	cenA=X->cenA[1],cenB=X->cenB[1];
	atm_typ	*hydA=X->hydA,*hydB=X->hydB;
	if(UseAW5){
	   assert(X->aaA == 'W');
	   RngA=X->RngAW; hydA=X->hydAW; cenA=X->cenA[2];
	}
	if(UseBW5){
	   assert(X->aaB == 'W');
	   RngB=X->RngBW; hydB=X->hydBW; cenB=X->cenB[2];
	}
	for(Rclo=this->BigDist,i=1; RngA[i]; i++){
	      for(j=1; RngB[j]; j++){
		d=DistanceAtoms(RngA[i],RngB[j]);
		if(d < Rclo){ Rclo=d; }
	      }
	} Rcen=DistanceAtoms(cenA,cenB);
	if(Rclo <= RcloMax && Rcen <= RcenMax){
if(0) fprintf(stderr,"%c%d%c vs %c%d%c: %.2lf %.2lf\n",
		X->aaA,sA,X->ca,X->aaB,sB,X->cb,Rclo,Rcen);
	      //=== find perpendicular positions to centroids...
	      atm_typ *atmNA= this->PerpendicularAtom(cenA,RngA[1],RngA[2]);
	      atm_typ *atmNB= this->PerpendicularAtom(cenB,RngB[1],RngB[2]);

	      //===== Find side of each ring that faces the other ring
	      atm_typ NA,NB; 
	      d=DistanceAtoms(atmNA[1],cenB); NA=atmNA[1];
	      if(d > DistanceAtoms(atmNA[2],cenB)) NA=atmNA[2];
	      d=DistanceAtoms(atmNB[1],cenA); NB=atmNB[1];
	      if(d > DistanceAtoms(atmNB[2],cenA)) NB=atmNB[2];

	      //===== Calc angles gamma (normal-to-normal), theta & delta angles
// PutAtom(stderr,cenA); PutAtom(stderr,cenB); PutAtom(stderr,NA);
	      theta=CalcAngle(NA,cenA,cenB); // in degrees...
	      delta=CalcAngle(NB,cenB,cenA); // in degrees...
	      gamma=this->Normal2NormalAngle(cenA,NA,cenB,NB);

	      // Find face2face(F), edge2face(E) and staggered(S)interactions
	      // Fig. 6 in Interdiscip Sci Comput Life Sci (2015) 7:211–220.
	      char typ=' ';
	      if(Rcen <= 7.2){
	        const double cenE=6.5,theE=60.0,gamE=50.0;
	        if(Rcen <= cenE && gamma >= gamE) typ='T'; // T-shaped 
		if(gamma <= 30.0){
	          const double cenF=4.5,theF=10.0;	// face-to-face
	          if(Rcen <= cenF && (theta <= theF  || delta <= theF)) typ='F';
	          const double cenS=5.5,theS=10.0;	// displaced stacked 
	          if(Rcen <= cenS && (theta >= theS && delta >= theS)) typ='D';
		}
	      }
	      BooLean NewA=FALSE,NewB=FALSE;
	      Int4  II=3;
	      if(isalpha(typ)){
		FILE	*aro_fp=0;
		if(X->pdbX == 1) II=0;
	        else if(X->pdbX == (psd->NumPDB()/2)) II=1;
		else if(X->pdbX == psd->NumPDB()) II=2;
	    	if(arofp[II]==0){
		   switch (II){
		     case 0:
			// arofp[0]=open_file(argv[2],"_aroF.pdb","w");
			arofp[0]=tmpfile();
		   	fprintf(arofp[0],"HEADER Fst_H.pdb\n");
		      break;
		     case 1:
	    	   	// arofp[1]=open_file(argv[2],"_aroM.pdb","w");
			arofp[1]=tmpfile();
		   	fprintf(arofp[1],"HEADER Mid_H.pdb\n");
		      break;
		     case 2:
	    	   	// arofp[2]=open_file(argv[2],"_aroL.pdb","w");
			arofp[2]=tmpfile();
		   	fprintf(arofp[2],"HEADER Lst_H.pdb\n");
		      break;
		     case 3:
		      break;
		     default: print_error("FindAroAro input error");
		   }
		} 
	        //===== Print out aromatic rings with centroid virtual atom.
	        if(II < 3){
	          NewA=this->PrintAromaticRing(arofp[II],X,NA,'A',IsPutRes[II][X->CA]);
	          NewB=this->PrintAromaticRing(arofp[II],X,NB,'B',IsPutRes[II][X->CB]);
		}
	      } else typ='I'; // an intermediate conformation...
	      if(NewA || NewB){
		 NumAroAroPi++;
		 const char *ID[]={"fst","mid","lst","nil"};
		 if(logfp == 0) logfp=open_file(argv[2],".log","w");
		 fprintf(logfp,
		  "%c%d%c\t%c%d%c\t%.2lf\t%.2lf\t%.2lf\t%.2lf\t%.2lf\t%c(%s)\n",
			X->aaA,sA,X->ca,X->aaB,sB,X->cb,Rclo,Rcen,
			gamma,theta,delta,typ,ID[II]);
	      }

	      X->Rcen=Rcen; X->gamma=gamma; X->theta=theta; X->Type=typ;
	      NumAroAro++;

	      //========= free up memory ============
	      if(atmNA){ NilAtom(atmNA[1]); NilAtom(atmNA[2]); }
	      if(atmNB){ NilAtom(atmNB[1]); NilAtom(atmNB[2]); }
	      free(atmNA); free(atmNB); 
	      // other atoms borrowed from atmNA, atmNB, etc.
	      dd=X->Rcen;
	} return dd;
}

double	spc_typ::GetAro_PiFast(FILE *fp, Int4 resI,Int4 resJ, float dmax,
		Int4 CA, Int4 CB, Int4 X, char &Type)
/*******************************************************************
   look for staggered aromatic pi-pi stacking...
   Vertical dist = 3.3 - 3.8 Angstroms (or 3.4-3.6);
   horizontal = 1.6-1.8 Angstroms (or 1.4-2.0???).
   Vertical = distance Virtual to Hydrogen in cis; 
   Horizontal = distance Virtual to Hydrogen in trans; 
 *******************************************************************
==== Interdiscip Sci Comput Life Sci (2015) 7:211–220. ====
   Parallel offset:		-2.48 Kcal/mol
      horizontal dist = 1.6-1.8 Angstroms 
      vertical dist = 3.4-3.6 Angstroms
   Edge-to-face (T-shaped):	-2.46Kcal/mol
      Centroid distance = 4.96-5.025 Angstroms 
   Face-to-face:		-1.48 Kcal/mol
      Centroid distance = 3.3-3.8 Angstroms
 *******************************************************************
==== Computational & Structural Biotechnology Journal (2021) 19:5960-5968 ====
   Aromatic groups are in contact if the distance between their centers < 7 Å
 *******************************************************************
=============== JBC (1998) 273(25):15458–15463. =================
   Centroid distance: 
	Parallel stacked: Rcen < 7.5 (use 7.0?); see Fig. 5a. 
	T-shaped: Rcen < 7.5 (use 7.0?); see Fig. 5b. 
   Distance between sidechains:
	Rclo <=4.5 ; see Fig. 3.
   theta = 
   gamma = 
 *******************************************************************
	 // To test for gamma == 90 degrees...
	 double  gamma=this->Normal2NormalAngle(cenA[1],NA,cenA[1],RngA[1]);
	 // To test for gamma == 0 degrees.
	 double  gamma=this->Normal2NormalAngle(cenA[1],NA,cenA[1],NA);
 *******************************************************************/
{
	if(CA ==0 || CB==0){ print_error("chain not found!"); }
	assert(psd != 0);
	double	d,dd=this->BigDist;
	pdb_typ P=psd->RtnPDB(X);
	if(CA > nChainsPDB(P) || CA < 1 || CB > nChainsPDB(P) || CB < 1)
		{ pdb_error("GetAro_PiFast( ): input error"); }
	res_typ ResA=psd->RtnResI(X,CA,resI);
	res_typ ResB=psd->RtnResI(X,CB,resJ);
	if(ResA == 0  || ResB == 0) return dd;
	if(ResB == ResA) return dd;

	// Int4	i,j,n,a,d,r,aI,aJ,rI,rJ;
	double	default_dmax=dmax; dmax=4.5;
	// double	gamma,theta,delta;
	// BooLean	na;
	// char	ca,cb,aaA,aaB;

	a_type	AB=AminoAcidAlphabetPDB(P),nAB=NucleotideAlphabetPDB(P);
//================== Find virtual atoms for centers of aromatic rings ================
	aai_typ aai=MakeAAI(ResA,ResB,AB,nAB);
	if(aai==0) return dd;
	if(X == 1 || X == (psd->NumPDB()/2) || X == psd->NumPDB()) aai->pdbX=X;
	else aai->pdbX=0;
	aai->CA=CA; aai->CB=CB;
	aai->ca=ChainCharPDB(CA,P); aai->cb=ChainCharPDB(CB,P);
	this->AromaticRingAtoms(aai,'A');
	this->AromaticRingAtoms(aai,'B');
	dd=this->FindAroAro(aai); Type=aai->Type;
	if(aai->aaA == 'W'){
	  d=this->FindAroAro(aai,TRUE,FALSE);
	  if(d < dd){ dd=d; Type=aai->Type; }
	}
	if(aai->aaB == 'W'){
	  d=this->FindAroAro(aai,FALSE,TRUE);
	  if(d < dd){ dd=d; Type=aai->Type; }
	  if(aai->aaA == 'W'){
	    d=this->FindAroAro(aai,TRUE,TRUE);
	    if(d < dd){ dd=d; Type=aai->Type; }
	  }
	}
	NilAAI(aai);
	dmax=default_dmax;
        return dd;
}

Int4	spc_typ::RunAllAroAroPi(FILE *ofp)
{
	Int4	N=0,i,j,k,x,y,p,MinRes,MaxRes,nI,nJ;
	FILE	*efp=0;
	Int4	default_minspacing=MinSpacing; MinSpacing=1;

	assert(mode == 'S');
	arofp[0]=arofp[1]=arofp[2]=arofp[3]=0;

	if(pmlfp){ this->PutHeaderPyMOL(); }
	// 1. open up one of the structures for reference.
	fprintf(stderr,"aromatic-aromatic interations done.\n"); // exit(1);
	pdb_typ	pdb=psd->RtnPDB(1); 
	nChain=nChainsPDB(pdb);
	// for(nChain=0 ; Chain[nChain]; ){ j=GetChainNumberPDB(pdb,Chain[i]); }
	Int4	nPDB=psd->NumPDB();
	MinRes=999999; MaxRes=0;
	assert(IsPutRes[0]==0); NEW(IsPutRes[0],nChain+3,set_typ);
	assert(IsPutRes[1]==0); NEW(IsPutRes[1],nChain+3,set_typ);
	assert(IsPutRes[2]==0); NEW(IsPutRes[2],nChain+3,set_typ);
	for(i=0; Chain[i]; i++){  // find start and end common to all chains.
	        j=GetChainNumberPDB(pdb,Chain[i]);
// fprintf(stderr,"Chain[%d]=%c; chn#=%d;",i,Chain[i],j);
		if((x=MinResPDB(j,pdb)) > 0){ MinRes=MINIMUM(Int4,MinRes,x); }
		if(i==0) MaxRes=MAXIMUM(Int4,MaxRes,MaxResPDB(j,pdb));
		else MaxRes=MINIMUM(Int4,MaxRes,MaxResPDB(j,pdb));
// fprintf(stderr,"MaxRes=%d\n",MaxRes);
		IsPutRes[0][j]=MakeSet(MaxRes+3);  // DEBUG
		IsPutRes[1][j]=MakeSet(MaxRes+3);  // DEBUG
		IsPutRes[2][j]=MakeSet(MaxRes+3);  // DEBUG
// PutSet(stderr,IsPutRes[0][j]); PutSet(stderr,IsPutRes[1][j]); PutSet(stderr,IsPutRes[2][j]);
	} // Only need to look at one; they are all the same.
	NumAroAro=NumAroAroPi=0;

//fprintf(stderr,"DEBUG 1\n");
#if 0
	if(sis==0) sis = new sis_typ(Chain,MaxRes);
	if(vsifp){
	  fprintf(vsifp,"~$=1.\n");
	  for(i=1; i <= nChain; i++){
	     fprintf(vsifp,"File%d=%s:%c  //\n",i,psd->RtnPath(1),Chain[i-1]);
	  } fprintf(vsifp,"\n1-10000.W15\n\n");
	  fprintf(vsifp,"%d-%d.Y80\n",MinRes,MaxRes);
	}
#endif

	i=GetChainNumberPDB(pdb,Chain[0]);
	e_type Sq=GetPDBSeq(i,pdb);
	a_type AB=AminoAcidAlphabetPDB(pdb); StrtMtrx=MinRes; EndMtrx=MaxRes;
// PutSeq(stderr,Sq,AB);
	// set_typ useSB=0;
	if(mode=='S' && PlotGrph){	// Create contact matrix...
	  this->CreateMtrx(pdb);
	} // else { useSB=MakeSet(MaxRes+4); FillSet(useSB); }
	Int4	NumVSI=0;
	char	setI=0,setJ=0,rI,rJ,Type;
	Int4	cX,cY;
	float	HA_dmax=1.2,dmax=4.5; 
	double	cut=2.5,cutX=7.2,Cut=0.01;
	if(logfp == 0) logfp=open_file(argv[2],".log","w");
	fprintf(logfp,"------ aromatic-aromatic interactions: ------\n");
	fprintf(logfp,"res1\tres2\tRclo\tRcen\tgamma\ttheta\tdelta\ttype\n");
	for(nI=MinRes; nI < MaxRes; nI++) {
	   if(PtrnSet){
	      if(!MemberSet(nI,PtrnSet[0]) && !show_all && TopDC_Set 
		  	&& !MemberSet(nI,TopDC_Set)) continue;
	      for(Int4 ii=1; PtrnSet[ii]; ii++){
		  if(MemberSet(nI,PtrnSet[ii])) setI=ii; 
	      }
	   }
	   cX=GetChainNumberPDB(pdb,Chain[0]); rI=psd->GetResChar(1,cX,nI);
	   if(rI==0) continue;
	   // if(!MemberSet(nI,useSB)) continue;
	   for(nJ=nI+1; nJ <= MaxRes; nJ++) {
	     if(abs(nI-nJ) < MinSpacing) continue;
	     cY=GetChainNumberPDB(pdb,Chain[0]); rJ=psd->GetResChar(1,cY,nJ);
	     if(rJ==0) continue;
	     if(this->InvalidHbondPair(rI,rJ,'A')) continue; 
		// ^ make sure both residues are aromatic.
	     if(PtrnSet){
	        if(!MemberSet(nJ,PtrnSet[0]) && !show_all && TopDC_Set
			&& !MemberSet(nJ,TopDC_Set)) continue;
		for(Int4 jj=1; PtrnSet[jj]; jj++){
		   if(MemberSet(nJ,PtrnSet[jj])) setJ=jj; 
		}
	     }
	     // if(!MemberSet(nJ,useSB)) continue;
	     //=========== Screen for pairs worthy of a deeper search.
	     double	d,dXX=this->BigDist,dXY=this->BigDist,dYX=this->BigDist;
	     for(x=0; Chain[x]; x++){
	       //-- only examine residues relatively near each other.
	       cX=GetChainNumberPDB(pdb,Chain[x]); 
// fprintf(stderr,"%c%d%c vs %c%d%c\n",rI,nI,Chain[0],rJ,nJ,Chain[x]);
	       if(DistancePDB(nI,"CA",cX,nJ,"CA",cX,pdb) < 30.0){
	          d=this->GetAro_PiFast(efp,nI,nJ,HA_dmax,cX,cX,1,Type);
	          dXX=MINIMUM(double,d,dXX);	// Dist cis
	       } 
	       if(nChain > 1){
		  y=x+1; if(Chain[y]==0) y=0; 
	          cY=GetChainNumberPDB(pdb,Chain[y]);
	          if(DistancePDB(nI,"CA",cX,nJ,"CA",cY,pdb) < 30.){
		    d=this->GetAro_PiFast(efp,nI,nJ,HA_dmax,cX,cY,1,Type);
	            dXY=MINIMUM(double,d,dXY);
		  }
	          if(DistancePDB(nI,"CA",cY,nJ,"CA",cX,pdb) < 30.0){
		    d=this->GetAro_PiFast(efp,nI,nJ,HA_dmax,cY,cX,1,Type);
	            dYX=MINIMUM(double,d,dYX);
		  }
	       }
	     }
	     char Str[150],str[10],StrI[100],StrJ[100],strTmp[20];
	     Int4 os=OffSetSeq(Sq);
	     char rI=AlphaChar(ResSeq(nI-os,Sq),AB);
	     char rJ=AlphaChar(ResSeq(nJ-os,Sq),AB);
	     sprintf(str,"sc2sc");
	     if(mode != 'S') print_error("This should not happen");
	     this->GetSeqContext(StrI,nI-os-1,3,Sq,AB); 
	     this->GetSeqContext(StrJ,nJ-os-1,3,Sq,AB);
	     sprintf(strTmp,"aro-to-aro");
	     if(dXX < cutX && dXX > Cut){	// cis: I vs J
	       sprintf(Str,"%c%d vs %c%d: %s %.2lf (%s)\n",rI,nI,rJ,nJ,strTmp,dXX,str);
	       d=this->RunOneAroAroFast(nI,nJ,FALSE,cut,ofp,Str,StrI,StrJ);
	     }
	     if(dXY < cutX && dXY > Cut){	// trans: I vs J
	       sprintf(Str,"%c%d vs %c%d: %s %.2lf (%s)\n",rI,nI,rJ,nJ,strTmp,dXY,str);
	       d=this->RunOneAroAroFast(nI,nJ,TRUE,cut,ofp,Str,StrI,StrJ);
	     }
	     if(dYX < cutX && dYX > Cut) {
	       sprintf(Str,"%c%d vs %c%d: %s %.2lf (%s)\n",rI,nI,rJ,nJ,strTmp,dYX,str);
	       d=this->RunOneAroAroFast(nI,nJ,TRUE,cut,ofp,Str,StrI,StrJ);
	     }
	     // } // end if((Dd <= cutX && Dd > Cut) || (DD <= cutX && DD > Cut)...
	  } // end for(nJ=nI+1; nJ <= MaxRes; nJ++)
	} // end for(nI=MinRes; nI <= MaxRes; nI++)
	// if(useSB) NilSet(useSB); 
	NilSeq(Sq); 
	if(logfp){
	  fprintf(logfp," %d pi-pi interactions among %d aromatic pairs\n",
			NumAroAroPi,NumAroAro);
	}
	for(j=0; j <=2; j++){
          for(i=1; i <= nChain; i++) if(IsPutRes[j][i] != 0) NilSet(IsPutRes[j][i]); 
          free(IsPutRes[j]); IsPutRes[j]=0; 
	}
	FILE *tfp=open_file(argv[2],"_aro.pdb","w");
	for(j=0; j <= 2; j++){
	   if(arofp[j]){
	      fprintf(arofp[j],"END\n");
	      char c; rewind(arofp[j]); 
	      while((c=getc(arofp[j])) != EOF) fprintf(tfp,"%c",c);
	      fclose(arofp[j]); arofp[j]=0;
	   }
	} fclose(tfp);
	
    // fprintf(stderr,"set sphere_scale,0.3,all\n");
#if 0
	if(vsifp){ fprintf(vsifp,"\n\n"); }
	if(pmlfp) this->PutTailPyMOL();
#endif
	MinSpacing=default_minspacing;
	return 0;
}

double	spc_typ::RunOneAroAroFast(Int4 nI,Int4 nJ, BooLean trans,double cut,
		FILE *outfp,char *Msg,char *StrI, char *StrJ)
// Compare residue nI and nJ for cis and trans aromatic-aromatic interactions
{
	double	dd,Dd,*Sum,**Value; 
	float	HA_dmax=2.5,dmax=3.6; 
	FILE	*efp=0,*ofp=0;		// efp=ofp=stderr;
	Int4	i,j,z,N,cI,cJ,nPDB,Mid,*NN;
	nPDB=psd->NumPDB(); Mid=nPDB/2;
	tri_typ	*tri=new tri_typ(Chain,nPDB);

	assert(mode == 'S'); assert(psd != 0);
	NEWP(Value,10,double);
	char	**Typ; NEWP(Typ,10,char);
	for(i=0; i <=3; i++){
	   NEW(Value[i],this->nChain+3,double); NEW(Typ[i],this->nChain+3,char); 
	}
	NEW(Sum, this->nChain+3,double); NEW(NN, this->nChain +3, Int4);

	if(0 && cut > 0.0) ofp=tmpfile(); 
	if(verbose) efp=stderr;

	//====== loop over all pdb files... ======
	for(N=1; N <= nPDB; N++){
	  pdb_typ pdb = psd->RtnPDB(N);
	  if(trans) j=1; else j=0;
	  for(i=0 ; Chain[i]; i++,j++){
	   if(Chain[j]==0) j=0;
	   cI=GetChainNumberPDB(pdb,Chain[i]);
	   cJ=GetChainNumberPDB(pdb,Chain[j]);
#if 0
	   if(ofp){
	     if(N==1 && i==0){
	       fprintf(ofp,"index\t");
	       for(Int4 x=0,y=1; Chain[x];y++,x++){
		if(Chain[y]==0) y=0; 
		if(trans) fprintf(ofp,"%c%c",Chain[x],Chain[y]);
		else fprintf(ofp,"%c",Chain[x]);
		if(y != 0) fprintf(ofp,"\t");
	       } fprintf(ofp,"\n"); 
	     }
	     if(i==0) fprintf(ofp,"%d\t",N);
	   }
#endif
	   if(cI == 0 || !IsProteinPDB(cI,pdb)){
		if(ofp){ fprintf(ofp,"-"); if(Chain[i+1]) fprintf(ofp,"\t"); }
		continue;
	   } if(cJ == 0 || !IsProteinPDB(cJ,pdb)){
		if(ofp){ fprintf(ofp,"-"); if(Chain[i+1]) fprintf(ofp,"\t"); }
		continue;
	   }
	   char typ=0;
	   dd=this->GetAro_PiFast(efp,nI,nJ,HA_dmax,cI,cJ,N,typ);
#if 0
	   if(efp) fprintf(efp,"===== %d->%d%c:%d%c %.2lf ==============\n",
					N,nI,Chain[i],nJ,Chain[j],dd);
	   if(ofp){
	      if(dd >= BigDist){
	        fprintf(ofp,"...."); fflush(ofp); 
	        // fprintf(ofp,"%.1lf",dd);
	      } else fprintf(ofp,"%.2lf",dd);
	   }
#endif
	   tri->AddValue(N,dd,i,typ);
// if(nI==234 && nJ==517) fprintf(stderr,"Value(%d,%i)=%.2lf; typ=%c\n",N,i,dd,typ);
	   if(N==1){ Value[0][i]=dd; Typ[0][i]=typ; }
	   else if(N==Mid){ Value[1][i]=dd; Typ[1][i]=typ; }
	   else if(N==nPDB){ Value[2][i]=dd; Typ[2][i]=typ; }
	
	   if(ofp && Chain[i+1]) fprintf(ofp,"\t"); 
	   if(dd < this->UnDefHBond){ Sum[i] += dd; NN[i]++; }
	  } if(ofp) fprintf(ofp,"\n"); 
	  // tri->Put(stderr,N);
	}
	// NEW(Value[3],this->nChain +3, double); allocated above...
	double	rtn=DBL_MAX;
	char	nhits=0;
// ofp=stdout;
	if(1 || efp && this->nChain > 0){
	   if(ofp) fprintf(ofp,"avg.\t");
	   for(Int4 x=0; Chain[x];x++){
	     dd=Sum[x]/(double) NN[x]; 
	     if(ofp) fprintf(ofp,"%.2lf",dd);
	     Value[3][x]=dd;
	     if(ofp && Chain[x+1]) fprintf(ofp,"\t");
	     rtn=MINIMUM(double,rtn,dd);
#if 0
	     if(PlotGrph && Mtrx){
		if(dd <= 2.5){
		   Mtrx[nI][nJ]++; Mtrx[nJ][nI]++; nhits++;
		}
	     }
	     if(dd <= 7.5){
		Int4 y,r;
		if(trans){	// x = Ct & y = Rt or x = Lt & y = Ct
	   	   if(Chain[x+1]==0) y=0; else y=x+1;
		   if(sis){
		     sis->AddSite(Ct,x,nI); sis->AddSite(Rt,x,nJ);
		     sis->AddSite(Lt,y,nI); sis->AddSite(Ct,y,nJ);
		   }
		} else {	// x = Ct & x = Ct
		   if(sis){ sis->AddSite(Ct,x,nI); sis->AddSite(Ct,x,nJ); }
		}
	     }
#endif
	   } if(ofp) fprintf(ofp,"\n"); 
	} free(Sum); free(NN);
#if 0
	if(nhits > 0 && PlotGrph && Mtrx){
		if(mode=='S'){ AddSet(nI,IsSC[nJ]); AddSet(nJ,IsSC[nI]); }
		Mtrx[nI][0]++; Mtrx[nJ][0]++; 
		JoinWdgraph(nI,nJ,this->nChain-nhits,wdg);
		JoinWdgraph(nJ,nI,this->nChain-nhits,wdg);
	}
	if(ofp && cut > 0){
	   if(rtn <= cut){ 
		if(Msg) fprintf(outfp,"%s",Msg);
		char c;
		if(ofp){
		    rewind(ofp); 
		    while((c=fgetc(ofp)) != EOF) fprintf(outfp,"%c",c); 
		}
	   }
	} if(ofp) fclose(ofp);
#endif
	char cmd[200],str[10],CT='C';
	if(trans) CT='T';
	if(mode == 'S') sprintf(str,"sc2sc");
	else print_error("This should not happen");
	sprintf(cmd,"sparc %s %s %s %d:%d%c\n",str,argv[2],argv[3],nI,nJ,CT);
	keytyp key=0;
#if 0
	char **msgF=tri->RtnResults(key,Msg,cmd,trans,Value,'F',Typ);
	if(msgF){
	   if(rphAF==0) rphAF = new rph_typ(rpheap_size,this->nChain);
	   for(z=0; msgF[z]; ) z++;
	   msgF[z] = AllocString(StrI); z++; msgF[z] = AllocString(StrJ);
	   rphAF->Insert(key,msgF);
	}
#endif
	char **msgS=tri->RtnResults(key,Msg,cmd,trans,Value,'S',Typ);
	if(msgS){
	   // for(z=0; msgS[z]; z++) fprintf(stderr,"%s",msgS[z]); fprintf(stderr,"\n");
	   for(z=0; msgS[z]; ) z++;
	   msgS[z] = AllocString(StrI); z++; msgS[z] = AllocString(StrJ);
	   // z++; msgS[z]=0; // set to zero by tri_typ...
	   if(rphAS==0) rphAS = new rph_typ(rpheap_size,this->nChain);
	   rphAS->Insert(-key,msgS);
	} // fprintf(stderr,"msgS key=%.2lf\n",key);
	for(i=0; i<=3; i++) if(Value[i]) free(Value[i]); free(Value);
	for(i=0; i<=3; i++) if(Typ[i]) free(Typ[i]); free(Typ);
	delete tri; tri=0;
	return rtn;
}

