/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in the
    Software without restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the
    following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/
#include "dds_typ.h"

res_typ *dds_typ::RES2COLS(res_typ *RsX,Int4 nres,Int4 &N)
{
	Int4	i,j;
	res_typ	*ColX; NEW(ColX,nres+3,res_typ);
	for(j=0,i=1; i <= nres; i++){
	  if(RsX[i] != 0){ j++; ColX[j]=RsX[i]; }
	} N=j; return ColX;
}

char *dds_typ::MSA2COLS(char *RsX,Int4 &N)
{
	Int4	i,j;
	char	c,*ColX;
	NEW(ColX,strlen(RsX)+3,char);
	for(j=0,i=0; RsX[i]; i++){
	    c=RsX[i];
	    if(isupper(c) || c == '-'){ j++; ColX[j]=c; }
	} N=j; return ColX;
}

res_typ *dds_typ::CheckMSA(res_typ *AllRes,Int4 nres,char *RsX,Int4 &NumCol)
// Check the consistency between the MSA and the res_typ array
{
	Int4	i,I,col,N,NN,end;
	char	x,c,*ColX;
	res_typ *colres;
	ColX=MSA2COLS(RsX,N);
	colres=RES2COLS(AllRes,nres,NN);
	if(efp) fprintf(efp,"N=%d; NN=%d\n",N,NN);
	if(NN < N){	// pdb file ends before seq file...
	   res_typ *tmp; NEW(tmp, N + 3, res_typ);
	   for(i=0; i <= NN; i++) tmp[i]=colres[i];
	   free(colres); colres=tmp;
	   while(NN < N){ NN++; colres[NN]=deleted; }
	}
	if(N != NN){	// i.e., if(NN > N)
	   fprintf(stderr,"N = %d; NN = %d\n",N,NN);
	   fprintf(stderr,"RsX = %s\n",RsX);
	   fprintf(stderr,"colres = ");
	   for(i=1; colres[i]; i++){
	      fprintf(stderr,"%c",GetCharResidue(colres[i],AB));
	   } fprintf(stderr,"\n");
	   fprintf(stderr,"ColX   = %s\n",ColX+1);
	   print_error("MSA & pdb inconsistent aligned lengths");
	} NumCol=NN; 
	for(i=1; i <= NN; i++){		// i -> MSA #1
	  if(colres[i] == deleted){
		I=0; c='-'; x='*'; 
	  } else {
	        I=ResidueID(colres[i]); c=GetCharResidue(colres[i],AB);
	        if(c != ColX[i] && toupper(ColX[i]) == 'X') ColX[i]=c;
	        else if(c != 'X' && c != ColX[i]){
		    fprintf(stderr,"sqI=%s\n",ColX +1);
		    fprintf(stderr,"%c%d != %c%d\n",c,I,ColX[i],i);
		    fprintf(stderr,"pdb=");
		    for(i=1; colres[i]; i++){
		      if(colres[i]==deleted) fprintf(stderr,"-");
		      else fprintf(stderr,"%c",GetCharResidue(colres[i],AB));
		    } fprintf(stderr,"\n");
		    print_error("MSA & pdb residues inconsistent");
		} x = ' ';
	  }
	  // fprintf(stderr,"%d. %c%d <--> %c%d%c\n",i,c,I,ColX[i],i-AdjstX,x);
	}  free(ColX); colres[0]=deleted;
	return colres;
}

float	**dds_typ::CalcDist(res_typ *AllRes,Int4 nrsX)
{
	Int4	i,j,k,I,J,ai,aj;
	res_typ rsI,rsJ;
	atm_typ atmI,atmJ;
	float	**Dist; 
	NEWP(Dist,nrsX + 3, float);
	for(i=1; i <= nrsX; i++){ NEW(Dist[i],nrsX+3, float); }
	for(i=1; i <= nrsX; i++){
	     for(j=i; j <= nrsX; j++){
	        Dist[i][j]=Dist[j][i]=-1.0;
	        if(AllRes[i] == 0) continue;
	        if(AllRes[j] == 0) continue;
	      	if(i == j) continue;
	        if(AllRes[i] == deleted || AllRes[j] == deleted) continue;
	        I=ResidueID(AllRes[i]); 
		J=ResidueID(AllRes[j]);
// fprintf(stderr,"I = %d; J = %d; i=%d; j=%d; nres=%d\n",I,J,i,j,nrsX);
		if(abs(I-J) <= 3) continue;
		rsI=AllRes[i]; 
	   	for(ai=1; ai <= ResidueAtomNumber(rsI); ai++){
		  atmI=AtomResidue(ai,rsI);
		  if(AlphaCarbonAtom(atmI)) break; else atmI=0;
		}
	        if(atmI == 0) continue;
		rsJ=AllRes[j];
	   	for(aj=1; aj <= ResidueAtomNumber(rsJ); aj++){
		  atmJ=AtomResidue(aj,rsJ);
		  if(AlphaCarbonAtom(atmJ)) break; else atmJ=0;
		}
	        if(atmJ == 0) continue;
	        Dist[i][j]=Dist[j][i]=DistanceAtoms(atmI,atmJ);
	     }
	} return Dist;
}

double dds_typ::CalcDistDiff(res_typ **AllRes,Int4 *nres,Int4 &CntUsd,set_typ not_usd)
{
	Int4	i,j,I,J,II,JJ,n=0,colI,colII,colJ,colJJ;
	float **Dist[4];
	double	score=0.0;
	char	cI,cII,cJ,cJJ;
	Dist[X]=CalcDist(AllRes[X],nres[X]);
	Dist[Y]=CalcDist(AllRes[Y],nres[Y]);
	for(colI=0,i=1; i <= nres[X]; i++){ if(AllRes[X][i] != 0) colI++; }
	for(colJ=0,i=1; i <= nres[Y]; i++){ if(AllRes[Y][i] != 0) colJ++; }
	if(colI != colJ){
	  for(i=1; i <= nres[X]; i++){		// i -> MSA #1
	    if(AllRes[X][i] == deleted){ fprintf(stderr,"%d. '-'\n",i); continue; }
	    I=ResidueID(AllRes[X][i]); cI=GetCharResidue(AllRes[X][i],AB);
	    fprintf(stderr,"%d. %c%d\n",i,cI,I);
	  }
	  for(i=1; i <= nres[Y]; i++){		// i -> MSA #1
	    if(AllRes[Y][i] == deleted){ fprintf(stderr,"%d. '-'\n",i); continue; }
	    I=ResidueID(AllRes[Y][i]); cI=GetCharResidue(AllRes[Y][i],AB);
	    fprintf(stderr,"%d. %c%d\n",i,cI,I);
	  }
	  fprintf(stderr,"colI = %d; colJ = %d\n",colI,colJ);
	} assert(colI == colJ);
	h_type HG=0; // HG=Histogram("distance difference",0,5,0.1);
	assert(nres[X] == nres[Y]);
	for(n=0,i=1; i <= nres[X]; i++){		// i -> MSA #1
if(not_usd && MemberSet(i,not_usd)) continue;
	  if(AllRes[X][i] != deleted && AllRes[Y][i] != deleted){
	    I=ResidueID(AllRes[X][i]); II=ResidueID(AllRes[Y][i]);
	    cI=GetCharResidue(AllRes[X][i],AB);
	    cII=GetCharResidue(AllRes[Y][i],AB);
	    if(efp) fprintf(efp,"%d. %c%d <--> %c%d\n",i,cI,I,cII,II);
	  } else continue;
	  for(j=i+1; j <= nres[X]; j++){	
	    if(AllRes[X][j] == deleted || AllRes[Y][j] == deleted) continue;
if(not_usd && MemberSet(j,not_usd)) continue;
	    J=ResidueID(AllRes[X][j]); JJ=ResidueID(AllRes[Y][j]);
	    if(abs(J-I) < 5) continue;
	    if(abs(JJ-II) < 5) continue;
	    cJ=GetCharResidue(AllRes[X][j],AB);
	    cJJ=GetCharResidue(AllRes[Y][j],AB);
if(efp) fprintf(stderr,"  %d. %c%d <--> %c%d\n",j,cJ,J,cJJ,JJ);
	    if(Dist[X][i][j] > 0.0 && Dist[Y][i][j] > 0.0){
		  double dd=(double) fabs(Dist[X][i][j] - Dist[Y][i][j]);
		  assert(dd >= 0.0);
		  n++; score += dd;   
		  if(HG) IncdHist(dd,HG);
if(efp && dd >= 15.0) fprintf(stderr,"%d.%d: %c%d %c%d vs %c%d %c%d dd=%.3lf\n",
		i,j,cI,I,cJ,J,cII,II,cJJ,JJ,dd);
	    }
	  }
	}
	for(i=1; i <= nres[X]; i++){		// MSA #1
	   if(Dist[X][i]) free(Dist[X][i]); 
	}
	for(i=1; i <= nres[Y]; i++){		// MSA #2
	   if(Dist[Y][i]) free(Dist[Y][i]);
	} free(Dist[X]); free(Dist[Y]);
	if(HG){ PutHist(stderr,60,HG); NilHist(HG); }
	CntUsd += n;
	score = score/(double)n;
if(n < minCol) print_error("Too few columns without gaps; exiting");
// if(n < 1) print_error("Too few columns without gaps; exiting");
assert(score >= 0.0 && score <= 9999.9);
	return score;
}

set_typ dds_typ::FindCommonRes(res_typ *RsA, res_typ *RsB)
{
	Int4	i,j,nA,nB,max,I,J;
	for(nA=max=0,i=1; RsA[i]; i++){ nA++; I=ResidueID(RsA[nA]); if(I > max) max=I; }
	for(nB=0,i=1; RsB[i]; i++){ nB++; I=ResidueID(RsB[nB]); if(I > max) max=I; }
	set_typ SetA=MakeSet(max+3); ClearSet(SetA);
	set_typ SetB=MakeSet(max+3); ClearSet(SetB);
	set_typ SetAB=CopySet(SetA);
	for(i=1; RsA[i]; i++){  I=ResidueID(RsA[i]); AddSet(I,SetA);}
	for(i=1; RsB[i]; i++){  I=ResidueID(RsB[i]); AddSet(I,SetB);}
	UnionSet3(SetA,SetB,SetAB);
	PutSet(stderr,SetA); PutSet(stderr,SetB);
	NilSet(SetA); NilSet(SetB);
	return SetAB;
}

void    dds_typ::Init()
{
        char ATM[]="ATOM      2  CA   XX A   0      27.000 -13.000  21.000  1.00  0.00";
        NEW(atm0,3,atm_typ); atm0[1]=MkAtom(ATM);
        deleted=MkRes(1,1,atm0);
        AB=AminoAcidAlphabetPDB(PDB[1]); NP=0;
        for(Int4 i=1; PDB[i]; i++) NP++;
	minCol=4;

	Int4	i,j,k,col,min,max,x,y,MaxLen=0;
	NEWP(AllRes, NP+3, res_typ);
	NEWP(ColRes[X], NP+3, res_typ); NEWP(ColRes[Y], NP+3, res_typ);
	NEW(nresX, NP+3,Int4); 
	NEW(ncolX[A],NP+3,Int4); NEW(ncolX[B],NP+3,Int4); 
	//============== Get aligned residues in PDB file ==============
// efp=stderr;
	for(x=1; x <= NP; x++){
	   Int4 chnK=GetChainNumberPDB(PDB[x],chn[x]);
	   AllRes[x]=MakeResPDB(chnK,&nresX[x],PDB[x]);
	   min=MinResPDB(chnK,PDB[x]); max=MaxResPDB(chnK,PDB[x]);
	   if(efp){
	     fprintf(efp,
		"=== %d.%s %c (%d..%d; %d res.; adjst=%d,%d) ===\nRsX=%s\nRsY=%s\n",
		x,FilenamePDB(PDB[x]),chn[x],min,max,nresX[x],AdjstX[x],AdjstY[x],
		ResX[x],ResY[x]);
	   } AllRes[x][0]=deleted;
	   i=strlen(ResX[x]); j=strlen(ResY[x]);
	   if(i > MaxLen) MaxLen=i; 
	   if(j > MaxLen) MaxLen=j; 
BooLean found=FALSE;
#if 0
if(strcmp(FilenamePDB(PDB[x]),"4enz")==0) efp=stderr;
if(strcmp(FilenamePDB(PDB[x]),"4bdv")==0) efp=stderr; 
#endif
	   ColRes[A][x]=GetCol2K(ResX[x],min,max,AdjstX[x],AllRes[x],ncolX[A][x]);
#if 1
	   if(ColRes[A][x]==0){
		fprintf(stderr,"%d.%s %c is problematic; msa A\n",
				x,FilenamePDB(PDB[x]),chn[x]);
		print_error("GetCol2K() input error");
	   }
#endif
// efp=0;
#if 0
if(strcmp(FilenamePDB(PDB[x]),"4bdv")==0){ efp=stderr; }
#endif
	   ColRes[B][x]=GetCol2K(ResY[x],min,max,AdjstY[x],AllRes[x],ncolX[B][x]);
#if 1
	   if(ColRes[B][x]==0){
		fprintf(stderr,"%d.%s %c is problematic; msa B\n",
				x,FilenamePDB(PDB[x]),chn[x]);
		print_error("GetCol2K() input error");
	   }
#endif
#if 0
#endif
// efp=0;
	   ColRes[A][x][0]=deleted; ColRes[B][x][0]=deleted;
// if(found) exit(1);
	}
	res_typ *AR[4];
	Int4	I=0,J=0,nr[4];
	for(x=1; x < NP; x++){
	   for(Int4 xx=x+1; xx <= NP; xx++){
		if(ncolX[A][x] != ncolX[A][xx]){
	           fprintf(stderr,"=== %d.%s_%c vs %d.%s_%c\n",x,
			FilenamePDB(PDB[x]),chn[x],xx,FilenamePDB(PDB[xx]),chn[xx]);
	   	   fprintf(stderr,"ncolA[%d]=%d;",x,ncolX[A][x]);
	           fprintf(stderr,"ncolA[%d]=%d\n",xx,ncolX[A][xx]);

		   fprintf(stderr,"Rs%d(%s)=%s\nColI=",x,FilenamePDB(PDB[x]),ResX[x]);
		   for(Int4 s=1; ColRes[A][x][s]; s++){
		        fprintf(stderr,"%c",GetCharResidue(ColRes[A][x][s],AB));
		   } fprintf(stderr,"\n");
		   fprintf(stderr,"Rs%d(%s)=%s\nColJ=",xx,FilenamePDB(PDB[xx]),ResX[xx]);
		   for(Int4 s=1; ColRes[A][xx][s]; s++){
		        fprintf(stderr,"%c",GetCharResidue(ColRes[A][xx][s],AB));
		   } fprintf(stderr,"\n");
		} assert(ncolX[A][x] == ncolX[A][xx]);
		if(ncolX[B][x] != ncolX[B][xx]){
	           fprintf(stderr,"=== %d.%s_%c vs %d.%s_%c\n",x,FilenamePDB(PDB[x]),
				chn[x],xx,FilenamePDB(PDB[xx]),chn[xx]);
	   	   fprintf(stderr,"ncolB[%d]=%d;",x,ncolX[B][x]);
	           fprintf(stderr,"ncolB[%d]=%d\n",xx,ncolX[B][xx]);

		   fprintf(stderr,"Rs%d=%s\nColI=",x,ResY[x]);
		   for(Int4 s=1; ColRes[B][x][s]; s++){
		        fprintf(stderr,"%c",GetCharResidue(ColRes[B][x][s],AB));
		   } fprintf(stderr,"\n");
		   fprintf(stderr,"Rs%d=%s\nColJ=",xx,ResY[xx]);
		   for(Int4 s=1; ColRes[B][xx][s]; s++){
		        fprintf(stderr,"%c",GetCharResidue(ColRes[B][xx][s],AB));
		   } fprintf(stderr,"\n");
		} assert(ncolX[B][x] == ncolX[B][xx]);
	   }
	}
	fflush(stderr);
	// Check consistency between MSA & ColRes
	NotUsd[A] = MakeSet(MaxLen + 3); ClearSet(NotUsd[A]);
	NotUsd[B] = MakeSet(MaxLen + 3); ClearSet(NotUsd[B]);
	for(x=1; x <= NP; x++){
	    AR[X]=ColRes[A][x]; nr[X]=ncolX[A][x]; 
	    if(efp) fprintf(stderr,"===== MSA 1: seq %d (%s) =====\n",
				x,FilenamePDB(PDB[x]));
	    res_typ *tmp=CheckMSA(ColRes[A][x],ncolX[A][x],ResX[x],col);
	    free(ColRes[A][x]); ColRes[A][x]=tmp; ncolX[A][x]=col;
	    tmp=CheckMSA(ColRes[B][x],ncolX[B][x],ResY[x],col);
	    free(ColRes[B][x]); ColRes[B][x]=tmp; ncolX[B][x]=col;
#if 0
	    set_typ SetAB=FindCommonRes(ColRes[A][x],ColRes[B][x]);
	    PutSet(stderr,SetAB);
	//IntersectSet1(set_typ B1, set_typ B2,set_typ IB);
#else
	    i=AddGaps(ColRes[A][x],NotUsd[A]);
	    j=AddGaps(ColRes[B][x],NotUsd[B]);
	    if(I > 0){
		if(i != I){ 
		   for(j=1; ColRes[A][x][j]; j++){
		      if(ColRes[A][x][j] == deleted) fprintf(stderr,"-");
		      else fprintf(stderr,"%c",GetCharResidue(ColRes[A][x][j],AB));;
		   }
		   fprintf(stderr,"\n%d. i=%d; I=%d; Adjst=%d; file=%s\n",
			x,i,I,AdjstX[x],FilenamePDB(PDB[x]));
		   print_error("dds_typ::Init(){ AddGaps(); } input error");
		}
	    } I=i;
	    if(J > 0){
		if(j != J){ 
		   fprintf(stderr,"%d. j=%d; J=%d; Adjst=%d; file=%s\n",
			x,j,J,AdjstY[x],FilenamePDB(PDB[x]));
		   print_error("dds_typ::Init(){ AddGaps(); } input error");
		}
	    } J=j;
#endif
	}
	i=CardSet(NotUsd[A]); j=CardSet(NotUsd[B]);
	if((I-i) < minCol || (J-j) < minCol){
	   fprintf(stderr,"MSA1: %d - %d = %d; MSA2: %d - %d = %d\n",I,i,I-i,J,j,J-j);
	   fprintf(stderr,"Too few ungapped MSA columns\n"); 
	   print_error("Remove structural seqs with lots of gaps and rerun.");
	}
// efp=0;
	if(efp){
	  PutSet(stderr,NotUsd[A]); PutSet(stderr,NotUsd[B]);
	  fprintf(stderr,"A: %d - %d = %d; B: %d - %d = %d\n",I,i,I-i,J,j,J-j);
	}
}

Int4	dds_typ::AddGaps(res_typ *col_res,set_typ Omit)
{
	Int4	i;
	for(i=1; col_res[i]; i++) if(col_res[i] == deleted) AddSet(i,Omit);
	return i-1;
}

double	**dds_typ::Put(FILE *ofp,double cutoff)
{
	//=============== Distance Difference Scores =================
	h_type  HG=0,HG2=0;
	HG=Histogram("MSA1: Distance difference scores",0,3,0.05);
	HG2=Histogram("MSA2: Distance difference scores",0,3,0.05);
	Int4	i,j,*NumUsd[4];
	double  dDD_avg,avgPrs[2],**score=this->Run(stdout,NumUsd);
	NP=this->RtnNP();
	double	**result; NEWP(result,8,double);
	NEW(result[1],NP+3,double); NEW(result[2],NP+3,double);
	NEW(result[3],NP+3,double); NEW(result[4],NP+3,double);
	for(j=1; j <=NP; j++){
	   IncdHist(score[0][j],HG); IncdHist(score[1][j],HG2);
	} avgPrs[0]=avgPrs[1]=0.0;
	double	mean1=MeanHist(HG),mean2=MeanHist(HG2);
	double	stdev1=sqrt(VarianceHist(HG)),stdev2=sqrt(VarianceHist(HG2));
	if(ofp){
	  fprintf(ofp,"========= Distance difference scores ========\n"); 
	  fprintf(ofp,"\nn\tdD1\tsd1\tpairs\tdD2\tsd2\tpairs\tpdbid\n");
	}
	for(dDD_avg=0.0,j=1; j <=NP; j++){
	   double dDD=score[0][j]-score[1][j];
	   double dd1=(score[0][j] - mean1)/stdev1;
	   double dd2=(score[1][j] - mean2)/stdev2;
	   result[1][j]=dd1; result[2][j]=dd2;
	   result[3][j]=score[0][j]; result[4][j]=score[1][j];;
	   if(ofp){
	     if(dd1 < 0.0) fprintf(ofp,"%d\t%.2lf\t-\t%d\t",
			j,score[0][j],NumUsd[0][j]); 
	     else fprintf(ofp,"%d\t%.2lf\t%.2lf\t%d\t",
			j,score[0][j],dd1,NumUsd[0][j]); 
	     if(dd2 < 0.0) fprintf(ofp,"%.2lf\t-\t%d\t%s%c",
			score[1][j],NumUsd[1][j],PDBID[j],chn[j]); 
	     else fprintf(ofp,"%.2lf\t%.2lf\t%d\t%s%c",
			score[1][j],dd2,NumUsd[1][j],PDBID[j],chn[j]); 
	     fprintf(ofp,"\n");
	   }
	   dDD_avg += dDD;
	   avgPrs[0] += NumUsd[0][j]; avgPrs[1] += NumUsd[1][j];
	}
	if(ofp) fprintf(ofp,"avg\t%.2lf\t-\t%.0lf\t%.2lf\t-\t%.0lf (with outliers)\n",
	  score[0][0],avgPrs[0]/(double)NP,score[1][0],avgPrs[1]/(double)NP);
	do { i=this->Refine(result,score,cutoff); } while(i > 0);
#if 1
	Int4	np=0;
	double	aP[3],sc[3]; aP[0]=aP[1]=0.0; sc[0]=sc[1]=0.0; 
	for(j=1; j <=NP; j++){
	  if(result[1][j] < cutoff && result[2][j] < cutoff){
		np++; sc[0] += score[0][j]; sc[1] += score[1][j];
	   	aP[0] += NumUsd[0][j]; aP[1] += NumUsd[1][j];
	  }
	}
	if(ofp) fprintf(ofp,"avg\t%.2lf\t-\t%.0lf\t%.2lf\t-\t%.0lf (no outliers)\n\n",
	  	sc[0]/(double)np,aP[0]/(double)np,sc[1]/(double)np,aP[1]/(double)np);
#endif
	free(NumUsd[0]); free(NumUsd[1]);
	free(score[0]); free(score[1]); free(score);
	this->GetNotUsed(i,j);
	fprintf(ofp,"\tMSA 1: %d cols, 1 SD=%.3lf; MSA 2: %d cols, 1 SD=%.3lf\n\n",
		LenX-i,stdev1,LenY-j,stdev2);
	if(efp){ PutHist(efp,60,HG); PutHist(efp,60,HG2); }
	NilHist(HG); NilHist(HG2); fflush(ofp);
	return result;
}

Int4	dds_typ::Refine(double **stdev,double **score,double cutoff)
{
	h_type HG=Histogram("MSA1: Distance difference scores",0,3,0.05);
	h_type HG2=Histogram("MSA2: Distance difference scores",0,3,0.05);
	Int4	i,j,hits=0;
	for(j=1; j <=NP; j++){
	   if(stdev[1][j] < cutoff) IncdHist(score[0][j],HG); 
	   if(stdev[2][j] < cutoff) IncdHist(score[1][j],HG2);
	} 
	double	mean1=MeanHist(HG),mean2=MeanHist(HG2);
	double	stdev1=sqrt(VarianceHist(HG)),stdev2=sqrt(VarianceHist(HG2));
	for(j=1; j <=NP; j++){
	   double dd1=(score[0][j] - mean1)/stdev1;
	   double dd2=(score[1][j] - mean2)/stdev2;
	   if(stdev[1][j] < cutoff){
		stdev[1][j]=dd1; if(dd1 >= cutoff) hits++; 
	   }
	   if(stdev[2][j] < cutoff){
		stdev[2][j]=dd2; if(dd2 >= cutoff) hits++; 
	   }
	}
// efp=stderr;
	if(efp){ PutHist(efp,60,HG); PutHist(efp,60,HG2); }
// efp=0;
	NilHist(HG); NilHist(HG2); 
	return hits;
}

double	**dds_typ::Run(FILE *ofp,Int4 *NumUsd[])
#if 0
  input: pdb_typ *PDB,char *chn, char **ResX,char **ResY, Int4 *AdjstX, 
	Int4 *AdjstY
  output: the distance difference between residue pairs in X aligned with 
	  residue pairs in Y
#endif
{
	Int4	i,j,k,x,y;
	res_typ *AR[4];
	//=========== Calculate Distance Differences for MSA A and B ===========
	Int4	Num,NumY=0,NumX=0,nr[4];
	double	**score; NEWP(score, 4,double);
	NEW(score[0], NP+3,double); NEW(score[1], NP+3,double);
	
	double	sum=0.0,sumX=0.0, sumY=0.0,scoreX,scoreY;

	NEW(NumUsd[A],NP + 3, Int4);
	NEW(NumUsd[B],NP + 3, Int4);
	h_type HG=0; 
	if(efp) HG=Histogram("MSA 1: avg. distance difference",0,5,0.25);
	for(x=1; x <= NP; x++){		// MSA #1
	   AR[X]=ColRes[A][x]; nr[X]=ncolX[A][x]; 
	   sum=0.0; Num=0;
	   for(y=1; y <= NP; y++){	// MSA #2.
	     if(x == y) continue;
	     if(efp) fprintf(stderr,"===== %d vs %d (%s vs %s) =====\n",x,y,
			FilenamePDB(PDB[x]),FilenamePDB(PDB[y]));
	     AR[Y]=ColRes[A][y]; nr[Y]=ncolX[A][y];
	     // scoreX=CalcDistDiff(AR,nr);
	     scoreX=this->CalcDistDiff(AR,nr,NumUsd[A][x],NotUsd[A]);
#if 0
if(HG==0){ HG=Histogram("MSA 1: avg. distance difference",0,5,0.25); }
if(x==1) fprintf(stderr,"%d(%s).%d(%s): nres = %d (%d vs %d); score=%.3lf\n", x,
		FilenamePDB(PDB[x]),y, FilenamePDB(PDB[y]),
		nr[Y]-CardSet(NotUsd[A]), nr[X],nr[Y],scoreX);
#endif
	     sum += scoreX; Num++;
	     if(efp) fprintf(efp,"MSA 1: score[%d][%d] = %.3lf\n",x,y,scoreX);
	     if(HG && y > x) IncdHist(scoreX,HG);
	   }
	   score[X][x] = sum/(double)Num;
	   sumX += sum; NumX += Num;
	}
	if(HG){ PutHist(stderr,60,HG); NilHist(HG); } HG=0; 
	if(efp) HG=Histogram("MSA 2: avg. distance difference",0,5,0.25);
	for(x=1; x <= NP; x++){		// MSA #1
	   AR[X]=ColRes[B][x]; nr[X]=ncolX[B][x]; 
	   sum=0.0; Num=0;
	   for(y=1; y <= NP; y++){	// MSA #2.
	     if(x == y) continue;
	     if(efp) fprintf(efp,"===== %d vs %d (%s vs %s) =====\n",x,y,
			FilenamePDB(PDB[x]),FilenamePDB(PDB[y]));
	     AR[Y]=ColRes[B][y]; nr[Y]=ncolX[B][y];
	     // scoreY=CalcDistDiff(AR,nr);
	     scoreY=CalcDistDiff(AR,nr,NumUsd[B][x],NotUsd[B]);
#if 0
if(HG==0) HG=Histogram("MSA 2: avg. distance difference",0,5,0.25);
if(x==1) fprintf(stderr,"%d(%s).%d(%s): nres = %d (%d vs %d); score=%.3lf\n", x,
		FilenamePDB(PDB[x]),y, FilenamePDB(PDB[y]),
		nr[Y]-CardSet(NotUsd[B]), nr[X],nr[Y],scoreY);
#endif
	     sum += scoreY; Num++;
	     if(efp) fprintf(efp,"MSA 1: score[%d][%d] = %.3lf\n",x,y,scoreX);
	     if(HG && y > x) IncdHist(scoreY,HG);
	   }
	   score[Y][x] = sum/(double)Num;
	   sumY += sum; NumY += Num;
	}
	if(HG){ PutHist(stderr,60,HG); NilHist(HG); }
	// fprintf(ofp,"scores: %.3lf vs %.3lf\n", sumX/(double)NumX,sumY/(double)NumY);
	score[0][0]=sumX/(double)NumX;
	score[1][0]=sumY/(double)NumY;
	return score;
}

void    dds_typ::Free()
{
	Int4	x,j;
	for(x=1; x <= NP; x++){ 
	   for(j=1; j <= nresX[x]; j++) if(AllRes[x][j]) NilRes(AllRes[x][j]); 
	   free(AllRes[x]);
	   free(ColRes[A][x]); free(ColRes[B][x]);
	} free(AllRes); free(nresX);
	free(ColRes[A]); free(ColRes[B]);
	free(ncolX[A]); free(ncolX[B]);
	NilRes(deleted); NilAtom(atm0[1]); free(atm0);
	NilSet(NotUsd[A]); NilSet(NotUsd[B]);
}


