
/******************************************************************************************
    Copyright (C) 1997-2019 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in the
    Software without restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the
    following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "cps_typ.h"
#include "compass_usage.h"

set_typ	cps_typ::FindPurgePDBIDs(Int4 percentID,cma_typ cma)
// get Purged KeyPDB list automatically.
{
	BooLean *skip;
	Int4 sq,N=NumSeqsCMSA(cma);
	cma_typ pdbcma=0;
	ss_type data = TrueDataCMSA(cma);
	a_type AB=AlphabetCMSA(cma);
	FILE *tfp=0;
#if 0
	NEW(skip,N+3,BooLean);
	for(Int4 i=1;i <= N; i++) {
	   e_type  sE;
	   sE=SeqSetE(i,data);
	   if(!PdbSeq(sE)) skip[i]=TRUE;
	} tfp=tmpfile();
	PutSelectCMSA(tfp,skip,cma); free(skip);
	rewind(tfp); pdbcma=ReadCMSA(tfp,AB); fclose(tfp);

	Int4	J,nJ=NumSeqsCMSA(pdbcma);
        set_typ InSet=MakeSet(nJ+4); FillSet(InSet);
        set_typ Set=RtnFastRepSetCMSA(stderr, percentID,InSet,pdbcma);

	e_type sqJ=0;
	char	**Rtn=0; NEWP(Rtn,nJ+3,char);
	for(Npdb=0,J=1; J <= nJ; J++){
		if(!MemberSet(J,Set)) continue;
		sqJ=TrueSeqCMSA(J,pdbcma);
		StrSeqID(str,10,sqJ); str[6]=0;
		Npdb++; Rtn[Npdb]=AllocString(str);
		fprintf(stderr,"%d. %s\n",Npdb,str);
	} TotalNilCMSA(pdbcma); NilSet(InSet); NilSet(Set);
	return Rtn;
#else
        set_typ InSet=MakeSet(N+4); ClearSet(InSet);
	for(Int4 i=1;i <= N; i++) {
	   e_type  sE=SeqSetE(i,data);
	   if(PdbSeq(sE)) AddSet(i,InSet);
	} 
        set_typ Set=RtnFastRepSetCMSA(stderr, percentID,InSet,cma);
	Int4	Npdb=0;
	for(Int4 i=1;i <= N; i++) {
	   if(MemberSet(i,Set)){
		e_type sqI=TrueSeqCMSA(i,cma);
		StrSeqID(str,10,sqI); str[6]=0; Npdb++; 
		fprintf(stderr,"%d. %s\n",Npdb,str);
	   }
	} NilSet(InSet); 
	return Set;
#endif
}

Int4	cps_typ::ReadPDB_ID()
{
	e_type	sqI,sqJ;
	Int4	ii,i,n,M,N,I,J,nI=NumSeqsCMSA(cmaA),nJ=NumSeqsCMSA(cmaB);
	Int4	Error=0;
	set_typ SetKey=this->FindPurgePDBIDs(PercentID,cmaA);
	set_typ	SetPDB=MakeSet(nI+5); ClearSet(SetPDB);
	if(nI == nJ){
	  NEWP(pdb_id,nI+5, char);
	  for(N=0,I=1; I <= nI; I++){
		sqI=TrueSeqCMSA(I,cmaA);
		sqJ=TrueSeqCMSA(I,cmaB);
		StrSeqID(str,10,sqI);
		if(strlen(str) != 6) continue;
                if(str[4] != '_') continue;
                if(!isdigit(str[0])) continue;
		if(!MemberSet(I,SetKey)) continue; 
		N++; pdb_id[N]=AllocString(str);
		AddSet(I,SetPDB);
		StrSeqID(str,10,sqJ);
		if(strncmp(pdb_id[N],str,6) != 0){
		   Error=2;
		   fprintf(stderr,"%d.%d: %s != %s\n",N,I,pdb_id[N],str);
		   break;
		} // else fprintf(stderr,"%d.%d: %s == %s\n",N,I,pdb_id[N],str);
		if(efp) fprintf(efp,"%d.%d: %s\n",N,I,pdb_id[N]);
	  }
	} else Error=1;
	if(SetKey) NilSet(SetKey);
	NumAlignedStrct=N;
	if(Error){
	   fprintf(stderr,"Fatal: lack of correspondence between MSAs\n");
	   if(Error==1){
	      fprintf(stderr,"MSAs differ in size (%d vs %d)\n",nI,nJ); 
	      fprintf(stderr,"Try running: '%s %s %s' to create 'msa1' & 'msa2'\n",
			argv[0],argv[1],argv[2]);
	   } else {
	      fprintf(stderr,"PDB seq '%s' in MSA1 is absent from MSA2\n",pdb_id[N]); 
	      fprintf(stderr,"MSA2 seq may be mislabeled\n",pdb_id[N]); 
	      fprintf(stderr,
		  "Otherwise try running: '%s %s %s' to create 'msa1' & 'msa2'\n",
			argv[0],argv[1],argv[2]);
	   }
	   print_error("  exiting...");
	}
	fprintf(outfp,"COMPASS: Comparison of multiple protein alignments");
	fprintf(outfp," by statistical score\n\n");
	fprintf(outfp,
          "     MSA 1(%s): %d columns; MSA 2(%s): %d columns; %d sequences.\n\n",
                        name1,LenA,name2,LenB,NumSeqsCMSA(cmaA));
	double	**RtnDel=0;
	N=this->RmPoorRefStrct(SetPDB,pdb_id,RtnDel);
	n=PercentIdentity(SetPDB,pdb_id,RtnDel);
	free(RtnDel[1]); free(RtnDel[2]); free(RtnDel);
	fflush(outfp);
#if 1	// remove poor pdb files here;
	assert(n==N);
	char **tmp_id; NEWP(tmp_id,N+5,char);
	double aln_cut=0.90;
	for(M=0,i=1; i <= N; i++){
           if(PCALN[1][i] < aln_cut || PCALN[2][i] < aln_cut){
	     if(PCALN[1][i] <  PCALN[2][i]){
                if(logfp) fprintf(logfp,
			"%d.%s in MSA 1 only %.1lf%c aligned; rejected\n",
                        i,pdb_id[i],PCALN[1][i],'%');
	     } else {
                if(logfp) fprintf(logfp,
			"%d.%s in MSA 2 only %.1lf%c aligned; rejected\n",
                        i,pdb_id[i],PCALN[2][i],'%');
	     } free(pdb_id[i]); 
	   } else { M++; tmp_id[M]=pdb_id[i]; }
	} free(pdb_id); pdb_id=tmp_id; N=M;
#else
	assert(n==N);
#endif
	NilSet(SetPDB); 
	NEW(SEQ,N+3,e_type); NEW(PDB,N+3,pdb_typ); NEW(CHN,N + 3, char);
	NEWP(ResX,N+3,char); NEWP(ResY,N+3,char);
	NEW(AdjstX,N+3,Int4); NEW(AdjstY,N+3,Int4); NEW(UsdSet,N+3,set_typ);
	NEWP(DCH[1],N+3,dch_typ); NEWP(DCH[2],N+3,dch_typ);
	NEWP(PDBID,N+3,char);
	NumPDB_ID=N;
	return N;
}

void    PutCMHtoCysAtom(FILE *fptr, atm_typ A)
{
        Int4    i;
        fprintf(fptr,"ATOM  ");
        fprintf(fptr,"%5d ",A->ser_num); /** atom name **/
        for(i=0; i < 4; i++) fprintf(fptr,"%c",A->name[i]);
          /** alternate location **/
        fprintf(fptr,"%cCYS",A->alt); /** residue name **/
        fprintf(fptr," %c",A->chain);   // residue number
        fprintf(fptr,"%4d",A->res);     // insert designator
        fprintf(fptr,"%c   ",A->insrt);
        fprintf(fptr,"%8.3f",A->X);     // X coordinate
        fprintf(fptr,"%8.3f",A->Y);     // Y coordinate
        fprintf(fptr,"%8.3f",A->Z);     // Z coordinate
        fprintf(fptr,"%6.2f",A->occ);   // occupancy
        if(A->temp > 0) fprintf(fptr,"%6.2f        ",A->temp);  // temperature
        fprintf(fptr,"\n");
}

// move this to pdb.cc after testing...
pdb_typ cps_typ::ChangeCMH2Cys(pdb_typ P,Int4 C)
// remove ' CM ' and 'HG  ' atoms...
// and all HETATM CMH to ATOM CYS
{
	Int4    a,n=0;
	pdb_typ	pdb=0;

	assert(C > 0 & C <=P->nchains);
        for(a=1; a<= P->natoms[C]; a++){
	   atm_typ atm=P->atom[C][a];
	   if(strncmp(AtomResName(atm),"CMH",3) == 0){
		if(strcmp(AtomName(atm)," CM ")==0 
			|| strcmp(AtomName(atm),"HG  ")==0) n++;
#if 0
		   PutCMHtoCysAtom(stdout,atm);
		   PutAtom(stdout, atm); n++; 
		} else {
		   PutCMHtoCysAtom(stdout,atm);
		   PutAtom(stdout, atm); n++; 
		}
#endif
	   }
	}
	if(n == 0) return 0;
	FILE *tfp=tmpfile();
        for(a=1; a<= P->natoms[C]; a++){
	   atm_typ atm=P->atom[C][a];
	   if(strncmp(AtomResName(atm),"CMH",3) == 0){
		if(strcmp(AtomName(atm)," CM ")==0 
			|| strcmp(AtomName(atm),"HG  ")==0){
			; // skip
			// PutAtom(stdout, atm); n++; break; 
		} else PutCMHtoCysAtom(tfp,atm);
	   } else PutAtom(tfp, atm);
	} rewind(tfp); pdb=MakePDB(tfp); fclose(tfp);
// if(strcmp(AtomName(atm)," CM "){ }
// if(strcmp(AtomName(atm),"HG  "){ }
        // fprintf(stderr,"%d CMH atoms changed to Cys.\n",n);
        return pdb;
}

pdb_typ cps_typ::SafeOpenPDB(char *pdbID,Int4 &min, Int4 &max, Int4 &chnK)
{
	Int4	j;
	FILE	*fp=0;
	// char	Str[200];

	for(j=0; j <=3; j++) pdbID[j]=tolower(pdbID[j]); pdbID[j]=0;
	sprintf(Str,"%s/%s_H.pdb",argv[3],pdbID);
	if((fp = fopen(Str,"r")) == NULL) {
	      sprintf(Str,"%s/%s.pdb",argv[3],pdbID);
	      if((fp = fopen(Str,"r")) == NULL) {
		if(logfp) fprintf(logfp,"%s does not exist; skipping\n",Str); 
		NumMissingStrct++;
		return 0;
	      } else fclose(fp);
	} else fclose(fp);
	pdb_typ pdb=MakeSafePDB(Str);  // returns 0 for pdb with residue IDs 
	if(pdb==0){			   // out of order (e.g., ..,R402,E401,..)
	      if(logfp) fprintf(logfp,"%s problematic; skipped\n",Str);
	      NumBadStrct++; return 0;
	} 
	char chn=pdbID[5];
	chnK=GetChainNumberPDB(pdb,chn);
	min=MinResPDB(chnK,pdb); max=MaxResPDB(chnK,pdb);
	if(max < 10){
		if(logfp) fprintf(logfp,"%s too short; skipping\n",Str); 
		NilPDB(pdb); 
		NumBadStrct++; return 0;
	}
#if 1
	{	// check for only alpha carbons...
	   Int4	n=CountAtomsPDB(min,max,chn,pdb);
	   Int4	j=CountResPDB(min,max,chn,pdb);
	   double d=(double)n/(double)j;
	   if(d < 3.0){
		// fprintf(logfp,"%s: %d atoms in %d res (%.3lf)\n",pdbID,n,j,d);
		if(logfp) fprintf(logfp,
			"%s: %d atoms for %d residues (%.3lf); skipped.\n",
			pdbID,n,j,d);
	   	NumBadStrct++; return 0;
	   }
	}
#endif
#if 0	// doesn't seem to matter if removed or not...
	pdb_typ pdb2=this->ChangeCMH2Cys(pdb,chnK);
	if(pdb2){
		NilPDB(pdb); pdb=pdb2; 
	  	fprintf(logfp,"%s: CMH residues changed to Cys.\n",pdbID);
	}
#endif
	if((j=ChangeMSE2MET_PDB(pdb)) > 0){
	  if(logfp) fprintf(logfp,"%s: %d MSE atoms changed to Met.\n",pdbID,j);
	}
	if(min <= 0){
		if(logfp) fprintf(logfp,
		  "Removing residues <= 0 from chain '%c' in '%s'; res=%d..%d\n",
                        chn,pdbID,min,max);
	     	min=MinPosResPDB(chnK,pdb);
		fp=tmpfile(); PutSubChainPDB(fp,min,max,chn,pdb); 
		rewind(fp); NilPDB(pdb); pdb=MakePDB(fp); fclose(fp);
	} ReNamePDB(pdbID,pdb);
	return pdb;
}

Int4	cps_typ::RmPoorRefStrct(set_typ SetPDB,char **pdb_id,double **&RtnDEL)
{
	Int4	numD,rJ,i,N,J,nA=NumSeqsCMSA(cmaA),nB=NumSeqsCMSA(cmaB);
	Int4	nS,nR,nR1,nR2,lenA=LengthCMSA(1,cmaA),lenB=LengthCMSA(1,cmaB);
	assert(nA==nB);
	double	del,Del; // CutDel=75.0;
	N=CardSet(SetPDB);
	char	**Reject; NEWP(Reject,N+3,char);
	char	**Save; NEWP(Save,N+3,char);
	double	*DEL[3]; NEW(DEL[1],N+3,double); NEW(DEL[2],N+3,double);
	double	**rtnDel;  NEWP(rtnDel,5, double);
	NEW(rtnDel[1],N+3,double); NEW(rtnDel[2],N+3,double);
	h_type	HG=Histogram("permutation scores",0,100,2.0);

	for(nS=nR=nR1=nR2=N=0,J=1; J <= nA; J++){
	      if(!MemberSet(J,SetPDB)) continue; else N++;
	      for(numD=0,i=1; i <= lenA; i++){
		rJ=ResidueCMSA(1,J,i,cmaA);
		if(rJ == 0) numD++; 
	      } del = 100.0*(double)numD/(double)lenA;
	      if(del >= CutDel){
	        if(logfp) fprintf(logfp,"%s in msa1 has %.1lf%c deletions; rejected\n",
			pdb_id[N],del,'%'); nR1++;
	      } 

	      for(numD=0,i=1; i <= lenB; i++){
		rJ=ResidueCMSA(1,J,i,cmaB);
		if(rJ == 0) numD++;
	      } Del=100.0*(double)numD/(double)lenB;
	      if(Del >= CutDel){
	        if(logfp) fprintf(logfp,"%s in msa2 has %.1lf%c deletions; rejected\n",
			pdb_id[N],Del,'%'); nR2++;
	      }
	      if(del >= CutDel || Del >= CutDel){
		 DeleteSet(J,SetPDB); nR++;
		 DEL[1][nR]=del; DEL[2][nR]=Del;
		 if(Del > del) IncdHist(Del,HG); else IncdHist(del,HG);
		 Reject[nR]=pdb_id[N];
	      } else { nS++; Save[nS]=pdb_id[N]; 
		if(Del > del) IncdHist(Del,HG); else IncdHist(del,HG);
		rtnDel[1][nS]=del; rtnDel[2][nS]=Del; 
	      }
	}
	double	mean = MeanHist(HG);
	double	maxD=MaxValHist(HG);
	// PutHist(stderr,60,HG); 
	NilHist(HG);
#if 0
	if(nR1 > 0) fprintf(outfp,
		"\tMSA 1: %d poorly reference structures rejected\n",nR1);
	if(nR2 > 0) fprintf(outfp,
		"\tMSA 2: %d poorly reference structures rejected\n",nR2);
#endif
	if(nR > 0){
	   fprintf(outfp,
	   "===== %d msa1 & %d msa2 poorly aligned reference structures rejected: =====\n",
		nR1,nR2);
	   for(i=1; i <= nR; i++){
	        if(DEL[1][i] >= CutDel || DEL[2][i] >= CutDel){
		  fprintf(outfp,
			"%d.\t%s: %.1lf%c deletions in msa1 & %.1lf%c in msa2\n",
			   i,Reject[i],DEL[1][i],'%',DEL[2][i],'%'); 
		  free(Reject[i]);
		}
	   }
	} fprintf(outfp,"\n");
	for(i=1; i <= nS; i++) pdb_id[i]=Save[i];
	for( ; i <= N; i++) pdb_id[i]=0;
	free(Reject); free(Save); free(DEL[1]); free(DEL[2]);
	NumRefStrctReject[0]=nR; NumRefStrctReject[1]=nR1; NumRefStrctReject[2]=nR2;
	RtnDEL=rtnDel;
	return CardSet(SetPDB);
}

Int4	cps_typ::PercentIdentity(set_typ SetPDB,char **pdb_id,double **RD)
{
	Int4	rI,rJ,i,j,N,M,I,J,nA=NumSeqsCMSA(cmaA),nB=NumSeqsCMSA(cmaB);
	Int4	sumM,sumMM,numM,numMM,SumM,SumMM,NumM,NumMM;
	Int4	lenA=LengthCMSA(1,cmaA),lenB=LengthCMSA(1,cmaB);
	assert(nA==nB);
	double	Dd,sum_dd=0.0,dd,MaxID,maxID;
	// fprintf(stderr,"CardSet=%d\n",CardSet(SetPDB));
	fprintf(outfp,
	   "========== Structural alignment percent identities ==========\n\n");
#if 0
	fprintf(outfp,"\t\tMSA1\t---\t---\t---");
	fprintf(outfp,"\tMSA2\t---\t---\t---\n");
	fprintf(outfp,"n\tpdbid\tMatch\tmisMat\tavg%cID\tmax%cID",'%','%');
	fprintf(outfp,"\tMatch\tmisMat\tavg%cID\tmax%cID\n",'%','%');
#else
	fprintf(outfp,"\t\tMSA1\t---\t---\tMSA2\t---\t---\n");
	fprintf(outfp,"n\tpdbid\tavg%cID\tmax%cID\t%calgn",'%','%','%');
	fprintf(outfp,"\tavg%cID\tmax%cID\t%calgn\n",'%','%','%');
#endif
	Int4 NumOver50PC=0; NEW(MaxPCID,CardSet(SetPDB)+3,double);
	NEW(PCALN[1],CardSet(SetPDB)+3,double);
	NEW(PCALN[2],CardSet(SetPDB)+3,double);
	for(M=N=0,I=1; I <= nA; I++){
	   if(!MemberSet(I,SetPDB)) continue; else N++;
	   sumM=sumMM=M=0; SumM=SumMM=0;
	   for(MaxID=maxID=0.0,M=0,J=1; J <= nA; J++){
	      if(J==I) continue;
	      if(!MemberSet(J,SetPDB)) continue; else M++;

	      for(numM=numMM=0,i=1; i <= lenA; i++){
		rI=ResidueCMSA(1,I,i,cmaA);
		rJ=ResidueCMSA(1,J,i,cmaA);
		if(rI == 0 && rJ == 0) continue; 
		else if(rI == rJ) numM++; else numMM++;
	      } sumM+=numM; sumMM+=numMM;
	      dd=(double)numM/(double)(numM+numMM);
	      if(maxID < dd) maxID=dd;

	      for(NumM=NumMM=0,i=1; i <= lenB; i++){
		rI=ResidueCMSA(1,I,i,cmaB);
		rJ=ResidueCMSA(1,J,i,cmaB);
		if(rI == 0 && rJ == 0) continue; 
		else if(rI == rJ) NumM++; else NumMM++;
	      } SumM+=NumM; SumMM+=NumMM;
	      Dd=(double)NumM/(double)(NumM+NumMM);
	      if(MaxID < Dd) MaxID=Dd;

	   }
	   if(maxID >= 0.5 || MaxID >= 0.5)
	   {
		MaxPCID[N]= MAXIMUM(double,maxID,MaxID);
		NumOver50PC++;
	   }
	   dd=(double)sumM/(double)(sumM+sumMM);
	   sum_dd += dd;
	   Dd=(double)SumM/(double)(SumM+SumMM);
	   sum_dd += Dd;
	   for(NumM=0,i=1; i <= lenA; i++) if(ResidueCMSA(1,I,i,cmaA) > 0) NumM++; 
	   double pc1=100.0*(double)NumM/(double)lenA;
	   for(NumMM=0,i=1; i <= lenB; i++) if(ResidueCMSA(1,I,i,cmaB) > 0) NumMM++; 
	   double pc2=100.0*(double)NumMM/(double)lenB;
	   PCALN[1][N]=pc1; PCALN[2][N]=pc2;
	   fprintf(outfp,"%d\t%s\t%.1lf\t%.1lf\t%.0lf\t%.1lf\t%.1lf\t%.0lf",
		N,pdb_id[N],100.0*dd,100.0*maxID,pc1,100.0*Dd,100.0*MaxID,pc2);
	   
#if 0
	   fprintf(outfp,"\n");
#else
	   fprintf(outfp,"\t%.1lf%c\t%.1lf%c\n",RD[1][N],'%',RD[2][N],'%');
#endif
	} fprintf(outfp,"   Overall %cID=%.1lf\n\n",'%',
					100.0*sum_dd/(2.0*(double)N));
	if(NumOver50PC > 0){
	   if(logfp) fprintf(logfp,
	     "WARNING: %d out of %d reference structures share > 50%c sequence identity.\n",
		NumOver50PC,N,'%');
	   fprintf(outfp,
	     "WARNING: %d out of %d reference structures share > 50%c sequence identity.\n\n",
		NumOver50PC,N,'%');
#if 0	// for debugging...
	   for(i=1;i <= N; i++){
		if(MaxPCID[i] > 0.0){
		   fprintf(stderr,"%d.%s: %.1lf%c\n",i,pdb_id[i],100.*MaxPCID[i],'%');
		   fprintf(logfp,"%d.%s: %.1lf%c\n",i,pdb_id[i],100.*MaxPCID[i],'%');
		}
	   } fprintf(logfp,"\n");
#endif
	}
	return N;
}

