/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "gpsi_typ.h"
#include "alphabet.h"
#include "residues.h"
#include "smatrix.h"
#include "cmsa.h"
#include "gpsi2cma.h"
#include "time.h"

#define USAGE "usage: gblastpgp fafile dbsfile [options]\n\
      options:\n\
        -C<file>  write checkpoint file\n\
        -c<file>  write sequences detected to file\n\
        -D        Don't SEG query sequence\n\
        -d=<file> delete gapped HSPs within sequences scoring significant hits (append)\n\
        -D=<file> delete gapped HSPs within sequences scoring significant hits (deleted only)\n\
        -d        Show distributions of HSP scores and E-values\n\
        -e<real>  E-value cutoff for showing scores and alignments\n\
        -F        Start with fake cma search for matrix\n\
        -f<real>  write to file those sequences with no indels & <real> identity to query\n\
	-G        force a global alignment of output cma file\n\
        -h<real>  E-value cutoff for inclusion in profile\n\
        -H<int>   heapsize (default 500000)\n\
	-i        Merge sequence hits before saving file (used with -I<int>:<int>)\n\
	-I<int>:<int> - save sequence hits with flanking regions of given lengths\n\
        -J        Perform jackknife test on cmsa file\n\
        -j<int>   maxrounds (default 1)\n\
        -L        do low complexity masking of database sequences\n\
        -M        print sequences in database missed by the search\n\
        -m<int>   print option m=0..6 (default 4)\n\
        -minlen=<int>   minimum alignment length for '-f' option (default 50)\n\
	-NoIndels Disallow indels (i.e., search for identical seqs)\n\
        -O<file>  Output a cma file of the alignment\n\
        -P<int><str><ing> Kannan's pattern option (E.g., -P10'{WFYH}P'80)\n\
        -Q        Add query to database file\n\
        -R<file>  Read checkpoint file\n\
        -R        Open file (dbsfile.cma), create a dbs from fullseqs, \n\
                   create a dbsfile.new.cma with hits from search removed\n\
        -r        print rasmol script\n\
        -S<int>   resample alignments <int> times at each iteration (default 0)\n\
        -s        perform second search of sequences extended on first search\n\
        -T<int>   blast word hit threshold (default 11)\n\
        -T	  Output alignment in Tabular form.(creates a *.doc file)\n\
        -t<real>  trim alignments at ends where marg. prob. is > <real>\n\
        -v        verbose output\n\
        -w<int>   alignment width (default 60)\n\
        -X<int>   X dropoff for gapxdrop\n\
        -z        see posMatrix\n\
        -x        dummy\n\n"

int	main(int argc, char *argv[])
{
	Int4    T=11,gap_open=11,gap_extend=1;
	time_t	time1=time(NULL);
	Int4	left_flank=0,right_flank=0;
        double	x_parameter=25.0;
	Int4	maxrounds=1,width=60;
	UInt4	hpsz=500000,printopt=1,sampling_size=0;
	double	Ethresh=0.001,Ecutoff=0.001;
	BooLean	see_smatrix=FALSE,verbose=FALSE,rasmol=FALSE;
	BooLean	second_srch=FALSE,seg_seq=TRUE,success=FALSE,putsegs=FALSE;
	BooLean	Jackknife=FALSE,merge=FALSE;
	BooLean	DoGlobalAlign=FALSE;
	char	mask_dbs=' ',*checkin=0,*checkout=0;
	FILE	*seqfile=NULL,*missedfile=0;
	char	*seqfilename=0;
	char	*cmafile=0;
	BooLean	mod_cmafile=FALSE;
	BooLean	fake_cma_srch=FALSE,show_dist=FALSE,put_table=FALSE;
	double	misalncut=2.0;
	char	*DeleteFile=0;
	BooLean	AppendDelete=FALSE;
#if 1	// For Kannan
	char    *residue_str=0;
	Int4	Nmatch=0,Percent=0;
	sst_typ	*Residues=0;
	BooLean	*PatPos=0,NoIndels=FALSE;
	double PerCut=0.0,fractionIDs=2.0;
	Int4	NumPatFound=0,NumPatSucceed=0;
	Int4	minlen=50;
#endif
	BooLean add_query_to_data=FALSE;

	if(argc < 3) print_error(USAGE);
        for(Int4 arg = 3; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(USAGE);
           switch(argv[arg][1]) {
             case 'C': checkout = AllocString(argv[arg]+2); break;
             case 'c': seqfilename=argv[arg]+2; break;
			// seqfile = open_file(argv[arg]+2,"","w"); break;
	     case 'd':
             case 'D':  if(argv[arg][2] == '='){
				if(argv[arg][1]=='d') AppendDelete=TRUE;
             			DeleteFile=AllocString(argv[arg]+3); 
				//fprintf(stderr,"DEBUG: 'D' --> %s\n",DeleteFile);
				if(DeleteFile[0]==0) print_error(USAGE); 
			} else {
				if(argv[arg][1]=='d') show_dist=TRUE;
				else seg_seq=FALSE; 
			} break;
             case 'e': Ecutoff=RealOption(argv[arg],'e',0.0,10000,USAGE); break;
	     case 'G': DoGlobalAlign=TRUE; break;
             case 'F': fake_cma_srch=TRUE; break;
             case 'f': fractionIDs=RealOption(argv[arg],'f',0.0,1.0,USAGE); break;
	     case 'H': hpsz=IntOption(argv[arg],'H',1,1000000,USAGE); break;
             case 'h': Ethresh=RealOption(argv[arg],'h',0.0,10000,USAGE); break;
	     case 'i': merge=TRUE; break;
             case 'I': putsegs=TRUE;
                  if(sscanf(argv[arg],"-I%d:%d",&left_flank,&right_flank) != 2)
                        print_error(USAGE); break;
	     case 'J': Jackknife=TRUE; break;
	     case 'j': maxrounds=IntOption(argv[arg],'j',0,1000,USAGE); break;
             case 'L': mask_dbs='x'; break;
             case 'M': missedfile = open_file(argv[1],".rsq","w"); break;
	     case 'm': {
		   if(sscanf(argv[arg],"-minlen=%ld",&minlen)==1){
			if(minlen < 1) print_error(USAGE);
		   } else printopt=IntOption(argv[arg],'m',0,6,USAGE); 
		} break;
	    case 'N': if(strcmp("-NoIndels",argv[arg])==0){ NoIndels=TRUE; }
                        else print_error(USAGE); break;
	     case 'O': cmafile=AllocString(argv[arg]+2); break;
#if 1	// For Kannan
             case 'P':
                   if(argv[arg][2] == 0) print_error(USAGE);
                   else {
			NEW(residue_str,strlen(argv[arg]+2),char);
                        if(sscanf(argv[arg],"-P%d%[a-zA-Z{}.]%d",
					&Nmatch,residue_str,&Percent) != 3){
			    fprintf(stderr,"%s\n",argv[arg]);
	   		    fprintf(stderr,"-P%d'%s'%d\n",Nmatch,residue_str,Percent);
				print_error(USAGE);
                        } 
			if(Percent < 1 || Percent > 100) print_error(USAGE);
			if(Nmatch < 1) print_error(USAGE);
                   } break;
#endif
             case 'Q': {
			add_query_to_data=TRUE;
                   } break;
             case 'R': {
		  if(argv[arg][2] == 0){ // use a cmafile for search
			mod_cmafile=TRUE; argv[arg][1]=' ';
		  } else {
			checkin = AllocString(argv[arg]+2); 
		  } break;
	       }
             case 'r': rasmol= TRUE; break;
             case 's': second_srch=TRUE; break;
             case 'S': sampling_size=IntOption(argv[arg],'S',0,100,USAGE); break;
	     case 'T': 
		if(argv[arg][2]==0) put_table=TRUE;
		else T=IntOption(argv[arg],'T',1,100,USAGE); 
			break;
	     case 't': misalncut=RealOption(argv[arg],'t',0.1,1.0,USAGE); break;
             case 'v': verbose=TRUE; break;
	     case 'w': width=IntOption(argv[arg],'w',5,50000,USAGE); break;
	     case 'X': x_parameter=IntOption(argv[arg],'X',1,1000,USAGE); break;
             case 'z': see_smatrix= TRUE; break;
             default: print_error(USAGE);
           }
        }

	// a_type		A=MkAlpha(AMINO_ACIDS,PROT_BLOSUM62);
	a_type	  A=MkAlpha(AMINO_ACIDS,GBLAST_BLOSUM62); // IMPORTANT!!
	e_type	  qE,queryE=0;
	ss_type	data;
	cma_typ	cma=0;
	char str[100];

	// TESTING FakeSearchCMSA( );
	if(fake_cma_srch){
		ss_type data2=MakeSeqSet(argv[2],200000,A);
		sprintf(str,"%s.cma",argv[1]);
		cma=ReadCMSA2(str,A);
		ss_type data1=TrueDataCMSA(cma);
		// qE=MkConsensusCMSA(cma);
		qE=CopySeq(SeqSetE(1,data1));
		// use correct qE that is first in same as before... 
		data=MergeSeqSets(data1, data2);
		NilSeqSet(data2);
	} else if(mod_cmafile){	// get database from cmafile
	    if(Jackknife) print_error("Can't combine mod_cmafile with Jackknife option");
	    qE=ReadSeqFA(argv[1], 0, A);
	    sprintf(str,"%s.cma",argv[1]);
	    cma=ReadCMSA2(str,A);
	    data=CopySeqSet(TrueDataCMSA(cma));
	    ReNumberSeqSet(data);
	} else {	// use normal settings...
	    qE=ReadSeqFA(argv[1], 0, A);
	    if(Jackknife) {
		sprintf(str,"%s.cma",argv[2]);
		cma=ReadCMSA2(str,A);
		data = TrueDataCMSA(cma);
	    } else data=MakeSeqSet(argv[2],200000,A);
	}
	
	Int4 LenPattern=0;
	Int4	max_pattern_length=100;
        if(!residue_str && add_query_to_data){
	   if(!IsInSeqSet(qE,data)){	// if query is not in sequence set...
		data=AddSeqSet(qE,data); // then add query sequence...
	   }
        }
#if 1   // ***************************** For Kannan... ********************************
	// ***************************** For Kannan... ********************************
	// ***************************** For Kannan... ********************************
	else if(residue_str){
	   if(!IsInSeqSet(qE,data)){	// if query is not in sequence set...
		data=AddSeqSet(qE,data); // then add query sequence...
	   }
	   queryE=CopySeq(qE);
	   if(seg_seq) ProcessSeqPSeg(17,2.2,2.5,100,queryE,A);
	   seg_seq=FALSE;
	   Int4 s,npat=0,p;
	   PerCut=0.01*(double)Percent;
	   fprintf(stderr,"-P%d%s%d %.3f\n",Nmatch,residue_str,Percent,PerCut);
	   // exit(1);
#if 1
	   Residues=StringToSmallSet(residue_str,max_pattern_length,USAGE,&LenPattern,A);
#else
	   char	state=0;
	   for(s=0; residue_str[s]; s++){
	     if(residue_str[s] == '{'){
		if(state == 0 || state == 'R' || state == 'r'){
		    state = '{';
		} else print_error(USAGE);
	     } else if(residue_str[s] == '}'){
		if(state == 'R'){		// end multiresidue position
		    state = '}'; npat++;
		} else print_error(USAGE);
             } else if(isalpha(residue_str[s])){
                residue_str[s]=toupper(residue_str[s]);
		if(state == 0){			// first == one residue position
			state = 'r'; npat++;
		} else if(state == '}'){	// new one residue position
			state = 'r'; npat++;	
		} else if(state == 'r'){	// at one residue position.
			state = 'r'; npat++;
		} else if(state == 'R'){	// within multiresidue position
		} else if(state == '{'){	// start multiresidue position
			state = 'R';
			// do nothing
		} else print_error(USAGE);
             }
	     std::cerr << state;
	   }
	   std::cerr << '\n';
	   NEW(Residues,npat+3,sst_typ);
	   for(state=0,p=s=0; residue_str[s]; s++){
	     if(residue_str[s] == '{'){
		if(state == 0 || state == 'R' || state == 'r'){	
		    state = '{'; Residues[p]=0;
		} else print_error(USAGE);
	     } else if(residue_str[s] == '}'){
		if(state == 'R'){		// end multiresidue position
		    state = '}'; p++;
		} else print_error(USAGE);
             } else if(isalpha(residue_str[s])){
                residue_str[s]=toupper(residue_str[s]);
		if(state == 0 || state == '}' || state == 'r'){	// one residue.
			Residues[p]=SsetLet(AlphaCode(residue_str[s],A));
			state = 'r'; p++;
		} else if(state == 'R' || state == '{'){  // multiresidue 
			state = 'R';
			sst_typ tmp_set=SsetLet(AlphaCode(residue_str[s],A));
                	Residues[p] = UnionSset(Residues[p],tmp_set);
			// do nothing
		} else print_error(USAGE);
             }
	     std::cerr << state;
	   }
	   std::cerr << '\n';
	   LenPattern=npat;
#endif
	   for(Int4 j=0; j < LenPattern; j++){
	    fprintf(stderr,"set %d: ",j);
	    PutSST(stderr,Residues[j],A); fprintf(stderr,"\n");
	   }
	   NEW(PatPos,LenSeq(qE)+3,BooLean);
	   // search for pattern in query...
	   BooLean success=FALSE;
	   for(s=1; s <= LenSeq(qE)-LenPattern+1; s++){
	      Int4 r=ResSeq(s,queryE);
	      if(MemSset(r,Residues[0])){
		Int4 s0;
	         for(s0=s+1,p=1; p < LenPattern; p++,s0++){
		    r=ResSeq(s0,queryE);
	            if(!MemSset(r,Residues[p])) break;
		 } if(p==LenPattern){
			PatPos[s] = TRUE; 
			fprintf(stderr,"Pattern found at position %d\n",s);
			success=TRUE;
			NumPatFound++;
		 }
	      }
	   }
	   if(!success) print_error("Pattern not in query sequence");
        }
#endif	// End Kannan section...
	// ***************************** End Kannan. **********************************
	// ***************************** End Kannan. **********************************
	// ***************************** End Kannan. **********************************

	Int4 HPSZ=hpsz;
        if(NoIndels){ hpsz=HPSZ*20; }
	gpsi_type gpsi(qE,data,Ethresh,Ecutoff,x_parameter,hpsz,maxrounds,checkin);
#if 1   // AFN: 2_17_2023: to find identical sequence matches.
        if(NoIndels){
	   gpsi.ResetIndelPenalties(5000, 5000); 
	}
        // fprintf(stderr,"NoIndels=%d\n",NoIndels);
#endif

	if(seg_seq) ProcessSeqPSeg(17,2.2,2.5,100,qE,A);
	Int4 iter=1;
        do { 
	   sap_typ sap=0;
	   if(fake_cma_srch){
	   	sap = gpsi.FakeSearchCMSA(T,cma); 
#if 0		// DEBUG...
		PutMultiGSeqAlign(stderr, sap, 60, A);
		exit(1);
#else
		// PutGSeqAlignList(stderr, sap, 60, A);
		// PutSitesCMSA(stderr,1,0, cma);
#endif
		TotalNilCMSA(cma); fake_cma_srch=FALSE;
	   } else {
		if(misalncut < 1.0) sap = gpsi.TrimmedSearch(T,misalncut,mask_dbs);
	   	else sap = gpsi.Search(T,mask_dbs);
	   }
	   if(sap) success=TRUE; else break;
	   if(verbose) gpsi.Test('M');
	   // if(verbose) gpsi.Test('A');
	   gpsi.ComputeMatrix(checkout);
#if 0	// For marginal probability...
	   qE;
	   gpsi.posMatrix();
	   gpsi.Lambda();
	   data; // use sequence number 1 for marginal probability alignment.
#endif
#if 1	// original sampling...
	   if(!fake_cma_srch || fake_cma_srch && iter > 1){
	    for(Int4 s=1; s <= sampling_size; s++){
	     if(gpsi.posMatrix){ // test sampling routines...
          	sap_typ sap = gpsi.SampleMatrix(checkout); // recomputes matrix...
	   	if(verbose) gpsi.Test('M');
             	// if(verbose) gpsi.ShowAlign(stdout,width,printopt);
	     }
	    }
	   }
#endif
	   if(second_srch && gpsi.NotConverged()){
		gpsi.Search(T,TRUE,mask_dbs);
		if(verbose) gpsi.Test('M'); gpsi.ComputeMatrix(checkout); 
	   }
	   if(see_smatrix && gpsi.posMatrix){ 
		FILE *ofp = open_file(argv[1],".psm","w");
		gpsi.SeeMatrix(ofp); 
		fclose(ofp);
	   }
	   iter++;
	} while(gpsi.NotConverged( ));
#if 0	// sampling at end...
	   for(Int4 s=1; s <= sampling_size; s++){
	     gpsi.ComputeMatrix(checkout);
	     if(gpsi.posMatrix){ // test sampling routines...
          	sap_typ sap = gpsi.SampleMatrix(checkout); // recomputes matrix...
	   	if(verbose) gpsi.Test('M');
             	// if(verbose) gpsi.ShowAlign(stdout,width,printopt);
	     }
	   }
#endif
	if(NoIndels){
	   sap_typ sap = gpsi.GetSAP();
	   sap_typ sap2 = SortBySeqIdentity(sap,HPSZ);
	   gpsi.ShowAlign(stdout,width,printopt,sap2);
	   return 0;
	} else gpsi.ShowAlign(stdout,width,printopt);
#if 1	// new deleteSeq ('D') option
	if(DeleteFile){
	  sap_typ sap = gpsi.GetSAP();
	  if(sap){
	   FILE *tmpfp= open_file(DeleteFile,"","w"); 
	   // gpsi.PutDeleteSeqs(tmfp); 
	   PutDeleteBestHSPs(tmpfp,sap,A); 
	   if(AppendDelete) gpsi.PutMissedSeq(tmpfp);	
	   fclose(tmpfp); 
	  }
	}
#endif
	if(show_dist){ gpsi.PutDistHits(stdout,60); }
	if(seqfilename){
	   seqfile = open_file(seqfilename,"","w"); 
	   if(fractionIDs <= 1.0) gpsi.PutSameSeqs(seqfile,minlen,fractionIDs);
	   else gpsi.PutSeqs(seqfile); 
	   fclose(seqfile); 
	}
	if(missedfile){ gpsi.PutMissedSeq(missedfile); fclose(missedfile); }
	if(putsegs){ 
		seqfile=open_file(argv[1],".rpts","w");
		if(merge) gpsi.PutMergedSubSeqs(seqfile,left_flank,right_flank);
		else gpsi.PutSubSeqs(seqfile,left_flank,right_flank); 
		fclose(seqfile); 
	}
	if(put_table){
		SeqAlignToTable(stdout,gpsi.GetSAP(),A);
	}

	if(residue_str){
#if 1   // For Kannan...
	  sap_typ sap = gpsi.GetSAP();
	  if(sap){
	   FILE	*fptr=tmpfile();
	   xSeqAlignToCMA(fptr, sap, left_flank, right_flank, A);
	   rewind(fptr); 
	   cma_typ cma = ReadCMSA(fptr,A); fclose(fptr);
	   BooLean success=FALSE,okay;
	   if(Nmatch <=  NumSeqsCMSA(cma)){
	        Int4 p,s0,s;
		double  **freq=ColResFreqsCMSA(1, cma);
		if(LengthCMSA(1,cma) != LenSeq(qE)){
			// WARNING: SEG will mess this up; implemented a fix w/ queryE.
			WriteCMSA("junk.cma", cma);
			PutSeq(stdout,qE,A);
			assert(LengthCMSA(1,cma) == LenSeq(qE));
		}
		double	total=0.0;
		for(s=1; s <= LenSeq(qE); s++){
		   if(PatPos[s]){
		     PutSeqRegion(stderr,s,LenPattern,qE,A); fprintf(stderr,"\n");
		     okay=TRUE;
	             for(s0=s,p=0; p < LenPattern; p++,s0++){
		       total=0.0;
                       for(Int4 r=0; r<= nAlpha(A); r++){
			   if(MemSset(r,Residues[p])){
				total+=freq[s0][r];
				fprintf(stderr,"freq[%d][%c]=%.4f\n",
					s0,AlphaChar(r,A),freq[s0][r]);
			   }
		       } fprintf(stderr,"total = %.4f\n\n",total);
		       if(total < PerCut) okay=FALSE;
		     }
		     if(okay){
			fprintf(stderr,"Found an alignment!\n");
			NumPatSucceed++;
		   	success=TRUE; // break; 
		     }
		   }
                }
	        if(success){
	   	  fptr=open_file(argv[1],".cma","w"); 
	   	  PutCMSA(fptr,cma);
	   	  fclose(fptr);
	        } else fprintf(stderr,"No patterns satisfying input parameters\n");
	   } else fprintf(stderr,"Too few sequences found\n");
	   free(cmafile); cmafile=0;
	   free(PatPos);
	   fptr=open_file(argv[1],".ptrn_rpt","w");
	   fprintf(fptr,"%d out of %d patterns conserved\n",NumPatSucceed,NumPatFound);
	   fclose(fptr);
	  } // end if(sap)...
#endif
	} else if(mod_cmafile){  // Redundant with Jackknife option!!!
		// BooLean *skip = gpsi.GetSeqHitArray(NumSeqsCMSA(cma));
		BooLean *skip = gpsi.GetHitList( );
		FILE *fp=open_file(argv[2],".cma","w"); 
                PutSelectCMSA(fp,skip,cma); fclose(fp);
		NilCMSA(cma); free(skip); 
	} else if(cmafile){ 
		sap_typ sap = gpsi.GetSAP();
#if 1		// forced global alignment...
		if(DoGlobalAlign) MakeGlobalSeqAlign(sap,A);
#endif
		FILE *fptr;
		fptr=open_file(cmafile,".cma","w"); 
		// tSeqAlignToCMA(fptr, sap, left_flank, right_flank, A);
		if(add_query_to_data) 
			QueryFirstSeqAlignToCMA(fptr,sap,left_flank,right_flank,A);
		else xSeqAlignToCMA(fptr, sap, left_flank, right_flank, A);
		// gpsi.PutSeqsAsCMA(stdout,left_flank,right_flank); 
		fclose(fptr); free(cmafile); cmafile=0;
	}
	if(rasmol) gpsi.PutRasMol(stdout); 
	if(Jackknife){ 
		BooLean	*skip; 
		skip = gpsi.GetHitList( );
		sprintf(str,"%s.jack",argv[2]);
		FILE *fp = open_file(str,".cma","w");
		PutSelectCMSA(fp,skip,cma); fclose(fp); NilCMSA(cma);
		free(skip);
	}
	if(queryE) NilSeq(queryE);
	NilSeq(qE); 
	NilSeqSet(data); NilAlpha(A);
	if(checkin) free(checkin); if(checkout) free(checkout);
	fprintf(stderr,"time = %d seconds\n",(Int4) difftime(time(NULL),time1));
	// fprintf(stderr,"time = %d seconds\n",difftime(time(NULL),time1));
	if(success) return 0;
	else return 1;
}

