/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include <iostream>
#include <math.h>
#include <time.h>
#include <cuda_runtime.h>
#include "residues.h"
#include "alphabet.h"
#include "blosum62.h"
#include "sequence.h"
#include "stdinc.h"
#include "cmsa.h"
#include "sset.h"

// Kernel function to add the elements of two arrays
__global__ void add_cuda(int n, double *x, double *y,
	const int nAB, int *q)
{
   int index = (blockIdx.x * blockDim.x) + threadIdx.x;
   int stride = blockDim.x*gridDim.x;
  for (int i = index; i < n; i+= stride){
      y[i] = log(x[i]) + log(y[i]);
      for(int j=1; j<=nAB; j++) 
      {
	y[i]+=q[j];
	// y[i]+=(double)qst[j];
      }
  }
}

int runCUDA(int N,sst_typ *qst,a_type AB)
{
  double  *x, *y;
  int	  i,*q;

fprintf(stderr,"N=%d\n",N);

  // Allocate Unified Memory – accessible from CPU or GPU
  cudaMallocManaged(&x, N*sizeof(double));
  cudaMallocManaged(&y, N*sizeof(double));
  cudaMallocManaged(&q, nAlpha(AB)*sizeof(int));

  // initialize x and y arrays on the host
  for (i = 0; i < N; i++) {
    x[i] = 1.0f;
    y[i] = 2.0f;
  }
  for (i=1; i <= nAlpha(AB); i++) { q[i]=1; }

  int blockSize = 256;
  int numBlocks = (int) ceil((double)(N + blockSize -1)/ (double) blockSize);

  // Run kernel on 1M elements on the GPU
  add_cuda<<<numBlocks, blockSize>>>(N, x, y,nAlpha(AB),q);
  // blks, thrds  (as multiples of 32)

  // Wait for GPU to finish before accessing on host
  cudaDeviceSynchronize();

  // Check for errors (all values should be 3.0f)
  double maxError = 0.0f;
  for (int i = 0; i < N; i++)
    maxError = fmax(maxError, fabs(y[i]-3.0f));
  std::cout << "Max error: " << maxError << std::endl;

  // Free memory
  cudaFree(x);
  cudaFree(y);
  cudaFree(q);
  
  return 0;
}

// function to add the elements of two arrays
void add_cpu(int n, double *x, double *y,a_type AB,int *q)
{
   for (int i=0; i < n; i++){
      // y[i] = x[i] + y[i];
      y[i] = log(x[i]) + log(y[i]);
      for(int j=1; j<=nAlpha(AB); j++) { y[i]+=q[j]; }
   }
}

int runCPU(int N,sst_typ *qst,a_type AB)
{
  double *x = new double [N];
  double *y = new double [N];
  int	 *q = new int [nAlpha(AB)];

fprintf(stderr,"N=%d\n",N);

  // initialize x and y arrays on the host
  for (int i = 0; i < N; i++) {
    x[i] = 1.0f;
    y[i] = 2.0f;
  }
  for(int i=1; i<=nAlpha(AB); i++) { q[i]=1; }

  // Run kernel on 1M elements on the CPU
  add_cpu(N, x, y,AB,q);

  // Check for errors (all values should be 3.0f)
  double maxError = 0.0f;
  for (int i = 0; i < N; i++)
    maxError = fmax(maxError, fabs(y[i]-3.0f));
  std::cout << "Max error: " << maxError << std::endl;

  // Free memory
  delete [] x;
  delete [] y;

//   cudaProfilerStop();
  return 0;
}

int main(Int4 argc,char *argv[])
{
  if(argc < 2) print_error("cuda_test <cmafile>");
  // int	N = 2000000000;
  char	str[100];
  int	r,N = 1 << 8;
  a_type AB = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62);
  sprintf(str,"%s.cma",argv[1]);
  cma_typ cma=ReadCMSA2(str,AB);
  e_type Sq=TrueSeqCMSA(1,cma);
  
  sst_typ *qst; NEW(qst,LenSeq(Sq) +3, sst_typ);
  for(r=1; r <= LenSeq(Sq); r++){
	unsigned char R = ResSeq(r,Sq);
	qst[r] = SsetLet(R);
  }

  int	time1 = time(NULL);
  clock_t t=clock();
  runCPU(N,qst,AB);
  fprintf(stderr,"\tCPU time: %d seconds (%0.2f minutes); ticks=%d\n",
	time(NULL)-time1,(float)(time(NULL)-time1)/60.0,clock()-t);
  time1 = time(NULL); t=clock();
  runCUDA(N,qst,AB);
  fprintf(stderr,"\tCUDA time: %d seconds (%0.2f minutes); ticks=%d\n",
	time(NULL)-time1,(float)(time(NULL)-time1)/60.0,clock()-t);
  TotalNilCMSA(cma); NilAlpha(AB);
}


