/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "tree2hpt.h"
#include "hpt_typ.h"

#define EditHpt_VERSION "\
  ----------------------------------------------------\n\
     EditHpt v0.1 (June 15, 2022)\n\
     www.igs.umaryland.edu/labs/neuwald/software/mapgaps/\n\
     Copyright 2022 The University of Maryland\n\
     Freely distributed under the MIT License\n\
  ----------------------------------------------------\n"

#define USAGE_START     "EditHpt: Edit a hyperpartition\n\
   USAGE: edit_hpt <mode> <prefix> <command> [options]\n\
        <mode>     The type of operation to be performed\n\
	<prefix>   The input hyperpartition file = <prefix>.hpt.\n\
	<command>  The operation to be performed.\n\
   mode:\n\
      'D': Delete all the nodes descended from the specified node in a tree\n\
	  syntax: <int> = the node (row & column) to be deleted\n\
      'd': Delete a row and/or column from the hpt\n\
	  syntax: <int1>:<int2> (e.g., 3:5\n\
	  semantics: Delete row <int1> and column <int2>\n\
	    if an <int> == 0; then nothing is deleted\n\
      'I': Insert a row and/or column into the hpt\n\
	  syntax: <name>=<int1>:<int2> (e.g., Ras=3:5\n\
	  semantics: Insert a row BEFORE row <int1> and \n\
            a column AFTER column <int2>\n\
	    if an <int> == 0; then nothing is inserted\n\
         options:\n\
           -A=<int>:<int> - Set default alpha hyperparameters a0:b0 (default: 90:5)\n\
			    a0 and b0 = matching and non-matching residue pseudocounts\n\
			    a0 > b0 --> patterns are more stringently conserved\n\
			    a0 < b0 --> patterns are less stringently conserved\n\
	   -Ri=<real>     - Set global prior probability of a sequence being assigned\n\
			    to the BPPS foreground partition (default: 0.1)\n\
	   -N=<int>       - Set default contrast setting for bpps rtf alignemtn file\n\
			    (<int> = the number of highlighted columns) (default: 25)\n\
      'L': \n\
      'S': if hpt corresponds to a tree then output in SmartArt format\n\
	      Open output as MSDOS format, select all, copy, and \n\
	      paste MS PowerPoint SmartArt Hierarchy as unformated text.\n\
      's': if hpt corresponds to a tree then output the subtree rooted at 'node'\n\
	  syntax: <int> = the node rooting the subtree\n\
      'T': if hpt corresponds to a tree then output it in newick format\n\
   Reference:\n\
     Neuwald, A.F. 2011. Surveying the manifold divergence of an entire protein \n\
	class for statistical clues to underlying biochemical mechanisms. \n\
	Stat Appl Genet Mol Biol 10(1): Article 36. (30 pages)\n\
     Neuwald, A.F. 2023? Identifying cross-conserved residue constraints in protein\n\
        seuqences. In preparation.\n\
        \n\n"

static void PrintError()
{
        // PrintLicenseStatement("bpps v1.0.8"); 
        fprintf(stderr,"%s",EditHpt_VERSION); print_error(USAGE_START);
}

int     main(int argc,char *argv[])
{
	Int4	a0=90,b0=5,N=25,arg,row=0,col=0,*P,node=0,x,i,j,rtn;
	double	Ri=0.1;
	char	name[100],argmt[200],mode;
	if(argc < 3) PrintError();
	if(argv[1][1] != 0) PrintError(); else mode=argv[1][0];
	hpt_typ *hpt = new hpt_typ(argv[2]);

   switch (mode){
    case 'I': {
	if(argc < 4) PrintError();
	rtn=sscanf(argv[3],"%[^=]=%d:%d",name,&row,&col);
	if(rtn != 3) PrintError();
	for(arg = 4; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(USAGE_START);
           switch(argv[arg][1]){
	      case 'A': if(sscanf(argv[arg],"-A%lf:%lf",&a0,&b0)==2){
                        if(a0 <= 0.0 || b0 <= 0.0) print_error(USAGE_START);
                       } else print_error(USAGE_START);
                break;
	      case 'N':
                if(sscanf(argv[arg],"-N=%d",&N)==1){
                        if(N< 1) print_error(USAGE_START);
                } else print_error(USAGE_START);
                break;
	      case 'R':
                if(sscanf(argv[arg],"-Ri=%lf",&Ri)==1){
                   if(Ri <= 0.0 || Ri >= 1.0){
                      fprintf(stderr,"Ri (%.2f) out of range\n",Ri);
                      print_error(USAGE_START);
                   }
                } else print_error(USAGE_START);
                break;
              default: print_error(USAGE_START); break;
	   }
	}
	if(col < 0 || col > hpt->NumBPPS()) print_error("input column invalid");
	if(row < 0 || row > hpt->NumSets()) print_error("input row invalid");
	// fprintf(stderr,"========== input Hyperpartition ========\n");
	// hpt->Put(stderr);
	fprintf(stderr,"========== inserting row %d & col %d ========\n",row,col);
	sprintf(argmt,"-A%d:%d -Ri=%.2lf -N=%d",a0,b0,Ri,N);
	hpt->PutInsertRowCol(stdout,row,col,name,argmt);	
    } break;
    case 'D': {
	if(argc < 4) PrintError();
	if(hpt->IsTree(P)){
	  row=atoi(argv[3]);
	  set_typ set=hpt->MkSubTreeSet(row);
          hpt_typ *xhpt=0;
          for(i=hpt->NumSets(); i > 0; i--){
                if(i != row && MemberSet(i,set)){
                   xhpt=hpt->Delete(i);  delete hpt; hpt=xhpt; xhpt=0;
                }
          } hpt->Put(stdout);
	}
    } break;
    case 'd': {
	if(argc < 4) PrintError();
	row=atoi(argv[3]);
	hpt->PutDelete(stdout,row);
    } break;
    case 'L': {
	if(argc < 4) PrintError();
	if(hpt->IsTree(P)){
	    node=atoi(argv[3]);
            hpt_typ *shpt=hpt->MkLineageTree(node,x);
            shpt->Put(stdout); delete shpt;
	} else print_error("Hpt is not a tree");
    } break;
    case 'T': {
	if(argc < 3) PrintError();
	if(hpt->IsTree(P)){
		hpt->PutAsTree(stdout);
	} else print_error("Hpt is not a tree");
    } break;
    case 's': {
	if(argc < 4) PrintError();
	node=atoi(argv[3]);
	if(hpt->IsTree(P)){
	   hpt->PutSubTree(stdout,node);
	} else print_error("Hpt is not a tree");
    } break;
    case 'S': {
	if(argc < 3) PrintError();
	if(hpt->IsTree(P)){
		hpt->PutAsSmartArt(stdout);
	} else print_error("Hpt is not a tree");
    } break;
    default: PrintError();
   }
#if 0	// flip a tree around...(partialy except for superfamily set.
	FILE *tfp=0;
	for(i=2,j=hpt->NumBPPS(); i < j; i++,j--){
	   tfp=tmpfile();
	   hpt->PutSwappedColumns(tfp,i,j);
	   rewind(tfp); delete hpt;
	   hpt=new hpt_typ(tfp); fclose(tfp);
	   // hpt->Put(stderr);
	}
#if 0
	tfp=tmpfile();
	hpt->PutSwappedRows(tfp,0,hpt->NumSets()-1);
	rewind(tfp); delete hpt;
	hpt=new hpt_typ(tfp); fclose(tfp);
#endif
	hpt->Put(stdout);
	exit(1);
#elif 0
	// hpt->DeleteRow(row); hpt->DeleteBPPS(col);
#else
#endif
#if 0
	hpt->Put(stdout);
	hpt->PutSwappedRows(stdout,2,3);
	hpt->PutSwappedColumns(stdout,2,3);
	hpt->PutInsertGrp(stdout,2);	
	for(row=0; row <= hpt->NumSets(); row++){
	  for(col=0; col <= hpt->NumBPPS(); col++){
	     // if(col == 1) continue;
	     fprintf(stdout,"========== inserting row %d & col %d ========\n",row,col);
	     hpt->Put(stdout);
	     hpt->PutInsertRowCol(stdout,row,col,name);	
	  }
	}
	hpt->PutWithout(stdout,2,3);	// col and/or row

	// from '-', to '+', row, col
	char	from='-',to='+';
	hpt->Change(from,to, row, col);
#elif 0
	  return tree2hpt(argc,argv); 
#endif
	delete hpt;
	return 0;
}

