/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "sma.h"
#include "cmsa.h"
#include "residues.h"
#include "gibbs.h"
#include "prtn_model.h"
#include "gss_typ.h"
#include "seqset.h"
#include "gsq_typ.h"
#include "histogram.h"
#include "random.h"
#include "gmb_typ.h"
#include "dheap.h"
#include "cma_gmb.h"
#include "hmm_typ.h"

#if 0	// moved to cma_gmb.cc
char    *AddInsertToOperationArray2(Int4 start, Int4 end, char *operation)
// ========== Add an insertion to an operational array. ===========
{

        char state,*new_operation=0;
        Int4 o,no,column;

        Int4 trace_length=strlen(operation);
        NEW(new_operation,trace_length+5,char);
        new_operation[0]='E'; no=1;
	for(no = o = 1; isalpha(operation[o]); o++){
		char c = operation[o];
		if(c == 'M' || c == 'D'){ break; }
		else { new_operation[no] = c; no++; }
	}
	if(!isalpha(operation[o])) print_error("AddInsertToOperationArray2() input error");
        // Int4 InsSize=end-start+1;
        // NEW(new_operation,trace_length+InsSize+5,char);
        for(column=1,state='E'; operation[o] != 'E'; o++){
          switch(operation[o]){
            case 'M':
            case 'm':
                if(column >= start && column <=end){
                           new_operation[no]='I';
                } else new_operation[no]=operation[o];
                no++; column++; break;
            case 'D':
            case 'd': // deletion in sequence relative to profile.
                if(column >= start && column <=end){
                           // do nothing in new_operation;
                } else { new_operation[no]=operation[o]; no++; }
                column++; break;
            case 'i': // insert is between profile blocks;
                new_operation[no]=operation[o]; no++;
                break;
            case 'I': // Insert ('-') within a profile block; delete from seq.
                new_operation[no]=operation[o]; no++;
                break;
            default:
            // fprintf(stderr,"operations = %s\n",operation);
            print_error("operation( ): input error"); break;
          }  state=operation[o];
        }
        new_operation[no]='E'; no++; new_operation[no]=0;
        // printf("new operation: %s\n",new_operation);
        return new_operation;
}
#endif

void	IronOutCMSA3(cma_typ cma)
// remove 'wrinkles' in a multiblock cmsa 
{
	Int4	sq,blk,X;
   	// a_type AB=AlphabetCMSA(cma);
	for(sq=1; sq <= NumSeqsCMSA(cma); sq++){
	        // ========== X. Get operational array for sequence. ===========
		gsq_typ *gsq=gsqCMSA(sq,cma);
		Int4	*sites=GetPosSitesCMSA(sq,cma);
		gsq_typ *gsq0=gsq->IronOut(nBlksCMSA(cma),sites,LengthsCMSA(cma));
		ReplaceCMSA(sq,gsq0,cma); // replace sequence s in CMSA & fmodel.
		for(blk=1; blk <= nBlksCMSA(cma); blk++) AddSiteCMSA(blk,sq,sites[blk],cma);
#if 0	// debug
		sites=GetPosSitesCMSA(sq,cma);
		gsq0->Put_cma_format(stderr,sq,nBlksCMSA(cma),sites,LengthsCMSA(cma),AB);
#endif
		free(sites); 
	} 
}

cma_typ	IronOutCMSA2(cma_typ cma)
// remove 'wrinkles' in a multiblock cmsa 
{
	Int4	blk,X;
   	a_type AB=AlphabetCMSA(cma);
	gss_typ *gss=gssCMSA(cma);
   	cma_typ rcma=EmptyCMSA(nBlksCMSA(cma),LengthsCMSA(cma),TrueDataCMSA(cma),
			gss->GapOpen(), gss->GapExtend(),PerNatsCMSA(cma),0,0);
	for(Int4 sq=1; sq <= NumSeqsCMSA(cma); sq++){
	        // ========== X. Get operational array for sequence. ===========
		gsq_typ *gsq=gsqCMSA(sq,cma);
		Int4	*sites=GetPosSitesCMSA(sq,cma);
		gsq_typ *gsq0=gsq->IronOut2(nBlksCMSA(cma),sites,LengthsCMSA(cma),X);
		ReplaceCMSA(sq,gsq0,rcma); // replace sequence s in CMSA & fmodel.
		for(blk=1; blk <= nBlksCMSA(rcma); blk++) AddSiteCMSA(blk,sq,sites[blk],rcma);
#if 0	// debug
		sites=GetPosSitesCMSA(sq,rcma);
		gsq0->Put_cma_format(stderr,sq,nBlksCMSA(rcma),sites,LengthsCMSA(rcma),AB);
#endif
		free(sites); 
	} return rcma;
}

BooLean	SampSlideColLeftCMSA(Int4 t, double *oldMap, double *newMap,
	cma_typ L, double temperature)
/*************************************************************************
  slide the end of one model onto the next model.

   --[**.*.*]--[*.:**.*...***]--
               |
               |		This operation allows escape from some 
               V                 nasty traps in alignment space.
   --[**.*.*.*]---[**.*...***]--

  returns TRUE if succeeds in sampling ends, otherwise returns FALSE.
 *************************************************************************/
{ 
    st_type		sites = SitesCMSA(L);
    fm_type		M,*model=ModelsCMSA(L);
    Int4		t1,t2,n,s,end,N,ntyp;
    Int4 		d0,d,i,j,e,start,length,offset,diff;
    double		map0,map,rand_no,*prob,total,ratio;
    double		*tmp_map;
    unsigned char	*seq;
    BooLean		right;
    
    ntyp=nTypeSites(sites);
    if(ntyp < 2) return FALSE; else if(t < 1 || t >= ntyp) return FALSE;
    t1=t; t2 = t1+1; // slide left: source = t2, denstination = t1.

    // 2. Make sure there are enough columns.
    if(nColsFModel(model[t2]) <= 3) return FALSE;
    map0 = RelMapCMSA(L); /** 1.b. Determine the alignment map. **/
    *oldMap=map0;

        // 3. Remove the last column from the source model. 
	M = model[t2]; length = LenFModel(M);
	d0=RmColumnMSA(t2,1,L);
	diff = length - LenFModel(M);
	length = LenFModel(model[t1]);

	// 4. Add the column to destination model.
	end = length + diff;
	tmp_map = new double [end+3];
	NEW(prob,end+3,double);
	for(total=1.0, i=length+1; i<=end; i++){
		AddColumnMSA(t1,i,L);
		map = RelMapCMSA(L);
		tmp_map[i]=map;
		ratio = exp(map-map0);	/** ratio of new to old **/
		if(temperature != 1.0) ratio = pow(ratio,temperature);
		prob[i] = ratio; total += ratio; 
		s = LenFModel(model[t1]); RmColumnMSA(t1,s,L);
	}
	// 5. Sample an alignment.
	rand_no = ((double)Random()/(double)RANDOM_MAX)*total;
	for(i=length+1; i<=end; i++){
		rand_no -= prob[i];
		if(rand_no <= 0.0){
			AddColumnMSA(t1,i,L);
// fprintf(stderr,"slide columns right to left (diff = %d)\n",diff);
			*newMap=tmp_map[i]; delete [] tmp_map;
			free(prob); return TRUE;
		} 
	}
	delete [] tmp_map;
	AddColumnMSA(t2,1-diff,L); free(prob); return FALSE;
}

BooLean	SampSlideColRightCMSA(Int4 t, double *oldMap, double *newMap, 
	cma_typ L, double temperature)
/*************************************************************************
  slide the end of one model onto the next model.

   --[**.*.*.*]---[**.*...***]--
               |
               |		This operation allows escape from some 
               V                 nasty traps in alignment space.
   --[**.*.*]--[*.:**.*...***]--

  returns TRUE if succeeds in sampling ends, otherwise returns FALSE.
 *************************************************************************/
{ 
    st_type		sites = SitesCMSA(L);
    fm_type		M,*model=ModelsCMSA(L);
    Int4		t1,t2,n,s,end,N,ntyp;
    Int4 		d0,d,i,e,start,length,offset,diff;
    double		map0,map,rand_no,*prob,total,ratio;
    unsigned char	*seq;
    BooLean		right;
    double		*tmp_map;
    
    ntyp=nTypeSites(sites);
    if(ntyp < 2) return FALSE; else if(t < 1 || t >= ntyp) return FALSE;
    t1=t; t2 = t1+1;
    if(nColsFModel(model[t1]) <= 3) return FALSE;
    map0 = RelMapCMSA(L); /** 1.b. Determine the alignment map. **/
    *oldMap=map0;
    // source = t1, denstination = t2.
    	M = model[t1]; length = LenFModel(M);
        /** 4. Remove the last column from the source model. **/
	RmColumnMSA(t1,length,L);
	diff = length - LenFModel(M);
	NEW(prob,diff+3,double);
	tmp_map = new double [diff+2];

	/** 5. Add the columns to destination model **/
	for(total=1.0, i=1; i<=diff; i++){
		d=AddColumnMSA(t2,-(i-1),L);
		map = RelMapCMSA(L);
		tmp_map[i]=map;
		ratio = exp(map-map0);	/** ratio of new to old **/
		if(temperature != 1.0) ratio = pow(ratio,temperature);
		prob[i] = ratio; total += ratio; 
		RmColumnMSA(t2,1,L);
	}
	/** sample an alignment **/
	rand_no = ((double)Random()/(double)RANDOM_MAX)*total;
	for(i=1; i <= diff; i++){
		rand_no -= prob[i];
		if(rand_no <= 0.0){
			AddColumnMSA(t2,-(i-1),L);
#if 1
fprintf(stderr,"slide columns from left to right adjacent models (diff = %d)\n",diff);
#endif
			*newMap=tmp_map[i]; delete [] tmp_map;
			free(prob); return TRUE;
		} 
	} delete [] tmp_map;
	AddColumnMSA(t1,length,L); free(prob);
	return FALSE;
}

dh_type	RtnBestSeqCMSA(Int4 Num, BooLean KeepFirst, cma_typ cma)
// Print out the Num best sequences in the alignment.
{
        Int4    i,j,N = NumSeqsCMSA(cma),start;
        dh_type dH=dheap(N+2,4);
        ss_type data = TrueDataCMSA(cma);
        BooLean *skip; NEW(skip,N+3,BooLean);
        if(KeepFirst){ start=2; Num--; } else start=1;
        assert(Num > 0);
        double prob;
	h_type HG=Histogram("prob scores for seqs",-5000,5000,50);
        for(Int4 J=start; J <= N; J++){
             prob=0.0;
             for(Int4 m=1; m <= nBlksCMSA(cma); m++){
               prob += GetProbCMSA(m,J,cma);
               // prob = GetGappedProbCMSA(m,J,cma);
             }
             if(!std::isfinite(prob)) prob=-99999999999999.9;
             insrtHeap(J,(keytyp) -prob,dH);
	     IncdHist(prob,HG);
        }
#if 0
	for(i=0; !emptyHeap(dH); ){
		prob = -(double)minkeyHeap(dH);
		j=delminHeap(dH);
		fprintf(stderr,"%d: prob=%.2f\n",j,prob);
                assert(j <= N); 
		i++; if(i >=  Num) break;
	} PutHist(stderr,50,HG); NilHist(HG);
        Nildheap(dH);
	return j;
#else
	PutHist(stderr,50,HG); NilHist(HG);
	return dH;
#endif
}


#define	USAGE_START	"USAGE: gsq_typ cmafile operation [options]\n\
      operation:\n\
          'A': add a column to the end of a block\n\
	       or A=<blk>:<add> (if <add> < 0 then add to left of block)\n\
          'B': realign starting from the best sequence\n\
          'C': Contract the alignment by removing marginal columns\n\
               or C=<int> (remove the worst BILD-score columns leaving <int> columns).\n\
          'D': delete unsupported blocks\n\
	       or D=<blk>\n\
          'E': extend fake to real\n\
          'F': fuse blocks (gap to insert)\n\
	       or F=<blk> (fused to following block)\n\
          'H': shift blocks (right or left)\n\
          'I': inserts column(s) after an existing column\n\
	       or I=<blk>:<site>:<length> (if length < 0 then insert before a column)\n\
          'J': fuse blocks (force together)\n\
          'L': Length blocks iteratively until convergence\n\
          'M': Transfer (move) column from one block to adjacent block\n\
	       or M=<blk><cols> where <cols> < 0 --> transfer to left block.\n\
          'N': iron out indels next to deletions\n\
          'O': convert to a one block alignment\n\
          'P': Output a hmmer 2.0 formated hmm from the input alignment\n\
          'Q': show alignment quality\n\
          'R': convert columns to inserts (find likely column)\n\
	       or R=<blk>:<site>:<length>\n\
          'S': Split up blocks with lots of indels\n\
	       or S=<blk>:<site>\n\
          'T': trim ends of conserved blocks\n\
	       or T=<blk>:<remove> (if <remove)> < 0 then remove from left side)\n\
          'V': remove overhangs on either end (needed for mapgaps)\n\
	       or V=<int>:<int> to shorten flank to <int> on either side\n\
          'X': Expand the alignment by adding columns\n\
          'Z': optimize only \n\
   options:\n\
     -adj=<real>- sequence weight adjustment (default: 1.0)\n\
     -dms=<char> - Choose the number of Dirichlet mixture components (default: 20)\n\
                     'T': 20 components\n\
                     'f': 52 components\n\
                     'F': 58 components\n\
                     'S': 72 components\n\
                     'M': 110 components\n\
                     'L': 134 components\n\
                     'P': use psi-blast pseudocounts\n\
     -maxiter=<int> - maximum number of iterations during optimization (default: 1)\n\
     -O         - optimize prior to operation\n\
     -o         - optimize after operation\n\
     -p<int1>:<int2>:<int3>:<int4>  - priors for computing transition probabilities\n\
                  int1 = average residue spacing between insertions (within blocks) (default: 1000)\n\
                  int2 = average length of insertion extensions (default: 5)\n\
                  int3 = average residue spacing between deletions (default: 1000)\n\
                  int4 = average length of deletion extensions (default: 2)\n\
     -temp=<real> - starting temperture for optimization (default: 0)\n\
     -wt=<real> - prior weight (default=1.0)\n\
     -x         - dummy\n\n"

/**************************** Global Variables ******************************/
int	main(Int4 argc,char *argv[])
{ 
	Int4	arg,i,s,time1,block=0,site=0,remove=0,add=0,length=0,maxiter=1;
	Int4	right=-1,left=-1;
	char	mode=0,dms_mode='T',str[300],code=0;
	// Int4    aa_per_io=2000,aa_per_do=2000,exp_ie=1,exp_de=1;
	// Int4    aa_per_io=500,aa_per_do=500,exp_ie=1,exp_de=1;
	Int4    aa_per_io=20,aa_per_do=150, exp_ie=1,exp_de=1;	// best setting...
	BooLean	Optimize=FALSE,PreOptimize=FALSE;
	double	SqWtAdj=0,PriorWt=1.0,temp=0.0;
        static double blosum62freq[21] = {  0.0, /*  X
            C     G     A     S     T     N     D     E     Q     K */
        0.025,0.074,0.074,0.057,0.051,0.045,0.054,0.054,0.034,0.058,
        /*  R     H     W     Y     F     V     I     L     M     P */
        0.052,0.026,0.013,0.032,0.047,0.073,0.068,0.099,0.025,0.039 };

	time1=time(NULL); 
	if(argc < 3) print_error(USAGE_START);
	mode=argv[2][0];
	if(mode == 'A' && argv[2][1] != 0){
	   if(sscanf(argv[2],"A=%d:%d",&block,&add) == 2){
		fprintf(stderr,"block = %d; add = %d\n",block,add);
		if(block <=0 || add == 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'C' && argv[2][1] != 0){
	   if(sscanf(argv[2],"C=%d",&length) == 1){
		fprintf(stderr,"len=%d\n",length);
		if(length <= 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'R' && argv[2][1] != 0){
	   if(sscanf(argv[2],"R=%d:%d:%d",&block,&site,&length) == 3){
		fprintf(stderr,"block = %d; site = %d; len=%d\n",block,site,length);
		if(block <=0 || site <= 0 || length <= 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'F' && argv[2][1] != 0){
	   if(sscanf(argv[2],"F=%d",&block) == 1){
		fprintf(stderr,"block = %d\n",block);
		if(block <=0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'I' && argv[2][1] != 0){
	   if(sscanf(argv[2],"I=%d:%d:%d",&block,&site,&length) == 3){
		fprintf(stderr,"block = %d; site = %d; len=%d\n",block,site,length);
		// if(block <=0 || site <= 0 || length <= 0) print_error(USAGE_START);
		if(block <=0 || site <= 0 || length == 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'M' && argv[2][1] != 0){
	   if(sscanf(argv[2],"M=%d:%d",&block,&length) == 2){
		fprintf(stderr,"block = %d; columns = %c\n",block,length);
		if(length ==0) print_error(USAGE_START);
		if(block <=0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'D' && argv[2][1] != 0){
	   if(sscanf(argv[2],"D=%d",&block) == 1){
		fprintf(stderr,"block = %d\n",block);
		if(block <= 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'S' && argv[2][1] != 0){
	   if(sscanf(argv[2],"S=%d:%d",&block,&site) == 2){
		fprintf(stderr,"block = %d; split_size = %d\n",block,site);
		if(block <=0 || site <= 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(mode == 'T' && argv[2][1] != 0){
	   if(sscanf(argv[2],"T=%d:%d",&block,&remove) == 2){
		fprintf(stderr,"block = %d; remove = %d\n",block,remove);
		if(block <=0 || remove == 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);
	} else if(mode == 'V' && argv[2][1] != 0){
	   print_error("mode V not working; needs to be fixed");
	   if(sscanf(argv[2],"V=%d:%d",&left,&right) == 2){
		fprintf(stderr,"left = %d; right = %d\n",left,right);
		if(left < 0 || right < 0) print_error(USAGE_START);
	   } else print_error(USAGE_START);

	} else if(argv[2][1] != 0) print_error(USAGE_START);
	if(!isupper(mode)) print_error(USAGE_START);
	if(strchr(" ABCDEFHIJLMNOPQRSTVXZ",mode)==NULL) print_error(USAGE_START);
	for(arg = 3; arg < argc; arg++){
	   if(argv[arg][0] != '-') print_error(USAGE_START);
	   switch(argv[arg][1]) {
	     case 'a': 
		if(sscanf(argv[arg],"-adj=%lf",&SqWtAdj) == 1){
                   if(SqWtAdj < 0.01 || SqWtAdj >= 1.0) print_error(USAGE_START);
		} break;
	     case 'd': 
		if(sscanf(argv[arg],"-dms=%c",&dms_mode) == 1){
                   if(strchr("TfFSMLP",dms_mode) == NULL) print_error(USAGE_START);
		} else print_error(USAGE_START);
		break;
	     case 'm': 
		if(sscanf(argv[arg],"-maxiter=%d",&maxiter) == 1){
                   if(maxiter < 0 || maxiter >= 1000) print_error(USAGE_START);
		} break;
	     case 'O': 
		if(argv[arg][2] != 0) print_error(USAGE_START); else PreOptimize=TRUE; break;
	     case 'o': 
		if(argv[arg][2] != 0) print_error(USAGE_START); else Optimize=TRUE; break;
	     case 'p':
                  if(sscanf(argv[arg],"-p%d:%d:%d:%d",&aa_per_io,&exp_ie,&aa_per_do,&exp_de) == 4){
                        argv[arg][1]=' ';
                  } break;
	     case 't': 
		if(sscanf(argv[arg],"-temp=%lf",&temp) == 1){
                   if(temp < 0 || temp > 300.0) print_error(USAGE_START);
		} break;
	     case 'w': 
		if(sscanf(argv[arg],"-wt=%lf",&PriorWt) == 1){
                   if(PriorWt < 0.01 || PriorWt > 100) print_error(USAGE_START);
		} break;
	     case 'x': break;
	     default: print_error(USAGE_START);
	   }
	}
	a_type AB = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62);
	gsq_typ *gsq;
	sRandom((UInt4) time(NULL)/2);

   FILE *fp=open_file(argv[1],".cma","r");
   cma_typ rcma=0,cma=ReadCMSA(fp,AB); fclose(fp);
   if(cma==0) print_error("gsq_typ input error");
   gmb_typ *igmb,*ogmb=0;
   double  D,DD,d,pn=PerNatsCMSA(cma); pn=1000.0;
   if(PreOptimize || mode == 'Z'){ 
	Int4	similarity=50; /// use clustering...
	if(mode == 'Z') ExtendFakeToRealCMSA(cma);
	igmb = new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
	igmb->SetMaxIter(maxiter);
	cma=igmb->Sample(0,'S',similarity,temp);
   } else {
	igmb = new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
	igmb->SetMaxIter(maxiter);
   }
   // N = NumSeqsCMSA(cma);
   Int4 Blk=0;
   fprintf(stderr,"Unextended original: map=%g\n",RelMapCMSA(cma));
   fprintf(stderr,"Unextended original: ungapped map=%g\n",UnGappedRelMapCMSA(cma));
   switch (mode){
	case 'A':
	  {
	     Int4  AddLeft=0,AddRight=0;
	     if(block > 0){
		assert(block <= nBlksCMSA(cma)); Blk=block;
	        if(add < 0) AddLeft=-add; else AddRight=add;
	     } else {
	  	Blk=random_integer(nBlksCMSA(cma))+1;   // random 0 <= integer < max..
#if 1
		if(random_integer(2) == 1) AddLeft=1; else AddRight=1;
#else
		do { AddLeft=random_integer(3); AddRight=random_integer(3); }
		while((AddLeft + AddRight) == 0);
#endif
	     } 
   	     fprintf(stderr,"Adding %d columns to left and %d columns to right of block %d\n",
					AddLeft,AddRight,Blk);
#if 0
	     cma_typ xcma=cma; rcma=0;
	     while(AddLeft > 0){
		if(rcma) xcma=rcma;
   	        fprintf(stderr,"  Adding a column to left\n");
		ExtendFakeToRealCMSA(xcma); 
		rcma=ExtendBlkCMSA(xcma, Blk,1, 0); AddLeft--;
		if(xcma != cma) NilCMSA(xcma); xcma = 0;
	     } assert(rcma != 0 || xcma != 0);
	     while(AddRight > 0){
		if(rcma) xcma=rcma;
   	        fprintf(stderr,"  Adding a column to right\n");
		ExtendFakeToRealCMSA(xcma); 
		rcma=ExtendBlkCMSA(xcma, Blk,0, 1); AddRight--;
		if(xcma != cma) NilCMSA(xcma);
	     }
#else
	        // ExtendFakeToRealCMSA(cma);	// need the extensions to be in fake seq. 
		rcma=ExtendBlkCMSA(cma, Blk,AddLeft,AddRight);
#endif
	  } break;
	case 'B':
	  {
	     Int4 j,Num=5;
	     // Int4	RtnBestSeqCMSA(Int4 Num, BooLean KeepFirst, cma_typ cma)
	     dh_type dH=RtnBestSeqCMSA(Num, FALSE, cma);
#if 1
	gss_typ *gss=gssCMSA(cma);
        rcma=EmptyCMSA(nBlksCMSA(cma),LengthsCMSA(cma),TrueDataCMSA(cma),
		gss->GapOpen(),gss->GapExtend(),PerNatsCMSA(cma),0,0);

	double prob = -(double)minkeyHeap(dH);
	j=delminHeap(dH);
	fprintf(stderr,"%d: prob=%.2f\n",j,prob);
	
	gsq_typ *gsq=gsqCMSA(j,cma);
	Int4    *sites=GetPosSitesCMSA(j,cma);
	char	*operation=gsq->Operation(nBlksCMSA(cma),sites,LengthsCMSA(cma));	// 
	fprintf(stderr,"%d: operation=%s\n",j,operation);
	gsq_typ *gsq0 = new gsq_typ(operation, gsq->TrueSeq(),sites);
	ReplaceCMSA(j,gsq0,rcma); free(sites);

	ssx_typ *ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,cma,dms_mode);
	for(i=0; !emptyHeap(dH); ){
		prob = -(double)minkeyHeap(dH);
		j=delminHeap(dH);
		fprintf(stderr,"%d: prob=%.2f\n",j,prob);

		Int4    start,sq,score,*newpos,t;
		e_type  sbjE = gss->TrueSeq(j);
                operation=ssx->GapAlnTrace(sbjE,0.0,start,score);
                // if(1){ ssx->PutGappedAln(stderr,sbjE,operation,start,cma); }
                Int4 trace_length=strlen(operation);
                gsq = new gsq_typ[1];
                NEW(newpos,nBlksCMSA(rcma)+3,Int4);
                gsq->initialize(gss->LeftFlank(),gss->RightFlank(),
                                operation,trace_length,start,sbjE,newpos);
                // gsq_typ *gsq=rgss->GetGSQ(sq);
                gsq_typ *ogsq=SwapGsqCMSA(j,gsq,rcma); // ssx->AddToAlign(j,newpos); 
                for(t=1; t<=nBlksCMSA(rcma); t++) AddSiteCMSA(t,j,newpos[t],rcma);
                free(newpos);
		delete []ogsq;

		i++; // if(i >=  Num) break;
	}
	PutCMSA(stdout,rcma); 
	exit(1);
#endif	    
	     Nildheap(dH);
	  } break;
	case 'C':	// contract the alignment by removing columns.
	  {
		ExtendFakeToRealCMSA(cma);
		// rcma=ContractColumnsCMSA(cma,dms_mode,SqWtAdj); 
		rcma=0;
	        ssx_typ *rssx,*ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,cma,dms_mode);
		cma_typ rcma=RmWorstColsCMSA(length,ssx);
		// double **BS=BILD_ScoresCMSA(ssx);
		// for(Int4 blk=1; blk <= nBlksCMSA(cma); blk++) free(BS[blk]); free(BS);
#if 0
		double dd,dsw,gp,DD;
		if(rcma){ rssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de,pn,rcma,dms_mode); }
		else rssx=ssx;
		// dd=UnGappedRelMapCMSA(cma); 
		// dd=ssx->GapMap(0.0,'B'); 
		// dd=ssx->GapMap( ); 
		dd=rssx->AdjstdBildLLR(284.8); 
		dsw=rssx->RtnWtNumSeqs(); 
		gp=ssx->RtnIndelPenalty();
		fprintf(stderr,"GISMO LLR (%d cols) = %.2f; SqWt=%.1f; adjLLR = %.2f; penalty=%.2f.\n",
			length,dd,dsw,dd/dsw,gp);
		if(rssx != ssx) delete rssx; delete ssx;
#else
		delete ssx;
#endif
		if(rcma){ ExtendFakeToRealCMSA(rcma); PutCMSA(stdout,rcma); return 0; }
		else { PutCMSA(stdout,cma); return 1; }
	  } break;
	case 'R':
	  {
	    if(block <= 0){
	      ssx_typ *ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,cma,dms_mode);
	      double	max=0,min=999999;
	      Int4	theblk=0,maxpos=0,limit=4;
	      h_type HG=Histogram("fraction of deletions",0,100,1);
	      for(Int4 blk=1; blk <= nBlksCMSA(cma); blk++){
	        for(Int4 pos=1; pos <= LengthCMSA(blk,cma); pos++){
#if 0
		  double d = FractDeletionsCMSA(blk, pos, cma);
		  if(d > max){ max=d; theblk=blk; maxpos=pos; }
		  IncdHist(100*d,HG);
#else
		  double RE=ssx->RelEntropy(blk,pos);
		  if(RE < min){ min=RE; theblk=blk; maxpos=pos; }
		  IncdHist(10*RE,HG);
#endif
	        }
	      } PutHist(stderr,50,HG); NilHist(HG);
	      Int4	len=LengthCMSA(theblk,cma);
	      fprintf(stderr,"blk=%d; len=%d; insert=%d..%d; fract_del=%.3f\n",
			theblk,LengthCMSA(theblk,cma),maxpos,maxpos,min);
	      if(len == limit) rcma=RmBlockCMSA(cma,theblk); 
	      else if(maxpos == 1) rcma=TrimBlkCMSA(cma,theblk,1,0,limit); 
	      else if(maxpos == len) rcma=TrimBlkCMSA(cma,theblk,0,1,limit); 
	      else rcma=ConvertColsToInsertsCMSA(cma,theblk,maxpos,maxpos);
	      if(rcma==0) print_error("TrimBlkCMSA()?: too many columns removed from alignment");
	      delete ssx;
	    } else {
	      if(block > nBlksCMSA(cma)) print_error("block out of range");
	      if(site > LengthCMSA(block,cma)) print_error("site out of range");
	      if(length <= 0) print_error("length < 1");
	      if((site + length-1) > LengthCMSA(block,cma))print_error("trimming goes beyond end of MSA");
	      rcma=ConvertColsToInsertsCMSA(cma,block,site,site+length-1);
	    } 
	  } break;
	case 'D':
	  {
	   Int4	   blk;
	   double  d,D;
	   if(block > 0){
	     rcma=RmBlockCMSA(cma,block); 
	   } else {
	    cma_typ tmp_cma=cma;
	    for(blk=1; blk <= nBlksCMSA(tmp_cma); blk++){
	     D=FieldRelMapCMSA(tmp_cma,blk);
	     fprintf(stderr,"block %3d: %.3f\n",blk,D);
	     if(D < 0.0) rcma=RmBlockCMSA(tmp_cma,blk); else continue;
	     PutConfigCMSA(stderr,rcma);
	     if(tmp_cma != cma) NilCMSA(tmp_cma); tmp_cma=rcma;
	     blk--;
	    }
	   }
	  } break;
	case 'E': ExtendFakeToRealCMSA(cma); PutCMSA(stdout,cma); break;
	case 'F':
	  ExtendFakeToRealCMSA(cma); // PutCMSA(stdout,cma);
	  if(block > 0) Blk=block;
	  else Blk=random_integer(nBlksCMSA(cma)-1)+1;   // random 0 <= integer < argument 
	  rcma=FuseBlocksCMSA(Blk,cma);
	  break;
	case 'H':
	  {
		if(nBlksCMSA(cma) > 1) print_error("fatal: 'H' option requires single block cma file");
		ssx_typ *ssx=igmb->RtnSSX();
		DD = ssx->GapMap(); d = DD/(double) TotalLenCMSA(cma);
		fprintf(stderr,"Original virtual LLR=%.3f (%.3f nats per column).\n",DD,d);
		cma_typ	xcma=0;
		Int4	blk,Blk=1,min_half_blk=3;  // use an even number.
min_half_blk=1;
		set_typ	Set=IndelBreakPntsCMSA(Blk,min_half_blk,cma);
		// if(CardSet(Set) > 0) xcma=SplitUpBlocksCMSA(Set,cma);
		if(CardSet(Set) > 0) xcma=SplitUpBlkCMSA(1,Set,cma);
		if(nBlksCMSA(xcma) == 1){
			fprintf(stderr,"SplitUpBlkCMSA() operation failed\n");
			NilCMSA(xcma); xcma=0; break; 
		} PutConfigCMSA(stderr,cma); PutConfigCMSA(stderr,xcma);
	  	Blk=random_integer(nBlksCMSA(xcma)-1)+1;   // random 0 <= integer < argument 
		Int4 shift=random_integer(2); if(shift==0) shift=-1;
		rcma=ShiftBlkCMSA(Blk,shift,xcma); NilCMSA(xcma);
		if(shift==-1) fprintf(stderr,"shifted block %d to left\n",Blk);
		else fprintf(stderr,"shifted block %d to right\n",Blk);
	        xcma=OneBlockCMSA(rcma);
	        PutConfigCMSA(stderr,rcma); PutConfigCMSA(stderr,xcma);
		NilCMSA(rcma); rcma=xcma;
	  } break;
	case 'I':
	  {
#if 0   // test out adding gap residues to the ends of sequences...
    {
        Int4    sq=244;
        gss_typ *gss=gssCMSA(cma);
        gsq_typ *gsq=gsqCMSA(sq,cma);
        gsq->Put(stderr,AB);
        unsigned short gap=5;
        UInt4   pos=LenSeq(gsq->FakeSeq());
        gss->InsertGap(sq,pos,gap);
        gsq->Put(stderr,AB);
        exit(1);
    }
#endif
#if 0   // DEBUG FixTmplOperation(); // in cma_gmb.cc
        {
        FILE *fp = open_file(argv[1],"","r");
        char str[5004];
        Int4    R,L,S=3,Cto=9;
        fgets(str,5000,fp);
        i=strlen(str); str[i-1]=0;  // get rid of end of line.
        fprintf(stderr,"operation=%s\n",str);
        for(i=0,j=0; str[i]; i++){ if(strchr(" MmIi",str[i])) j++; }
        e_type rE=RandomSeq(j+Cto,1,blosum62freq,AB);
        char    *Op=FixTmplOperation(str,rE,S,L,R);
        fprintf(stderr,"Operation=%s\n",Op);
        exit(1);
        }
#endif
#if 0	// use for multiple cmas...sloppy right now..
        fp = open_file(argv[1],".cma","r");
	Int4	Number;
	cma_typ *IN_CMA=MultiReadCMSA(fp,&Number,AB); fclose(fp);
        fp = open_file(argv[1],"_out.cma","w");
	for(i=1; i <= Number; i++){
	    cma=IN_CMA[i];
	     if(block > 0){
	      assert(block <= nBlksCMSA(cma));
	      assert(site <= LengthCMSA(block,cma)); assert(length != 0);
	      rcma=InsertColumnsCMSA(cma,block,site,length);
	      PutCMSA(fp,rcma);
	     } else { print_error(" 'I' option not yet implemented"); }
	} fclose(fp); exit(1);
#else
	     if(block > 0){
	      assert(block <= nBlksCMSA(cma));
	      assert(site <= LengthCMSA(block,cma)); assert(length != 0);
	      rcma=InsertColumnsCMSA(cma,block,site,length);
	     } else { print_error(" 'I' option not yet implemented"); }
#endif
	  } break;

	case 'J':
	  Blk=random_integer(nBlksCMSA(cma)-1)+1;   // random 0 <= integer < argument 
	  // rcma=SimpleFuseBlksCMSA(Blk, 500,cma); // works with 2 blocks only!!!
	  rcma=FuseBlksCMSA(Blk,500,cma);
	  break;
	case 'L':	// Lengthen the alignment...
	  {
		Int4	blk,R,AddLeft=0,AddRight=0;
		char	C;
		ssx_typ *ssx=0,*rssx=0;
		dh_type dH=dheap(nBlksCMSA(cma) *3,4);
		for(blk=1; blk <= nBlksCMSA(cma); blk++){
			R=blk*2; insrtHeap(R,(keytyp) Random(),dH);
			R=blk*2+1; insrtHeap(R,(keytyp) Random(),dH);
		}
		char	method='B';
		method=' ';
#if 1
		rcma=LengthenBlksCMSA(cma);
#else
#if 0
		cma_typ xcma=0;
		set_typ bset=IndelBreakPntsCMSA(Blk,min_half_blk,cma);
		if(CardSet(bset) > 0) xcma=SplitUpBlkCMSA(1,bset,cma);
		NilSet(bset);
#else
		cma_typ xcma=cma;
#endif
		while(!emptyHeap(dH)){
		   R=delminHeap(dH); AddLeft=0; AddRight=0;
		   if(R%2 == 0){ blk = R/2; AddLeft=1; C='L'; }
		   else { blk = (R-1)/2; AddRight=1; C='R'; }
		   rcma=ExtendBlkCMSA(xcma,blk, AddLeft, AddRight);
		   if(rcma){
			fprintf(stderr,"Try adding column to block %d%c\n",blk,C);
   			ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,xcma,dms_mode); 
			DD = ssx->GapMap(0,method); d = DD/(double) TotalLenCMSA(xcma);
			fprintf(stderr,"Original BILD LLR=%.3f (%.3f nats per column).\n",DD,d);
			delete ssx;
   			rssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,rcma,dms_mode); 
	  		D = rssx->GapMap(0,method); d = D/(double) TotalLenCMSA(rcma);
			fprintf(stderr," Modified BILD LLR=%.3f (%.3f nats per column).\n",D,d);
			delete rssx;
			if(D > DD){
				fprintf(stderr,"Column added to block %d%c\n",blk,C);
				insrtHeap(R,(keytyp) Random(),dH);
				if(xcma!= cma) NilCMSA(xcma); xcma=rcma;
			} else {
				NilCMSA(rcma); rcma=0;
			}
		   }
		} if(xcma != cma) rcma=xcma; else rcma=0;
#endif
	  } break;
	case 'M':
	    {
#if 1
		// 'H'
		if(nBlksCMSA(cma) > 1) print_error("fatal: 'H' option requires single block cma file");
		ssx_typ *ssx=igmb->RtnSSX();
		DD = ssx->GapMap(); d = DD/(double) TotalLenCMSA(cma);
		fprintf(stderr,"Original virtual LLR=%.3f (%.3f nats per column).\n",DD,d);
		cma_typ	xcma=0;
		Int4	blk,Blk=1,min_half_blk=3;  // use an even number.
min_half_blk=1;
		set_typ	Set=IndelBreakPntsCMSA(Blk,min_half_blk,cma);
		// if(CardSet(Set) > 0) xcma=SplitUpBlocksCMSA(Set,cma);
		if(CardSet(Set) > 0) xcma=SplitUpBlkCMSA(1,Set,cma);
	        PutConfigCMSA(stderr,cma); PutConfigCMSA(stderr,xcma);

		if(nBlksCMSA(xcma) == 1){
			fprintf(stderr,"SplitUpBlkCMSA() operation failed\n");
			NilCMSA(xcma); xcma=0; break; 
		}
	  	block=random_integer(nBlksCMSA(xcma)-1)+1;   	// block: 1..nBlks-1;
	  	if(random_integer(2) == 1){ length = 1; }
		else { length=-1; block++; }			// block: 2..nBlks.
		fprintf(stderr,"block = %d; length = %d\n",block,length);
		if(length > 0){ rcma=TransferColumnCMSA(block,TRUE,length,xcma); }
		else { assert(length < 0); rcma=TransferColumnCMSA(block,FALSE,-length,xcma); }
		if(length ==-1){	// transfer to left.
		   fprintf(stderr,"transferred column from block %d to block %d\n", block,block-1);
		} else {		// transfer to right.
		   fprintf(stderr,"transferred column from block %d to block %d\n", block,block+1);
		} NilCMSA(xcma); xcma=OneBlockCMSA(rcma);
	        PutConfigCMSA(stderr,rcma); PutConfigCMSA(stderr,xcma);
		NilCMSA(rcma); rcma=xcma;
#else
	        if(block==0){
	  	   block=random_integer(nBlksCMSA(cma)-1)+1;   	// block: 1..nBlks-1;
	  	   if(random_integer(2) == 1){ length = 1; }
		   else { length=-1; block++; }			// block: 2..nBlks.
		} fprintf(stderr,"block = %d; numCols= %c\n",block,length);
		if(length > 0){ rcma=TransferColumnCMSA(block,TRUE,length,cma); }
		else { assert(length < 0); rcma=TransferColumnCMSA(block,FALSE,-length,cma); }
#endif
	    } break;
	// case 'N': rcma=RmWrinklesCMSA(cma); break;
	case 'N': IronOutCMSA(cma);PutCMSA(stdout,cma);  break;
	case 'O': rcma=OneBlockCMSA(cma); break;
	case 'S':
	  {
		Int4	blk,Blk=1,min_half_blk=2;  // use an even number.
		set_typ	Set=0;
		if(block > 0){	// input block and site given.
		  Set=MakeSet(LengthCMSA(block,cma)+3);
		  ClearSet(Set); AddSet(site,Set);
		  rcma=SplitUpBlkCMSA(block,Set,cma);
		} else if(nBlksCMSA(cma) > 1){
		  dh_type dH=dheap(nBlksCMSA(cma)+3,4);
		  for(blk=1; blk <= nBlksCMSA(cma); blk++){
			   insrtHeap(blk,(keytyp) Random(),dH);
		  }
		  while((Blk=delminHeap(dH)) != 0){
		        if(Set) NilSet(Set);
			Set=IndelBreakPntsCMSA(Blk,min_half_blk,cma);
			if(CardSet(Set) > 0) break;
		  } if(Blk > 0) rcma=SplitUpBlkCMSA(Blk,Set,cma);
		  Nildheap(dH);
		} else {
		   Set=IndelBreakPntsCMSA(Blk,min_half_blk,cma);
		   // if(CardSet(Set) > 0) rcma=SplitUpBlocksCMSA(Set,cma);
		   if(CardSet(Set) > 0) rcma=SplitUpBlkCMSA(1,Set,cma);
		} PutSet(stderr,Set); NilSet(Set); 
	  } break;
	case 'T':	// trim ends of blocks
	  {
	     Int4 limit=4,min_half_blk=3;  // use an even number.
	     if(block > 0){
		Int4 RmLeft=0,RmRight=0;
		assert(block <= nBlksCMSA(cma));
		if(remove < 0) RmLeft=-remove; else RmRight=remove;
		rcma=TrimBlkCMSA(cma,block,RmLeft,RmRight,limit); 
	         if(rcma==0) print_error("TrimBlkCMSA(): too many columns removed from alignment");
	     } else if(nBlksCMSA(cma) > 1){
		rcma=TrimDownCMSA(cma,dms_mode,limit);
	     } else {
#if 0
		cma_typ	xcma;
		set_typ Set=IndelBreakPntsCMSA(1,min_half_blk,cma);
		if(CardSet(Set) > 0){
			xcma=SplitUpBlkCMSA(1,Set,cma);
	  		PutConfigCMSA(stderr,cma);
	  		PutConfigCMSA(stderr,xcma);
			rcma=TrimDownCMSA(xcma,dms_mode, limit);
			NilCMSA(xcma); xcma=0;
	  		PutConfigCMSA(stderr,rcma);
		} else { rcma=TrimDownCMSA(cma,dms_mode,limit); }
		if(nBlksCMSA(rcma) > 1){ xcma=OneBlockCMSA(rcma); NilCMSA(rcma); rcma=xcma; }
		NilSet(Set);
#else	// trim ends based on first sequence...
		Int4 pos[4],x,RmLeft=0,RmRight=0,limit=4,lenCMA=LengthCMSA(1,cma);
		// PosSiteCMSA(1,1,pos,cma); // consensus has no flanking regions or inserts...
		for(x=1; IsDeletedCMSA(1,1,x, cma); x++){ RmLeft++; assert(RmLeft < (lenCMA -limit)); }
		for(x=lenCMA; IsDeletedCMSA(1,1,x,cma); x--){ RmRight++; assert(x > RmLeft); }
		rcma=TrimBlkCMSA(cma,1,RmLeft,RmRight,limit); 
	        if(rcma==0) print_error("TrimBlkCMSA(): too many columns removed from alignment");
		// remove internal deletions?  or just do twkcma -iron=0.50?
	        // rcma=ConvertColsToInsertsCMSA(cma,block,site,site+length-1);
#endif
	     }
	  } break;
	case 'V': 
	  {
		if(left >= 0){
		   PutShrinkFlanksCMSA(stdout,left,right,cma);
	        } else {
	   	   print_error("mode V not working; needs to be fixed");
		   rcma=RmOverHangsCMSA(cma); // PutCMSA(stdout,rcma);  TotalNilCMSA(rcma); rcma=0;
		}
	  } break;
	case ' ':	// test mode...
	  {
		double	d,D,oldMap,newMap;
		if(SampSlideColLeftCMSA(2,&oldMap, &newMap, cma, 0.0)){
		   fprintf(stderr,"slide column left: old=%g; new=%g\n",oldMap,newMap);
		}
		//if(SampSlideColRightCMSA(1,&oldMap, &newMap, cma, 0.0);
	  } break;
	case 'P':	// HMMER profile HMM construction...
	  {
	    // pn=693; // NOTE: 693 nats == 1000 bits.
	    // pn=1000;  // This is default for GISMO... 1/1000 nats
	    pn=1443;  // This is what Sean Eddy is using: ln(x)*1.443 * 1000; 1/1443 nats == 1/1000 bits.
	    // dms_mode='O'; 
	    dms_mode='T';	// nearly as good as 'F'.
	    dms_mode='f';	// nearly as good as 'F'.
	    dms_mode='F';	// best setting...
	    // Int4   aa_per_io=200,aa_per_do=200,exp_ie=1,exp_de=1;
	    Int4    aa_per_io=20,aa_per_do=150, exp_ie=1,exp_de=1;	// best setting...
	    ssx_typ *ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,cma,dms_mode);
	    if(PriorWt > 0) ssx->SetPriorWt(PriorWt);
	    if(SqWtAdj > 0) ssx->SetSqWtAdjust(SqWtAdj);
#if 0	// Test out new hmm_typ.... 
	    smx_typ *smx = ssx->RtnSMX( );
	    PutSMatrix(stderr,smx[1]);
	    ssx->PutPenalties(stderr);
	    exit(1);
#else
	    // ssx->OldSMXs(0);
	    // smx_typ *smx = ssx->RtnSMX( );
#if 0
	    smx_typ *smx = ssx->RtnStraightSMX( );	// this does a bit better?
#else
	    smx_typ *smx = ssx->RtnWtCntsSMX( );	// this does a bit better?
#endif
	    Int4 **mat_emit= ValuesSMatrix(smx[1]),**ins_emit=0;
	    char *name= NameCMSA(cma);
	    Int4 len=LengthCMSA(1,cma);
	    Int4 *mm,*mi,*md,*ii,*im,*dd,*dm,*bm,*me=0;
	    ndl_typ *ndl=ssx->RtnNDL();
	    ndl->GetTransProb(&mm,&mi,&md,&ii,&im,&dd,&dm,&bm);
	    hmm_typ *hmm = new hmm_typ(name,len,mat_emit,ins_emit,mm,mi,md,ii,im,dd,dm,bm,me,AB,'R');
	    hmm->Put(stdout);
	    fprintf(stderr,"Use hmmcalibrate to add EVD parameters. Use hmmsearch to search for hits.\n");
// for(Int4 x =1; smx[x] != 0; x++) NilSMatrix(smx[x]); free(smx);
	    delete hmm;
#endif
	  } break;
	case 'Q':
	{
	    // Int4   aa_per_io=200,aa_per_do=200,exp_ie=1,exp_de=1;
	    ssx_typ *ssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,cma,dms_mode);
	    if(PriorWt > 0) ssx->SetPriorWt(PriorWt);
	    if(SqWtAdj > 0) ssx->SetSqWtAdjust(SqWtAdj);
	    ColumnQualityCMSA(ssx);
	    ssx->PutPenalties(stderr);
#if 0
	    fprintf(stderr,"GAMBIT LLR = %.2f; GISMO LLR = %.2f; gLLR = %.2f (%d columns).\n",
			ssx->GapMap(),UnGappedRelMapCMSA(cma),RelMapCMSA(cma),NumColumnsCMSA(cma));
#else
	    fprintf(stderr,"GISMO LLR = %.2f; pcLLR = %.2f (%d columns).\n",
			UnGappedRelMapCMSA(cma),UnGappedRelMapCMSA(cma)/NumColumnsCMSA(cma),NumColumnsCMSA(cma));
#endif
	    delete ssx;
	    PutIndelsCMSA(stderr,cma);
	    exit(1);

	    rcma=CopyCMSA(cma);
#if 1
	    ExtendFakeToRealCMSA(rcma);
#else
	    SaveBestCMSA(rcma);
	    PutAlnCMSA("junk.cma",rcma,NULL); 
	gss_typ *gss=gssCMSA(rcma);
	for(Int4 sq=1; sq <= gss->NumSeq(); sq++){
		gsq_typ *gsq=gss->GetGSQ(sq);
		gsq->Put(stderr,AB);
	}
ExtendFakeToRealCMSA(rcma);
	  NilCMSA(rcma);
	  TotalNilCMSA(cma); 
	  NilAlpha(AB);
	  return 1;
#endif
	} break;
	case 'X':	// expand the alignment by adding columns.
	  {
		ExtendFakeToRealCMSA(cma);
		rcma=ExtendColumnsCMSA(cma,dms_mode,SqWtAdj); 
		if(rcma) ExtendFakeToRealCMSA(rcma); else { PutCMSA(stdout,cma); return 1; }
	  } break;
	case 'Z': { 
		PutCMSA(stdout,cma); 
	} break;
	default: print_error(USAGE_START); break;
   }
#if 1	// new optimize configuration...
   if(rcma && Optimize){
	Int4	similarity=50; /// use clustering...
	ogmb = new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,rcma,dms_mode);
	ogmb->SetMaxIter(maxiter);
	rcma=ogmb->Sample(0,'S',similarity,temp);
   }
#endif
   {
	fprintf(stderr,"PerNatsCMSA = %g\n",pn);
	// PutRelMapCMSA(stderr, cma);
	ExtendFakeToRealCMSA(cma);
	ssx_typ *rssx=0,*ssx = 0;
	ssx=igmb->RtnSSX();
#if 0
	{
	  for(Int4 i=1; i <= LengthCMSA(1,cma); i++){
		i=15;
		fprintf(stderr,"Column 1.%d: LLR=%.3f.\n\n",i,ssx->GapMap(0.0,1,i));
		fprintf(stderr," Virtual LLR2=%.3f.\n\n",ssx->VirtualMap2());
		break;
	  }
	}
#endif
	DD = ssx->GapMap(); d = DD/(double) TotalLenCMSA(cma);
	fprintf(stderr,"Original virtual LLR=%.3f (%.3f nats per column).\n",DD,d);
	if(rcma){
	  // if(ogmb) rssx=ogmb->RtnSSX(); else 
	  // rssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,rcma,dms_mode); 
	  rssx = new ssx_typ(aa_per_io,aa_per_do,exp_ie, exp_de, pn,rcma,dms_mode,ssx); 
	  D = rssx->GapMap(); d = D/(double) TotalLenCMSA(rcma);
	  fprintf(stderr," modified virtual LLR=%.3f (%.3f nats per column).\n",D,d);
	  fprintf(stderr," diff LLR = %.3f\n",D-DD);
	  // if(ogmb==0) delete rssx;
	  delete rssx;
	}
   }
	if(rcma){
	  // WriteCMSA("junkIn.cma",rcma);
	  PutConfigCMSA(stderr,cma); PutConfigCMSA(stderr,rcma);
	  if(Blk) fprintf(stderr,"blk=%d; len=%d\n", Blk,LengthCMSA(Blk,rcma));
	  fprintf(stderr,"Original: map=%g\n",RelMapCMSA(cma));
	  fprintf(stderr," modified: map=%g\n",RelMapCMSA(rcma));
	  fprintf(stderr,"Original: ungapped map=%g\n",UnGappedRelMapCMSA(cma));
	  fprintf(stderr," modified: ungapped map=%g\n",UnGappedRelMapCMSA(rcma));
	  // PutRelMapCMSA(stderr,cma);
	  // PutRelMapCMSA(stderr,rcma);
	  PutCMSA(stdout,rcma);  fflush(stdout);
	  if(ogmb){ delete ogmb; ogmb=0; }
	  if(TrueDataCMSA(rcma) == TrueDataCMSA(cma)) NilCMSA(rcma);
	  else TotalNilCMSA(rcma);
	} else {
		// PutCMSA(stdout,cma); 
		if(ogmb) delete ogmb;
	} delete igmb; TotalNilCMSA(cma); NilAlpha(AB);
	fprintf(stderr,"\ttime: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	return 0;
}



