/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "twp_typ.h"

char	*twp_typ::BaseName(char *path)
{
	for(Int4 i=strlen(path)-1; i >= 0; i--){
           if(path[i] == '/'){ i++; return (&path[i]); }
        } return path;
}

Int4	twp_typ::ListChainsPDB(FILE *fp,pdb_typ pdb)
{
	Int4	C;
	fprintf(fp,"File '%s':\n",this->BaseName(FilenamePDB(pdb)));
	for(C=1; C <= nChainsPDB(pdb); C++){
		char chain=ChainCharPDB(C,pdb);
		atm_typ atm=AtomPDB(C, 1, pdb);
		if(IsWaterAtom(atm)){
		  fprintf(fp,"chain %d: '%c'  Water\n",C,chain);
		} else if(IsNucleicAcidAtom(atm)){
		  fprintf(fp,"chain %d: '%c'  DNA/RNA\n",C,chain);
		  // fprintf(fp," %d-%d\n",MinResPDB(C,pdb),MaxResPDB(C,pdb));
		} else if(AminoAcidAtom(atm)){
		  fprintf(fp,"chain %d: '%c'  Protein",C,chain);
		  fprintf(fp," %d-%d\n",MinResPDB(C,pdb),MaxResPDB(C,pdb));
		} else if(IsHeteroAtom(atm)){
		  fprintf(fp,"chain %d: '%c'  Ligand\n",C,chain);
		} else {
		  fprintf(fp,"chain %d: '%c'  ???\n",C,chain);
		}
	} return C;
}

void twp_typ::ReadCETransformFile(char *filename,double xyz[4][5],const char *usage)
/*******************************************************************************
                  xyz[.][1]       xyz[.][2]        xyz[.][3]        xyz[.][4]
xyz[1]:   X2 = ( 0.983195)*X1 + ( 0.109599)*Y1 + ( 0.146000)*Z1 + (  -13.846273)
xyz[2]:   Y2 = (-0.059883)*X1 + ( 0.949108)*Y1 + (-0.309206)*Z1 + (   14.920112)
xyz[3]:   Z2 = (-0.172458)*X1 + ( 0.295267)*Y1 + ( 0.939721)*Z1 + (  -35.552807)
 *******************************************************************************/
{
	FILE *fp=open_file(filename,"","r");
        // if(fscanf(fp,"     X2 = (%9.6lf)*X1 + (%9.6lf)*Y1 + (%9.6lf)*Z1 + (%12.6lf)\n",
        if(fscanf(fp,"     X2 = (%lf)*X1 + (%lf)*Y1 + (%lf)*Z1 + (%lf)\n",
		&xyz[1][1], &xyz[1][2], &xyz[1][3], &xyz[1][4]) != 4){
		fprintf(stderr,"X2 input error\n");
		print_error(usage);
	}
        // if(fscanf(fp,"     Y2 = (%9.6lf)*X1 + (%9.6lf)*Y1 + (%9.6lf)*Z1 + (%12.6lf)\n",
        if(fscanf(fp,"     Y2 = (%lf)*X1 + (%lf)*Y1 + (%lf)*Z1 + (%lf)\n",
		&xyz[2][1], &xyz[2][2], &xyz[2][3], &xyz[2][4]) != 4){
		fprintf(stderr,"Y2 input error\n");
		print_error(usage);
	}
        // if(fscanf(fp,"     Z2 = (%9.6lf)*X1 + (%9.6lf)*Y1 + (%9.6lf)*Z1 + (%12.6lf)\n",
        if(fscanf(fp,"     Z2 = (%lf)*X1 + (%lf)*Y1 + (%lf)*Z1 + (%lf)\n",
		&xyz[3][1], &xyz[3][2], &xyz[3][3], &xyz[3][4]) != 4){
		fprintf(stderr,"Z2 input error\n");
		print_error(usage);
	}
	fclose(fp);
}


Int4	ShortenBond(atm_typ fixed, atm_typ moved,float Len)
// shorten the bond length between 'fixed' and 'moved' by moving the latter closer to the former
{
	float	xf,yf,zf,xm,ym,zm;
	float	dx,dy,dz;	// original difference.
	float	dX,dY,dZ,X,Y,Z;	// new difference.
	float	len,factor;
	xf=AtomX(fixed); xm=AtomX(moved);
	yf=AtomY(fixed); ym=AtomY(moved);
	zf=AtomZ(fixed); zm=AtomZ(moved);
	dx=xf-xm;
	dy=yf-ym;
	dz=zf-zm;
	fprintf(stderr,"dx=%.3lf\n",dx);
	fprintf(stderr,"dy=%.3lf\n",dy);
	fprintf(stderr,"dz=%.3lf\n",dz);
	double dd=(double)((dx*dx) +(dy*dy) + (dz*dz));
	len=(float)sqrt(dd);
	PutAtom(stdout,moved);
	fprintf(stderr,"Len=%.3lf\n",Len);
	fprintf(stderr,"dist=%.3lf; dd=%.3lf\n",len,dd);
	factor = Len/len;
	fprintf(stderr,"factor=%.3lf\n",factor);
	dX = factor * dx;
	dY = factor * dy;
	dZ = factor * dz;
	X = xf - dX;	// xm=xf-dx
	Y = yf - dY;
	Z = zf - dZ;
	atm_typ atm=VirtualAtom(X,Y,Z,AtomChain(moved));
	PutAtom(stdout,atm);
	return 1;
}

/*************************************************************
 need: 
   Get conserved residue positions.

	cma_typ cma = ReadCMSA2(cmafilename,AB);
	fprintf(stderr,"cmafile=%s; cutoff = %d; chains = %s\n",
				cmafilename,cmacutoff,cma_chns);

 *************************************************************/
char	*twp_typ::Foo(cma_typ cma, Int4 cmacutoff, char *cma_chns,
		Int4 *ResStart,Int4 *ResEnd, pdb_typ P)
{
	char	chain,*skip,chn;
	Int4	resStart,resEnd;
	// open cma file.
	a_type AB = AminoAcidAlphabetPDB(P);

	// check that pdb file and chain are the same:
	if(cma_chns[0] == 0) chn=1;
	else chn=GetChainNumberPDB(P,cma_chns[0]);
	e_type pdbE=GetPDBSeq(chn,P);

	e_type trueE=TrueSeqCMSA(1,cma);
	e_type fakeE=FakeSeqCMSA(1,cma);
	if(!IdentSeqs(pdbE,trueE)){
		PutSeq(stderr,pdbE,AB);
		PutSeq(stderr,trueE,AB);
		assert(IdentSeqs(pdbE,trueE));
	}
	assert(IdentSeqs(pdbE,fakeE));
	assert(LengthCMSA(1,cma)==LenSeq(pdbE));
	assert(nBlksCMSA(cma)==1);
	chain = cma_chns[0];

	// get freqs...
	double  **freq=ColResFreqsCMSA(1,cma);
	double freq0=(double)cmacutoff/100.0;
	NEW(skip,LengthCMSA(1,cma)+OffSetSeq(trueE)+2,BooLean);
	resStart=resEnd=0;
	for(Int4 s=1; s <= LengthCMSA(1,cma); s++){
		Int4 r=ResSeq(s,fakeE);
		Int4 s_adj = s+OffSetSeq(trueE);
		double frq = freq[s][r];
		double xfrq,tfq,tfq1,tfq2;
		char res_str[21];
		xfrq=freq[s][AlphaCode('X',AB)];
#if 0
// May also want to check freq in superfamily and main set!!
Int4    **rawCnts=0;
double *wt=HenikoffWeightsCMSA(cma,&rawCnts);
// double *HenikoffWeightsCMSA(cma_typ cma,Int4 ***RawCnts);
free(wt);
for(i=TotalLenCMSA(cma);i > 0;i--){ free(rawCnts[i]); } free(rawCnts);
#endif
	        sprintf(res_str,"%c",AlphaChar(r,AB));
		switch(AlphaChar(r,AB)){ // ND,DE,NQ,YF,ST,LVI,KR,ST,HN
		  case 'D': 	// ND or ED
		    { 
			tfq1=frq+freq[s][AlphaCode('N',AB)]; 
			tfq2= frq+freq[s][AlphaCode('E',AB)]; 
			frq=MAXIMUM(double,tfq1,tfq2);
		    } break;
		  case 'E': 
		    { frq=frq+freq[s][AlphaCode('D',AB)]; } break;
		  case 'F': 
		    { frq=frq+freq[s][AlphaCode('Y',AB)]; } break;
		  case 'K': 
		    { frq=frq+freq[s][AlphaCode('R',AB)]; } break;
		  case 'M':
		  case 'L': case 'I': case 'V': 
		    { 
		      tfq=freq[s][AlphaCode('M',AB)];
		      tfq+=freq[s][AlphaCode('I',AB)];
		      tfq+=freq[s][AlphaCode('L',AB)];
		      tfq+=freq[s][AlphaCode('V',AB)];
		      if(tfq > frq) { sprintf(res_str,"ILV"); frq=tfq; }
		    } break;
		  case 'N': 
		    { 
			tfq1= frq+freq[s][AlphaCode('Q',AB)]; 
			tfq2= frq+freq[s][AlphaCode('D',AB)]; 
			frq=MAXIMUM(double,tfq1,tfq2);
		    } break;
		  case 'Q': 
		    { frq=frq+freq[s][AlphaCode('N',AB)]; } break;
		  case 'R': 
		    { frq=frq+freq[s][AlphaCode('K',AB)]; } break;
		  case 'S': 
		    { frq=frq+freq[s][AlphaCode('T',AB)]; } break;
		  case 'T': 
		    { frq=frq+freq[s][AlphaCode('S',AB)]; } break;
		  case 'Y': 
		    { frq=frq+freq[s][AlphaCode('F',AB)]; } break;
		  default: break;
		}
		if(xfrq > 0.0) frq = frq/(1.0-xfrq);
		if(frq > freq0){
			if(resStart==0) resStart=s_adj;
			resEnd=s_adj;
			skip[s_adj]=FALSE;
			if(xfrq > 0.0){
			 if(chain) fprintf(stdout,
			    "#%c%d%c.W // %s %.3f conserved (%.3f '-').\n",
				AlphaChar(r,AB),s_adj,chain,res_str,frq,xfrq);
			 else fprintf(stdout,
			    "#%c%d.W // %s %.3f conserved (%.3f '-').\n",
				AlphaChar(r,AB),s_adj,res_str,frq,xfrq);
			} else {
			 if(chain) fprintf(stdout,
			 	"#%c%d%c.W // %s %.3f conserved.\n",
				AlphaChar(r,AB),s_adj,chain,res_str,frq);
			 else fprintf(stdout,
			 	"#%c%d.W // %s %.3f conserved.\n",
				AlphaChar(r,AB),s_adj,res_str,frq);
			}
		} else skip[s_adj]=TRUE;
	}
	*ResStart=resStart;
	*ResEnd=resEnd;
	NilSeq(pdbE); 
	return skip;
	// PutContactsPDB(stdout,resStart,resEnd,HA_dmax,dmax,C,C2,chain,skip,P);
}	// free(skip); NilSeq(pdbE); TotalNilCMSA(cma);

//========================= min Distance code ======================

double	twp_typ::GetResDistance(FILE *fptr, Int4 resC,Int4 resD, char chnC, char chnD, pdb_typ P,
		BooLean noMainChn)
/************************************************************
 ************************************************************/
{
	Int4 res0,C,D,ai,aj,r,i,j,x,num_resC,num_resD,nI,nJ;
	C=GetChainNumberPDB(P,chnC); // C = query residue chain.
	D=GetChainNumberPDB(P,chnD); // D = query residue chain.
	if(C==0){ fprintf(stderr,"chnC = %c\n",chnC); print_error("chain not found!"); }
	if(D==0){ fprintf(stderr,"chnD = %c\n",chnD); print_error("chain not found!"); }
	res_typ *ResAllC=MakeResPDB(C,&num_resC,P);
	res_typ *ResAllD=MakeResPDB(D,&num_resD,P);

	a_type AB = AminoAcidAlphabetPDB(P);
	// Find minimum Res-Res distance ...
	atm_typ	atmI,atmJ,minAI,minAJ;
	res_typ	ResI,ResJ;
	char	cI,cJ;
	float	max,min,dd;
	for(i=1; i <= num_resC; i++){
	   ResI=ResAllC[i]; cI=GetCharResidue(ResI,AB);
	   nI=ResidueID(ResI);
	   if(nI != resC) continue;
	   for(j=1; j <= num_resD; j++){
	        ResJ=ResAllD[j]; cJ=GetCharResidue(ResJ,AB);
	        nJ=ResidueID(ResJ);
	        if(nJ != resD) continue;
		dd=0.0; min=999999.0;
		for(ai=1; ai <= ResidueAtomNumber(ResI); ai++){
		  atmI=AtomResidue(ai,ResI);
		  if(noMainChn){
		    if(!(SideAtom(atmI) || (cI=='G' && AlphaCarbonAtom(atmI)))) continue;
		  }
		  //fprintf(stderr,"%s: \n",ResidueName(ResI)); PutAtom(stderr,atmI);
		  for(aj = 1; aj <= ResidueAtomNumber(ResJ); aj++){
		    atmJ=AtomResidue(aj,ResJ);
		    if(noMainChn){
		      if(!(SideAtom(atmJ) || (cJ=='G' && AlphaCarbonAtom(atmJ)))) continue;
		    }
		    dd=DistanceAtoms(atmI,atmJ);
		    if(dd < min){ min=dd; minAI=atmI; minAJ=atmJ; }
		  }
		}
		fprintf(fptr,"%c%d%c\t%c%d%c\t%.2lf\n",cI,nI,chnC,cJ,nJ,chnD,min); 
		PutAtom(fptr,minAI); PutAtom(fptr,minAJ);
		for(x=1; x <= num_resC; x++) NilRes(ResAllC[x]); free(ResAllC); 
		for(x=1; x <= num_resD; x++) NilRes(ResAllD[x]); free(ResAllD); 
		return min;
	   }
	}
	for(i=1; i <= num_resC; i++) NilRes(ResAllC[i]); free(ResAllC); 
	for(i=1; i <= num_resD; i++) NilRes(ResAllD[i]); free(ResAllD); 
	return 0.0;
} 

int	twp_typ::Run(FILE *outfp)
{ 
	Int4	start,arg,n,chain_num=0;
	char	c=' ',chain=' ',chainB=' ';
	Int4	chain_id=1;
	BooLean	silent = FALSE,angles=FALSE,FullOutput=FALSE;
	UInt4 seed=18364592;
	pdb_typ	pdb=0;
	Int4	res0=200,C=1,C1=1,C2=0;
	float	HA_dmax=0.0,dmax=0.0;
	Int4	shift=0;
	Int4	atm_no[5],atom_no=0;
	char	tofile[100],fromfile[100];
	Int4	resStart=0,resEnd=0;
	double	xyz[4][5];
	char	cmafilename[100],cma_chns[50];
	Int4	cmacutoff;
	Int4	water=-1;
	Int4	file_id=0;
	char	this_res=0;
	Int4	shift_start=INT4_MIN,shift_end=INT4_MAX;
	Int4	del_start=0,del_end=0;
	Int4	prnt_start=0,prnt_end=0;
	Int4	renumber_start=0;
	char	*chains=0;
	Int4	SubStart=0,SubEnd=0;
	double	chn_dist=-999.0;
	char	*donor=0,*accept=0;
	BooLean	noMainChn=TRUE;
	char	tmp_str[100];
	FILE	*ofp=stdout;
	if(outfp!=0) ofp=outfp;
// new shrink mode
	Int4	atm_num1=0,atm_num2=0;
	float	lenNew=0.0;
// 

#if 1	// -dist option...
	Int4	nI=0,nJ=0;
	char	cI,cJ,chnI,chnJ;
#endif
	BooLean	AddTER=FALSE;

	if(argc < 2) print_error(WUSAGE);
	TurnOffLicenseStatement();
	start = 2; 
        for(arg = start; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(WUSAGE);
           switch(argv[arg][1]) {
	 	case 'm':
		  if(sscanf(argv[arg],"-m%[^:]:%[^:]:%d%c",tofile,fromfile,&res0,&chain)
				!= 4) print_error(WUSAGE);
		  c = 'm'; silent= TRUE;
                  break;
	 	case 'M':
		  if(sscanf(argv[arg],"-M%[^:]:%[^:]",tofile,fromfile) != 2) print_error(WUSAGE);
		  c = 'M'; silent= TRUE;
                  break;
	 	case 'A':
		  if(sscanf(argv[arg],"-A=%d:%d",&C1,&C2) != 2) print_error(WUSAGE);
		  // if(C1 == C2) print_error(WUSAGE);
		  c = 'A';
                  break;
	 	case 'B':
		 {
		  NEW(chains,101,char);
		  if(sscanf(argv[arg],"-B=%[A-Z]:%lf",chains,&chn_dist) != 2) print_error(WUSAGE);
		  c = 'B';
		 } break;
	 	case 'b': c = 'b'; break;
                case 'a': c = 'a'; silent = TRUE; 
		   if(sscanf(argv[arg],"-accept=%s",tmp_str) == 1){
			accept=AllocString(tmp_str); 
		   } else if(sscanf(argv[arg],"-angles=%c",&chain) == 1){
		   	angles=TRUE; 
		   } else if(sscanf(argv[arg],"-a%d:%d:%d",
			&atm_no[1],&atm_no[2],&atm_no[3]) != 3) print_error(WUSAGE);
		   break;
	 	case 'C':
	 	  if(argv[arg][2] == 0) c = 'C'; // remove MSEs.
		  else C1=IntOption(argv[arg],'C',1,1000,WUSAGE); 
		  break;
	 	case 'c':
		  {
		    Int4 s,e; char c;
		    if(argv[arg][2] == 0) C=IntOption(argv[arg],'c',1,1000,WUSAGE); 
		    else if(sscanf(argv[arg],"-cnt=%d:%d:%c",&s,&e,&c) == 3){
			pdb_typ tmp_pdb = MkPDB(argv[1]);
			Int4 n=CountAtomsPDB(s,e,c,tmp_pdb);
			fprintf(ofp,"%d..%d%c = %d atoms; %d residues\n",
				s,e,c,n,CountResPDB(s,e,c,tmp_pdb));
			NilPDB(tmp_pdb);
			return 0;
		    } else print_error(WUSAGE);
		  }
		  break;
                case 'H': c = 'H'; silent = TRUE; break;
                case 'h': c = 'h'; silent = TRUE; break;
	 	case 'D': 
		  if(isdigit(argv[arg][2])){
		    if(sscanf(argv[arg],"-D%f",&dmax) != 1 || dmax > 100 || dmax < 0)
                        print_error(WUSAGE);
		  } else {
		    c = 'D';
		    if(sscanf(argv[arg],"-D=%c:%[^:]:%c",&chain,tofile,&chainB) != 3)
                        print_error(WUSAGE);
		  } break;
	 	case 'd':
		  if(sscanf(argv[arg],"-dist=%d%c:%d%c",&nI,&cI,&nJ,&cJ) == 4){
		    if(nI < 1) print_error(WUSAGE);
		    fprintf(stderr,"nI=%d; nJ=%d; cI=%c; cJ=%c\n",nI,nJ,cI,cJ);
		    silent=TRUE;
		    // run below...
		  } else if(sscanf(argv[arg],"-donor=%s",tmp_str) == 1){
			donor=AllocString(tmp_str); 
		  } else if(sscanf(argv[arg],"-d%f",&HA_dmax) != 1 || HA_dmax > 100 || HA_dmax < 0)
                        print_error(WUSAGE);
                  break;
	 	case 'E': 
		  c = 'E';
		  if(sscanf(argv[arg],"-E=%c",&this_res) != 1){
		    this_res=0;
		    if(sscanf(argv[arg],"-E%c:%c",&chain,&chainB) != 2)
                        print_error(WUSAGE);
		  } break;
	 	case 'e':
		  c='e';
		  if(sscanf(argv[arg],"-e=%c:%c",&this_res,&chain) != 2){
		    if(sscanf(argv[arg],"-e%d%c",&res0,&chain) != 2){
			res0=IntOption(argv[arg],'e',0,100000,WUSAGE);
			// print_error(WUSAGE);
			chain=0;
		    }
		  }
		  break;
	 	case 'l': 
		  if(strcmp("-list",argv[arg]) == 0) c='l';
		  else print_error(WUSAGE);
		  break;
	 	case 'L': 
		  c='L';
		  break;
	 	case 'F': 
		  if(strcmp("-Full",argv[arg]) == 0){
			FullOutput=TRUE;
		  } else file_id=IntOption(argv[arg],'F',0,100000,WUSAGE);
		  break;
	 	case 'f': 
		   if(sscanf(argv[arg],"-f%[^:]:%d%s",cmafilename,&cmacutoff,cma_chns) != 3)
			print_error(WUSAGE);
		   if(cmacutoff < 1 || cmacutoff > 100) print_error(WUSAGE);
		   for(c=0; cma_chns[c]; c++) if(!isalpha(cma_chns[c])) print_error(WUSAGE);
		   c='f'; silent= TRUE;
		  break;
	 	case 'R': 
		  {
		    char mode='N';
	if(sscanf(argv[arg],"-Read=%c",&mode)==1){	// Test Kanti's routine
	   if(strchr(" SONC",mode) ==0) print_error("Donor atom must be C,N,S,or O)");
	   mps_typ *mps= new mps_typ(argv[1]);
	   h_type tHG=Histogram("theta (angle Xd-DA-Xa",0,4,0.1);	// range: 0...pi
	   h_type dHG=Histogram("donor-to-acceptor distances",0,5,0.05);
	   h_type dhaHG=Histogram("hydrogen-to-acceptor distances",0,5,0.05);
	   h_type haHG=Histogram("angleDHA",0,5,0.05);
	   FILE *tfp=open_file("junk",".txt","w");
	   // if(dmax == 0.0) dmax = 3.6;
	   if(dmax == 0.0) HA_dmax = 2.5;
	   for(Int4 f=1; f<= mps->NumPDB; f++){
	     pdb = mps->pdb[f];
	     char *str=strrchr(FilenamePDB(pdb),'/');
	     fprintf(stderr,"====== %d: file %s (%d chains) ======\n",f,str+1,nChainsPDB(pdb));
	     for(C1=1; C1 <= nChainsPDB(pdb); C1++){
	       Int4 r,i,j,num_res;
	       res_typ *Res=MakeResPDB(C1,&num_res,pdb);
	       for(i=1; i <= num_res; i++){
	         for(j=1; j <= num_res; j++){
		    if(i == j) continue; 
		    // char    GetCharResidue(res_typ R,a_type AB);
		    GetHBondThetaAndDistPDB(stderr,tfp,Res[i],C1,C1,HA_dmax,Res[j],1,
			  pdb,mode,donor,accept,tHG,dHG,dhaHG,haHG);
		    fflush(tfp);
	         }
	       } for(i=1; i <= num_res; i++) NilRes(Res[i]);
	     }
	   } delete mps;
	   PutHist(ofp,60,dHG); NilHist(dHG); 
	   PutHist(ofp,60,tHG); NilHist(tHG);
	   PutHist(ofp,60,haHG); NilHist(haHG); 
	   PutHist(ofp,60,dhaHG); NilHist(dhaHG); 
	   fclose(tfp);
	   return 0;
	}
       			// -R=<int><char> Renumber residues in chain <char> starting from <int>
		   if(argv[arg][2]== '=') {
		    if(sscanf(argv[arg],"-R=%d%c",&renumber_start,&chain) != 2){
                        print_error(WUSAGE);
		    } else if(renumber_start < 1) print_error(WUSAGE);
		   } else if(argv[arg][2]!=0) {
		    if(argv[arg][2] == 'R' && argv[arg][3]==0){
			AddTER=TRUE;
		    } else if(sscanf(argv[arg],"-R%d%c",&chain_num,&chain) != 2)
                        print_error(WUSAGE);
		   }
		   c = 'R'; silent= TRUE; 
		} break;
	 	case 'Q':
		  c='Q';
		  break;
	 	case 'K':
		  c='k';
		  if(sscanf(argv[arg],"-K%d..%d%c",&resStart,&resEnd,&chain) != 3){
		  	if(sscanf(argv[arg],"-K%d..%d",&resStart,&resEnd) != 2){
				print_error(WUSAGE);
		        } else chain=0;
		  } if(resStart < 1 || resEnd < 1 || resStart > resEnd)
		  		print_error(WUSAGE);
		  if(chain) fprintf(stderr,"resStart=%d; resEnd=%d; chain = %c\n",
						resStart,resEnd,chain);
		  else fprintf(stderr,"resStart=%d; resEnd=%d\n",resStart,resEnd);
		  break;
	 	case 'k':
		  c='k';
		  if(sscanf(argv[arg],"-k%d%c",&res0,&chain) != 2){
			res0=IntOption(argv[arg],'k',0,100000,WUSAGE);
			// print_error(WUSAGE);
			chain=0;
		  } resStart=resEnd=res0;
		  break;
	 	case 'r':
		  if(isdigit(argv[arg][2])){
		    res0=IntOption(argv[arg],'r',0,100000,WUSAGE); 
		  } else { c='r'; }
		  break;
	 	case 'O': c = 'O'; 
		  if(sscanf(argv[arg],"-O%d%c%d",&resStart,&chain,&resEnd) == 3){
		    if(resStart > resEnd) print_error(WUSAGE);
		  }  else print_error(WUSAGE);
		  break;
	 	case 'o': c = 'o'; break;
                case 'p': c = 'p'; silent = TRUE; break;
                case 'P': c = 'P'; silent = TRUE; break;
                case 'N': c = 'N'; silent = TRUE; 
		   if(sscanf(argv[arg],"-N%d",&atom_no) != 1) print_error(WUSAGE);
		   break;
                case 'n': c = 'n'; silent = TRUE; 
		  if(isalpha(argv[arg][2])) chain=argv[arg][2];
		  else print_error(WUSAGE); break;
	 	case 'S': c = 'S'; silent = TRUE; break;
	 	case 's': 
fprintf(stderr,"shrink debug: %s\n",argv[arg]);
		  if(sscanf(argv[arg],"-shrink=%d:%d:%f",&atm_num1,&atm_num2,&lenNew) == 3){
			if(lenNew > 10) print_error(WUSAGE);
			fprintf(stderr,"-shrink=%d:%d:%.3f\n",atm_num1,atm_num2,lenNew);
		        c = 's'; silent= TRUE; break;
			// do nothing...
		  } else if(sscanf(argv[arg],"-sub=%d:%d:%c",&SubStart,&SubEnd,&chain) == 3){
			if(SubStart < 1 || SubStart > SubEnd) print_error(WUSAGE);
			// do nothing...
		  } else if(sscanf(argv[arg],"-s=%d%c:%d..%d",&shift,&chain,&shift_start,&shift_end) != 4){
		    shift_start=INT4_MIN; shift_end=INT4_MAX;
		    if(sscanf(argv[arg],"-s%d%c",&shift,&chain) != 2)
                        print_error(WUSAGE);
		  } else fprintf(stderr,"shift_start=%d; shift_end=%d\n",shift_start,shift_end);
		  if(shift_start > shift_end) print_error(WUSAGE);
		  if(!isalpha(chain)) print_error(WUSAGE);
		   c = 's'; silent= TRUE; 
		   break;
                case 'T': c = 'T'; silent = TRUE; break;
                case 't': c = 't'; silent = TRUE; 
			ReadCETransformFile(argv[arg]+2,xyz,WUSAGE);
			break;
		case 'u':
			if(strcmp(argv[arg],"-useMC")==0){ noMainChn=FALSE; } 
			else print_error(WUSAGE); break;
                case 'W': c = 'W'; silent = TRUE; break;
                case 'w': c = 'w'; silent = TRUE; 
		  if(isalpha(argv[arg][2])) chain=argv[arg][2];
		  else if(argv[arg][2] == '='){
			if(sscanf(argv[arg],"-w=%d..%d%c",&prnt_start,&prnt_end,&chain) != 3){
				print_error(WUSAGE);
			} else {
				if(prnt_start < 1 || prnt_start > prnt_end) print_error(WUSAGE);
			}
		  } else if(isdigit(argv[arg][2])) {
			if(sscanf(argv[arg],"-w%d%c",&water,&chain) != 2) print_error(WUSAGE);
		  } else print_error(WUSAGE); 
		  break;
                case 'x': c = 'x';  silent = TRUE;
		  if(argv[arg][2] == '='){
			if(sscanf(argv[arg],"-x=%d..%d%c",&del_start,&del_end,&chain) != 3){
				print_error(WUSAGE);
			} else {
				if(del_start < 1 || del_start > del_end) print_error(WUSAGE);
			}
		  } else print_error(WUSAGE);
		    break;
                default: print_error(WUSAGE);
            }
        }
	if(c=='k') silent=TRUE;
	if(!silent) {
          for(arg=0; arg < argc; arg++) fprintf(stderr,"%s ",argv[arg]); 
	  fprintf(stderr,"\n");
	}
	time_t	time1=time(NULL);
        if(seed == 18364592) seed = (unsigned int) (time(NULL)/2);
	if(!silent) fprintf(stderr,"seed = %d\n",seed);
	sRandom(seed);
	pdb = MkPDB(argv[1]);
#if 1	// temporary test for distance between two residues.
	if(nI > 0) {
	  double dd=GetResDistance(ofp, nI,nJ,cI,cJ,pdb,noMainChn);
	  if(cI != cJ) dd=GetResDistance(ofp,nI,nJ,cJ,cI,pdb,noMainChn);
	  return 0;
	}
#endif
	switch(c){
	   case 'm':	// twistpoint option
	    {
		pdb_typ	ToP = MkPDB(tofile);
		pdb_typ	FromP = MkPDB(fromfile);
	        // if(chain!='X') C=GetChainNumberPDB(pdb,chain); else C=1; 
	        C=GetChainNumberPDB(pdb,chain); 
		fprintf(stderr,"twistpoint = %d; chain = %d\n",res0,C);
		TriadTwistChnPDB(ofp,ToP,FromP,C,res0,pdb);
		NilPDB(ToP);
		NilPDB(FromP);
	    } break;
	   case 'M':
	    {
		pdb_typ	ToP = MkPDB(tofile);
		pdb_typ	FromP = MkPDB(fromfile);
		TriadSuperImposePDB(ofp,ToP,FromP,pdb);
		NilPDB(ToP);
		NilPDB(FromP);
	    } break;
	   case 'H':
	    if(dmax == 0.0) dmax = 3.6;
	    for(res0=MinResPDB(C1,pdb); res0 <= MaxResPDB(C1,pdb); res0++){
	      for(C=1; C <= nChainsPDB(pdb); C++){
		n=FindPotentialHBondsPDB(res0, C1, C, dmax, pdb);
		// if(n > 0) fprintf(ofp,"Chain %d/Res %d: n=%d\n",C1,res0,n);
	     }
	    }
	    break;
	   case 'P':	// non-aromatic H-pi interactions
	    {
	      // 
	      Int4 r,i,j,num_resA,num_resD;
	      if(dmax == 0.0) dmax = 3.6;
	      res_typ *ResA,*ResD=MakeResPDB(C1,&num_resD,pdb);
	      if(C != C1) ResA=MakeResPDB(C,&num_resA,pdb);
	      else { ResA=ResD; num_resA=num_resD; }
	      for(i=1; i <= num_resA; i++){
		// if(ResidueID(ResA[i]) == res0) {
		  // PrintResidueAtoms(ofp,ResA[i]);
	          for(j=1; j <= num_resD; j++){
		    FindOtherPiHbondsPDB(ResD[j],ResA[i],C1,C,dmax,pdb);
		  }
		// }
	      }
	      for(i=1; i <= num_resD; i++) NilRes(ResD[i]);
	      if(C != C1) for(i=1; i <= num_resA; i++) NilRes(ResA[i]);
	    }
	    break;
	   case 'p':	// aromatic H-pi interactions
	    {
	      // 
	      Int4 r,i,j,num_resA,num_resD;
	      if(dmax == 0.0) dmax = 4.5;
	      res_typ *ResA,*ResD=MakeResPDB(C1,&num_resD,pdb);
	      if(C != C1) ResA=MakeResPDB(C,&num_resA,pdb);
	      else { ResA=ResD; num_resA=num_resD; }
	      for(i=1; i <= num_resA; i++){
		// if(ResidueID(ResA[i]) == res0) {
		  // PrintResidueAtoms(ofp,ResA[i]);
	          for(j=1; j <= num_resD; j++){
		    FindAromaticHbondsPDB(ResD[j],ResA[i],C1,C,dmax,pdb);
		  }
		// }
	      }
	      for(i=1; i <= num_resD; i++) NilRes(ResD[i]);
	      if(C != C1) for(i=1; i <= num_resA; i++) NilRes(ResA[i]);
	    }
	    break;
	   case 'h':
	    {
	      // 
	      Int4 r,i,j,num_res;
	      // if(dmax == 0.0) dmax = 3.6;
	      if(dmax == 0.0) HA_dmax = 2.5;
	      res_typ *Res=MakeResPDB(C1,&num_res,pdb);
#if 1	// Test Kanti's routine
	      h_type tHG=Histogram("theta",0,5,0.2);
	      h_type dHG=Histogram("distance",0,5,0.05);
	      h_type dhaHG=Histogram("distHA",0,5,0.05);
	      h_type haHG=Histogram("angleDHA",0,5,0.05);
	      FILE *tfp=open_file("junk",".txt","w");
	      for(i=1; i <= num_res; i++){
// fprintf(stderr,"\n====================================\n");
// PutBondsInRes(stderr,Res[i]); fprintf(stderr,"\n");
	         for(j=1; j <= num_res; j++){
		    if(i == j) continue; 
		    char mode='C';
		    GetHBondThetaAndDistPDB(stderr,tfp,Res[i],C1,C1,HA_dmax,Res[j],
						1,pdb,mode,donor,accept,tHG,dHG,dhaHG,haHG);
	         }
	      } fclose(tfp);
// PutBondLengthHGs(stderr);
	      PutHist(ofp,60,dHG); NilHist(dHG); 
	      PutHist(ofp,60,haHG); NilHist(haHG); 
	      PutHist(ofp,60,tHG); NilHist(tHG); 
	      PutHist(ofp,60,dhaHG); NilHist(dhaHG); 
#else
	      for(i=1; i <= num_res; i++){
		//if(ResidueID(Res[i]) == res0) {
		   // PrintResidueAtoms(ofp,Res[i]);
		   // FindHBondsPDB(Res[i],C,C1,HA_dmax,pdb);
		   FindHBondsPDB(Res[i],C,C1,HA_dmax,pdb);
		// }
	      }
#endif
	      for(i=1; i <= num_res; i++) NilRes(Res[i]);
	    }
	    break;
	   case 'A':
	      if(dmax == 0.0) dmax = 3.6;
	      for(res0=MinResPDB(C1,pdb); res0 <= MaxResPDB(C1,pdb); res0++){
		n=ContactsResPDB(res0, C1, C2, dmax, pdb);
		if(n > 0) fprintf(ofp,"Chain %d/Res %d: n=%d\n",C1,res0,n);
	      } break;
	   case 'B': 
	     {
		Int4 Chn[101],i;
	        if(chains && chn_dist > 0.0){
		 for(i=0; chains[i] > 0; i++){
		  Chn[i]=GetChainNumberPDB(pdb,chains[i]); // C = query residue chain.
#if 0
	          // if(chain && chain!='X') C=GetChainNumberPDB(pdb,chain); 
		  BooLean	*AC=AdjacentSubunitsPDB(C,dmax,pdb);
		  PutAdjacentSubunitsPDB(ofp,AC,dmax,pdb); 
#else
		 }
		 Chn[i]=0;
		 if(i > 0) MultiPutAdjacentSubunitsPDB(ofp,Chn,chn_dist,pdb); 
#endif
		}
	     } break; 
	   case 'W': 
	        if(dmax == 0.0) dmax = 3.6;
		RmDistWaterPDB(ofp,dmax,pdb); 
		break; 
	   case 'C': 
		{
		   long natms=ChangeMSE2MET_PDB(pdb); 
		   fprintf(stderr,"%d MSE atoms changed to Met.\n",natms);
		   natms=ChangePO3toAtmPDB(pdb); 
		   fprintf(stderr,"%d phosphorylated amino acids changed to ATOM.\n",natms);
		   natms=HIDorHIEorHIP2HIS_PDB(pdb);
		   fprintf(stderr,"%d histidine heteroatoms changed to ATOM.\n",natms);
		   PutPDBwithTER(ofp,pdb); 
		}
		break; 
	   case 'b': 
	    {
	     // 	print_error("-b bondlength option not yet implemented.");
	     bnd_typ bnd=MkBondLengthHGs();
	     res_typ **Res=0; NEWP(Res,nChainsPDB(pdb) +4, res_typ);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		Int4 num_res=0;
		Res[C2]=MakeResPDB(C2,&num_res,pdb,bnd);
	     } PutBondLengthHGs(stdout,bnd);
	     NilBndLen(bnd);
	     // need to free up residues.
	    } break; 
	   case 'O': 
		PutChainPDB(ofp,resStart,resEnd,chain,pdb); 
		break; 
	   case 'o': 	// output non-protein chains
	    {
	     res_typ **ResALL;
	     Int4	i,*num_resALL;
	     NEWP(ResALL,nChainsPDB(pdb)+3,res_typ);
	     NEW(num_resALL,nChainsPDB(pdb)+3,Int4);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	        ResALL[C2] = MakeResPDB(C2,&num_resALL[C2],pdb);
		if(c == 'o' && !IsProteinPDB(C2,pdb)){
		   fprintf(ofp,"*************** 'Chain' %d (%d molecules) ***************\n",
					C2,num_resALL[C2]);
		   for(i=1; i <= num_resALL[C2]; i++){
		        fprintf(ofp,"=================== molecule %d ==================\n",i);
			PrintResidueAtoms(ofp,ResALL[C2][i]);
		   }
		}
	     }
	     // if(c == 'b') PutBondLengthHGs(ofp);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		for(i=1; i <= num_resALL[C2]; i++) NilRes(ResALL[C2][i]);
		free(ResALL[C2]);
	     } free(ResALL); free(num_resALL);
	    }
	      break;
	   case 'n':
		{
		   PutNotPrtnChainPDB(ofp,chain,pdb); 
		} break;
	   case 'S': 
		for(C=1; C <= nChainsPDB(pdb); C++){
		  if(IsProteinPDB(C,pdb)) PutPDBSeq(ofp, C, pdb);
		}
	   break; 
	   case 'R': 
	     {
		// if(chain_num==0) LabelNullChainsPDB(pdb,TRUE);
		if(AddTER==TRUE){ PutPDBwithTER(ofp,pdb); return 0; }
		if(renumber_start > 0){
		    RenumberChainPDB(chain, renumber_start, pdb);
		} else if(chain_num==0) LabelNullChainsPDB(pdb);
		else if(!ReNameChainPDB(chain_num, chain,pdb)) print_error(WUSAGE);
		PutPDB(ofp,pdb); 
	     } break;
	   case 'r': 
	     {
		char	*chn_str;
		// char	*DATADIR==getenv("CHN_RAS_DIR");
		NEW(chn_str,nChainsPDB(pdb)+5,char);
	     	for(C=0; C < nChainsPDB(pdb); C++){
		   chn_str[C]=ChainCharPDB(C+1,pdb);
	     	} 
		fprintf(stderr,"chains=%s\n",chn_str);
		char outfilename[100];
		strncpy(outfilename,argv[1],4);
		for(Int4 j=0; j< 4; j++){
			outfilename[j]=toupper(outfilename[j]);
		}
		strcat(outfilename,chn_str);
		strcat(outfilename,"H.pdb");
		fprintf(stderr,"%s\n",outfilename);
#if 1
		// FILE *tfp=open_file(DATADIR,outfilename,"w");
		FILE *tfp=open_file(outfilename,"","w");
		PutPDB(tfp,pdb); 
		fprintf(ofp,"File0=XXX/%s:A  // \n",outfilename);
		fclose(tfp);
#endif
	     } break;
	   case 's': 
		fprintf(stderr,"debug -shrink=%d:%d:%.3f\n",atm_num1,atm_num2,lenNew);
		if(atm_num1 > 0 && atm_num2 > 0){	// -shrink option
#if 1
		   atm_typ fixed=AtomPDB(atm_num1,pdb);
		   atm_typ move=AtomPDB(atm_num2,pdb);
		   ShortenBond(fixed,move,lenNew);
		   exit(1);
#else
		  print_error("not yet implemented");
#endif
		} else if(SubEnd > 0){
			PutSubChainPDB(ofp, SubStart, SubEnd, chain,pdb);
		} else if(ShiftResChainPDB(chain,shift,shift_start,shift_end,pdb)) PutPDB(ofp,pdb);
		else fprintf(stderr,"Subunit ' ' not found!\n",chain);
	   break; 
	   case 'T': 
	     {
		Int4	*triad[5],a,i;
		n=TriadPDB(C1, pdb, triad);
		for(i=1; i <= n; i++){
		   if(triad[0][i] != 0){
		     for(a=1; a < 5; a++){
			PutAtom(ofp,AtomPDB(C1, triad[a][i], pdb));
		     }
		     printf("\n");
		   } 
		}
		if(n>0) for(a=0; a < 5; a++) free(triad[a]);
		// fprintf(ofp,"n=%d\n",n);
	     }
	   break; 
	   case 't': 
		TransformPDB(xyz,pdb);
		PutPDB(ofp,pdb);
	   break; 
	   case 'N': { if(dmax == 0.0) dmax = 4.5; ContactsAtomPDB(atom_no,dmax,pdb); } break; 
	   case 'Q': { 
		if(dmax == 0.0) dmax = 3.0;
		Int4 hits=FindQuadWatersPDB(dmax,pdb);
		if(hits > 0) fprintf(ofp,"total hits: %d\n\n",hits);
	     } break; 
	   case 'a': 
	     if(angles){
	        if(HA_dmax == 0.0) HA_dmax = 2.5;
	    	if(dmax == 0.0) dmax = 3.6;
		resStart=MinResPDB(C,pdb);
		resEnd=MaxResPDB(C,pdb);
	    	PutAngularPDB(ofp,file_id,resStart,resEnd,HA_dmax,dmax,C,C2,chain,0,0,pdb);
		
	     } else {
		atm_typ atom,atm[5];
		Int4 a;
		atm[1]=atm[2]=atm[3]=0;
		if(C > 0 && C <= nChainsPDB(pdb)){
		  for(n=0,a=1; a <= MaxAtomsPDB(C,pdb); a++){
			atom = GetAtomPDB(pdb,C,a);
			if(AtomID(atom) == atm_no[1]) atm[1]=atom;
			if(AtomID(atom) == atm_no[2]) atm[2]=atom;
			if(AtomID(atom) == atm_no[3]) atm[3]=atom;
		  }
		  if(atm[1] && atm[2] && atm[3]){
			PutAtom(ofp,atm[1]);
			PutAtom(ofp,atm[2]);
			PutAtom(ofp,atm[3]);
		  	double  angle = CalcAngle(atm[1],atm[2],atm[3]);
		  	fprintf(ofp,"Angle between atoms %d, %d, %d = %.2f\n",
				atm_no[1],atm_no[2],atm_no[3],angle);
		  } else fprintf(ofp,"Some atoms not found!\n");
		}
	     }
	   break; 
	   case 'D': 
	    {
	     Int4 r,i,j;
	     a_type AB = AminoAcidAlphabetPDB(pdb);
	     pdb_typ	ToP = MkPDB(tofile);

	     if(dmax == 0.0) dmax = 4.5;
	     // if(chain && chain!='X') C=GetChainNumberPDB(pdb,chain); 
	     if(chain) C=GetChainNumberPDB(pdb,chain); // C = query residue chain.
	     if(C==0){ fprintf(stderr,"chain = %c\n",chain); print_error("chain not found!"); }
	     Int4 CB;
	     // if(chainB != 'X') CB=GetChainNumberPDB(ToP,chainB); else CB=1;
	     CB=GetChainNumberPDB(ToP,chainB);

#if 1		
	     // 1. Check to make sure both pdb files have essentially the same sequence.
	     e_type E1 = GetPDBSeq(C,pdb);
	     e_type E2 = GetPDBSeq(CB,ToP);
	     Int4	min_leng = MINIMUM(Int4,LenSeq(E1),LenSeq(E2));
	     // Int4 score = AlnSeqSW(12, 4, E1, E2, AB);
	     Int4 selfscore = PutDiagonalSeq(ofp, 0, E1, E1, AB);	// 0 == main diagonal...
	     Int4 score = PutDiagonalSeq(ofp, 0, E1, E2, AB);
	     double ratio = (double)score/(double)selfscore;
	     printf("selfscore = %d; score = %d; ratio = %g\nlength seq1 = %d; length seq2 = %d (min = %d)\n",
			     	selfscore,score,ratio,LenSeq(E1),LenSeq(E2),min_leng);
	     // assert(LenSeq(E1) == num_resC);
	     if(!IdentSeqs(E1,E2)){
		     //print_error("chains not identical!");
		     if(ratio < 0.90) print_error("chains are too different!");
#if 0
		     if(LenSeq(E1) != LenSeq(E2)) print_error("chains are different lengths!");
		     if(ResSeq(1,E1) != ResSeq(1,E2)) print_error("chains don't start with the same residue!");
		     if(ResSeq(LenSeq(E1),E1) != ResSeq(LenSeq(E2),E2)) 
			     print_error("chains don't end with the same residue!");
#endif
	     }

#endif

	     // 2. make residues structures for chain of pdb and chainB of ToP.
	     Int4	num_resC;
	     res_typ	*ResC = MakeResPDB(C,&num_resC,pdb);
	     Int4	num_resCB;
	     res_typ	*ResCB = MakeResPDB(CB,&num_resCB,ToP);

	     printf("num_resC = %d; C = %d; CB = %d; num_resCB = %d\n",num_resC,C,CB,num_resCB);

	     // 3. find ...
	     double buried;
	     Int4 NumInC[MAX_NGRPS_PDB],NumInCB[MAX_NGRPS_PDB];
	     assert(nChainsPDB(pdb) < MAX_NGRPS_PDB); 
	     assert(nChainsPDB(ToP) < MAX_NGRPS_PDB); 
	     for(C2=1; C2 < MAX_NGRPS_PDB; C2++) NumInC[C2]=NumInCB[C2]=0;
	     NumInC[C]=MaxAtomsPDB(C,pdb);  
	     NumInCB[CB]=MaxAtomsPDB(CB,ToP);  
	     dh_type dH = dheap(num_resC+3,4);
	     Int4 j1=0,j2=0;
	     for(j1=j2=j=1; j1 <= num_resC && j2 <= num_resCB; j++,j1++,j2++){
		if(ResidueID(ResC[j1]) == ResidueID(ResCB[j2]) 
				&& GetCharResidue(ResC[j1],AB) == GetCharResidue(ResCB[j2],AB)
				&& AtomsInResidue(ResC[j1]) == AtomsInResidue(ResCB[j2])){
			double buried1=CalculateResidueSurface(ResC[j1],nChainsPDB(pdb),NumInC,AtomsPDB(pdb));
			double buried2=CalculateResidueSurface(ResCB[j2],nChainsPDB(ToP),NumInCB,AtomsPDB(ToP));
			buried = buried1-buried2;
			// buried = buried2-buried1;
			if(buried > 0.0) insrtHeap(j1,(-(keytyp)buried),dH);
		} else {
		  fprintf(ofp,"ResidueID(ResC[%d]) = %d (%d;%d) --> %c (%d atoms)\n",
				  j1,ResidueID(ResC[j1]),ResAtom(AtomResidue(1,ResC[j1])),
				  ResidueChain(ResC[j1]),GetCharResidue(ResC[j1],AB),
				  AtomsInResidue(ResC[j1])); 
		  fprintf(ofp,"ResidueID(ResCB[%d]) = %d (%d;%d) --> %c (%d atoms)\n",
				  j2,ResidueID(ResCB[j2]),ResAtom(AtomResidue(1,ResCB[j2])),
				  ResidueChain(ResCB[j2]),GetCharResidue(ResCB[j2],AB),
				  AtomsInResidue(ResCB[j2])); 
		  if(ResidueID(ResC[j1]) < ResidueID(ResCB[j2])) ;
		}
	     }
	     while((buried = (double) -minkeyHeap(dH)) >= 0.0 && (j=delminHeap(dH)) != 0){
			fprintf(ofp,"#%c%d%c.W // %.2f buried\n",GetCharResidue(ResC[j],AB),
					ResidueID(ResC[j]),chain,buried);
	     } Nildheap(dH);

#if 0
	     for(i=1; i <= num_resC; i++) NilRes(ResC[i]);
	     for(i=1; i <= num_resCB; i++) NilRes(ResCB[i]);
#endif
	    } break; 
	   case 'e': 
	    {
	     Int4 r,i,j,num_resA,num_resD;
	     if(dmax == 0.0) dmax = 4.5;
	     if(chain) C=GetChainNumberPDB(pdb,chain); // C = query residue chain.
	     if(C==0){ 
		     fprintf(stderr,"chain = %c\n",chain);
		     print_error("chain not found!");
	     }
	     // Int4 Start,End;
	     res_typ *ResCD,**ResALL;
	     Int4	num_resC,num_resO,*num_resALL;
	     NEWP(ResALL,nChainsPDB(pdb)+3,res_typ);
	     NEW(num_resALL,nChainsPDB(pdb)+3,Int4);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	       ResALL[C2] = MakeResPDB(C2,&num_resALL[C2],pdb);
	     }
	     num_resC=num_resALL[C]; ResCD=ResALL[C];
	     if(this_res){
	      fprintf(ofp,"*********** Surface (%d%c): ***********\n",res0,chain);
	      fprintf(ofp,"Res   pos   exposed  buried   total   ratio\n");
	      a_type AB = AminoAcidAlphabetPDB(pdb);
	      double d,d_ave=0;
	      Int4 Nd=0;
	      for(j=1; j <= num_resC; j++){
		char res;
		if(GetCharResidue(ResCD[j],AB) == this_res) {
			// PrintResidueSurface(ofp,ResCD[j]);
			d=CalculateResidueSurface(ResCD[j],nChainsPDB(pdb),
				NumAtomsPDB(pdb), AtomsPDB(pdb));
			PrintResidueSurface(ofp,ResCD[j]);
			d_ave+=d; Nd++;
		}
	      }
	      fprintf(ofp,"Ave = %.2f\n",d_ave/(double)Nd);
	     } else {
	      fprintf(ofp,"*********** Surface (%d%c): ***********\n",res0,chain);
	      fprintf(ofp,"Res   pos   exposed  buried   total   ratio\n");
	      for(j=1; j <= num_resC; j++){
		if(ResidueID(ResCD[j]) == res0 || res0 == 0) {
			// PrintResidueSurface(ofp,ResCD[j]);
			CalculateResidueSurface(ResCD[j],nChainsPDB(pdb),NumAtomsPDB(pdb),
					AtomsPDB(pdb));
			PrintResidueSurface(ofp,ResCD[j]);
		}
	      }
	     }
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		for(i=1; i <= num_resALL[C2]; i++) NilRes(ResALL[C2][i]);
		free(ResALL[C2]);
	     } free(ResALL); free(num_resALL);
	    } break; 
	   case 'E': 
	    {
	     Int4 r,i,j,num_resA,num_resD;
	     if(dmax == 0.0) dmax = 4.5;
	     res_typ *ResCD,**ResALL;
	     Int4	num_resC,num_resO,*num_resALL;
	     NEWP(ResALL,nChainsPDB(pdb)+3,res_typ);
	     NEW(num_resALL,nChainsPDB(pdb)+3,Int4);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	       ResALL[C2] = MakeResPDB(C2,&num_resALL[C2],pdb);
	     }
	     if(this_res){
	      Int4 Nd=0;
	      double d,d_ave=0;
	      a_type AB = AminoAcidAlphabetPDB(pdb);
	      h_type HG=Histogram("Exposed surface area", 0,500,25.0);
	      for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	        num_resC=num_resALL[C2];
	        fprintf(ofp,"*********** Surface (%d%c): ***********\n",res0,chain);
	        fprintf(ofp,"Res   pos   exposed  buried   total   ratio\n");
	        for(j=1; j <= num_resC; j++){
		  char res;
		  if(GetCharResidue(ResALL[C2][j],AB) == this_res) {
			// PrintResidueSurface(ofp,ResALL[C2][j]);
			d=CalculateResidueSurface(ResALL[C2][j],nChainsPDB(pdb),
				NumAtomsPDB(pdb), AtomsPDB(pdb));
			IncdHist(d,HG);
			PrintResidueSurface(ofp,ResALL[C2][j]);
			d_ave+=d; Nd++;
		  }
	        }
	      }
	      PutHist(ofp,60,HG); NilHist(HG);
	      fprintf(ofp,"Ave = %.2f\n",d_ave/(double)Nd);
	     } else { 
	        if(chain) C=GetChainNumberPDB(pdb,chain); // C = query residue chain.
	        if(C==0){ 
		     fprintf(stderr,"chain = %c\n",chain);
		     print_error("chain not found!");
	        }
	        num_resC=num_resALL[C]; ResCD=ResALL[C];
	        Int4 CB=GetChainNumberPDB(pdb,chainB); // C = query residue chain.
		Int4 numCB=MaxAtomsPDB(CB,pdb);
		Int4 NumInChain[MAX_NGRPS_PDB];
		double buried;
		assert(nChainsPDB(pdb) < MAX_NGRPS_PDB); 
		for(C2=1; C2 <= nChainsPDB(pdb); C2++) NumInChain[C2]=MaxAtomsPDB(C2,pdb);
		NumInChain[CB]=0;  // compute without this subunit.
		dh_type dH = dheap(num_resC+3,4);
	        for(j=1; j <= num_resC; j++){
			double free=CalculateResidueSurface(ResCD[j],nChainsPDB(pdb),NumInChain,
					AtomsPDB(pdb));
			double bound=CalculateResidueSurface(ResCD[j],nChainsPDB(pdb),NumAtomsPDB(pdb),
					AtomsPDB(pdb));
			buried = free-bound;
			if(buried > 0.0) insrtHeap(j,(-(keytyp)buried),dH);
#if 0
			fprintf(ofp,"%s%d%c: %.2f buried\n",ResidueName(ResCD[j]),
						ResidueID(ResCD[j]),chain,buried);
			// PrintResidueSurface(ofp,ResCD[j]);
#endif
		}
		a_type AB = AminoAcidAlphabetPDB(pdb);
		double TotBuried=0;
		while((buried = (double) -minkeyHeap(dH)) >= 0.0 && (j=delminHeap(dH)) != 0){
			TotBuried+=buried;
#if 0
			fprintf(ofp,"%s%d%c: %.2f buried\n",ResidueName(ResCD[j]),
					ResidueID(ResCD[j]),chain,buried);
#else
			fprintf(ofp,"#%c%d%c.W // %.2f buried\n",GetCharResidue(ResCD[j],AB),
					ResidueID(ResCD[j]),chain,buried);
#endif
		} Nildheap(dH);
		fprintf(ofp,"// total buried = %.2lf\n",TotBuried);
	     }
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		for(i=1; i <= num_resALL[C2]; i++) NilRes(ResALL[C2][i]);
		free(ResALL[C2]);
	     } free(ResALL); free(num_resALL);
	    } break; 
	   case 'f': 	// 
	    {
		fprintf(stderr,"cmafile=%s; cutoff = %d; chains = %s\n",
				cmafilename,cmacutoff,cma_chns);
		// if(cma_chns[0]=='X') cma_chns[0]=0; // treat 'X' chains the same
	        // open cma file.
	        if(dmax == 0.0) dmax = 3.6;
	        if(HA_dmax == 0.0) HA_dmax = 2.5;
		a_type AB = AminoAcidAlphabetPDB(pdb);
		cma_typ cma = ReadCMSA2(cmafilename,AB);
		char *skip=Foo(cma, cmacutoff, cma_chns,&resStart,&resEnd, pdb);
		chain = cma_chns[0];
		char color=1;
		PutContactsPDB(ofp,file_id,resStart,resEnd,HA_dmax,dmax,C,C2,chain,skip,color,pdb);
		free(skip); TotalNilCMSA(cma);
	    } break; 
	   case 'K': 
	    {
	     Int4 r,i,j,num_resA,num_resD;
	     if(dmax == 0.0) dmax = 4.5;
	     if(HA_dmax == 0.0) HA_dmax = 2.5;
	     if(chain) C=GetChainNumberPDB(pdb,chain); // C = query residue chain.
	     if(C==0){ 
		     fprintf(stderr,"chain = %c\n",chain);
		     print_error("chain not found!");
	     }
	     // Int4 Start,End;
	     res_typ *ResCA,*ResOA,*ResCD,*ResOD,**ResALL;
	     Int4	num_resC,num_resO,*num_resALL;
	     NEWP(ResALL,nChainsPDB(pdb)+3,res_typ);
	     NEW(num_resALL,nChainsPDB(pdb)+3,Int4);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	      ResALL[C2] = MakeResPDB(C2,&num_resALL[C2],pdb);
	     }
	    for(Int4 k=MinResPDB(C,pdb); k <= MaxResPDB(C,pdb); k++){
	     res0=k;
#if 0
	     fprintf(ofp,"*********** Contacts: ***********\n");
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	      n=ContactsResPDB(res0, C, C2, dmax, pdb);
	     }
#endif
	     num_resC=num_resALL[C]; ResCA=ResCD=ResALL[C];
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	      ResOD=ResOA=ResALL[C2];	// other than query residue.
	      num_resO=num_resALL[C2];
	      
	      fprintf(ofp,"*********** H-bond donor (%d%c): ***********\n",res0,chain);
	      for(j=1; j <= num_resC; j++){
		if(ResidueID(ResCD[j]) == res0) FindHBondsPDB(ResCD[j],C,C2,HA_dmax,pdb);
	      }
	      fprintf(ofp,"*********** H-bond acceptor (%d%c): ***********\n",res0,chain);
	      for(j=1; j <= num_resO; j++){
		FindHBondsPDB(ResOD[j],C2,C,HA_dmax,res0,pdb); // res0 is acceptor.
	      }

	      fprintf(ofp,"*********** pi-bonds: ***********\n");
	      for(i=1; i <= num_resC; i++){
		if(ResidueID(ResCD[i]) == res0){	// --> ResCA[i] == res0
	         for(j=1; j <= num_resO; j++){
		  // PrintResidueAtoms(ofp,ResA[i]);
		  FindAromaticHbondsPDB(ResCD[i],ResOA[j],C,C2,dmax,pdb);
		  FindAromaticHbondsPDB(ResOD[j],ResCA[i],C2,C,dmax,pdb);
		  FindOtherPiHbondsPDB(ResCD[i],ResOA[j],C,C2,dmax,pdb);
		  FindOtherPiHbondsPDB(ResOD[j],ResCA[i],C2,C,dmax,pdb);
		 }
		}
	      }
	     }
	    }
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		for(i=1; i <= num_resALL[C2]; i++) NilRes(ResALL[C2][i]);
		free(ResALL[C2]);
	     } free(ResALL); free(num_resALL);
	    } break; 
	   case 'l':	// list chain components
	    {
	        if(dmax == 0.0) dmax = 3.6;
		this->ListChainsPDB(stdout,pdb);
	    } break;
	   case 'L':	// label chains automatically
	   {
#if 0
	    BooLean *Used;
	    NEW(Used,265,char);
	    char ChainLabels[]="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	    for(C=1; C <= nChainsPDB(pdb); C++){
		char chain=ChainCharPDB(C,pdb);
		if(chain != ' ' && Used[chain]){
		  fprintf(stderr,"duplicate chain %d: '%c'\n",C,chain);
		  chain = ' ';
	    	  if(!ReNameChainPDB(C,chain,pdb)) print_error(WUSAGE);
	 	  char chain=ChainCharPDB(C,pdb);
		  fprintf(stderr,"changed %d to: '%c'\n",C,chain);
		} else if(chain != ' '){
		  fprintf(stderr,"chain %d: '%c'\n",C,chain);
		  Used[chain]=TRUE;
		} 
	    }
	    for(C=1; C <= nChainsPDB(pdb); C++){
		char chain=ChainCharPDB(C,pdb);
		if(chain == ' '){
		  Int4 j=0;
		  do {
		    chain=ChainLabels[j]; j++;
		  } while(chain != 0 && Used[chain]);
		  if(chain == 0) print_error("FATAL: too many chains");
		  fprintf(stderr,"chain %d: ' ' --> '%c'\n",C,chain);
		  if(!ReNameChainPDB(C,chain,pdb)) print_error(WUSAGE);
		  Used[chain]=TRUE;
		}
	    } free(Used);
#else
	    ReLabelPDB(pdb);
#endif
	    PutPDB(ofp,pdb); 
	   } break; 
	   case 'k': 
	    if(HA_dmax == 0.0) HA_dmax = 2.5;
	    if(dmax == 0.0) dmax = 3.6;
// fprintf(stderr,"C=%d; C2=%d; chain =%d\n",C,C2,chain);
	    if(FullOutput) PutContactsPDB(ofp,file_id,resStart,resEnd,HA_dmax,dmax,C,C2,chain,0,0,pdb);
	    else PutContactsPDB(ofp,file_id,resStart,resEnd,HA_dmax,dmax,C,C2,chain,0,0,pdb,'X');
	    break; 
	   case 'w': 
	if(prnt_start > 0){
	   PutChainPDB(ofp, prnt_start, prnt_end, chain, pdb);
	} else if(water < 0){
	   PutPrtnChainPDB(ofp,chain,pdb); 
	} else {
	     Int4 r,i,j,num_resA,num_resD;
	     if(dmax == 0.0) dmax = 4.5;
	     if(HA_dmax == 0.0) HA_dmax = 2.5;
	     if(chain) C=GetChainNumberPDB(pdb,chain); // C = query residue chain.
	     if(C==0){ 
		     fprintf(stderr,"chain = %c\n",chain);
		     print_error("chain not found!");
	     }
	     // Int4 Start,End;
	     res_typ *ResCA,*ResOA,*ResCD,*ResOD,**ResALL;
	     Int4	num_resC,num_resO,*num_resALL;
	     NEWP(ResALL,nChainsPDB(pdb)+3,res_typ);
	     NEW(num_resALL,nChainsPDB(pdb)+3,Int4);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	      ResALL[C2] = MakeResPDB(C2,&num_resALL[C2],pdb);
	     }
	     num_resC=num_resALL[C]; ResCA=ResCD=ResALL[C];
	     fprintf(ofp,"*********** Water H-bond donor (%d%c): ***********\n",
			water,chain);
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
	      for(j=1; j <= num_resC; j++){
		FindWaterHBondsPDB(ofp,ResCD[j],C,C2,HA_dmax,0,0,pdb,water);
	      }
	     }
	     fprintf(ofp,"*********** Water H-bond acceptor (%d%c): ***********\n",
				water,chain);
	      for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
// fprintf(stderr,"C2=%d; min=%d; max=%d\n",C2,MinResPDB(C2,pdb),MaxResPDB(C2,pdb));
	         ResOD=ResOA=ResALL[C2];	// other than query residue.
	         num_resO=num_resALL[C2];
// fprintf(stderr,"k=%d; num_resO=%d; %d\n",k,num_resO,pdb->maxres[C]);
	         for(j=1; j <= num_resO; j++){
		  FindWaterHBondsPDB(ofp,ResOD[j],C2,C,HA_dmax,0,0,pdb,water);
		  // FindWaterHBondsPDB(ofp,ResOD[j],C2,C,HA_dmax,water,0,pdb,water);
		  // FindHBondsPDB(ResOD[j],C2,C,HA_dmax,res0,pdb); // res0 is acceptor.
		 }
	      }
	     for(C2=1; C2 <= nChainsPDB(pdb); C2++){ 
		for(i=1; i <= num_resALL[C2]; i++) NilRes(ResALL[C2][i]);
		free(ResALL[C2]);
	     } free(ResALL); free(num_resALL);
	}
		break; 
	   case 'x': 
/********************
       -w=<int>..<int><char> Output residues <int1> to <int2> in chain <char>
       -x=<int>..<int><char> Output all of chain <char> except for residues <int1> to <int2>
/********************/
	    if(del_start > 0){
		// print_error("Not yet implemented");
		Int4 chn = GetChainNumberPDB(pdb, chain);
	        if(del_start  <= MinResPDB(chn,pdb)){
		   if(del_end < MaxResPDB(chn,pdb)){
	   		PutChainPDB(ofp, del_end+1, MaxResPDB(chn,pdb), chain, pdb);
		   }
		} else {	// del_start > MinResPDB(chain,pdb)
	   	   PutChainPDB(ofp, MinResPDB(chn,pdb), del_start-1, chain, pdb);
		   if(del_end < MaxResPDB(chn,pdb)){
	   		PutChainPDB(ofp, del_end+1, MaxResPDB(chn,pdb), chain, pdb);
		   } else {
			// fprintf(stderr,"del_end = %d; MaxResPDB(%c,pdb) = %d\n",del_end,chain,MaxResPDB(chn,pdb));
		   }
		}
	    }
	    break; 
	   default:
	     if(dmax == 0.0) dmax = 3.6;
	     this->ListChainsPDB(stdout,pdb);
	   break; 
	}
	// PutPDB(ofp,pdb);
	NilPDB(pdb);
	if(!silent){
	  double runtime=difftime(time(NULL),time1);
	  fprintf(stderr,"\ttime: %0.1f seconds (%0.2f minutes)\n",runtime,runtime/60.0);
	}
	return 0;
}

