/******************************************************************************************
    Copyright (C) 1997-2019 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "spc_typ.h"

// # using namespace std;

double spc_typ::tri_typ::AddValue(Int4 n, double dd, Int4 i, char typ)
{
	double d;
	assert(i < nChn && i >= 0);
	Int4 j=n%window;
	Sum[i] -= array[i][j]; array[i][j] = dd;
	Sum[i] += dd;
	if(i < window) d=dd; else d=Sum[i]/(double)window;
	double cutLow=2.5,cutMid=4.0;

	if(typ){
           switch (typ){
             case 'T': cutLow=6.5; cutMid=7.2; break;
             case 'D': cutLow=4.1; cutMid=5.5; break;
             case 'F': cutLow=3.8; cutMid=4.5; break;
             case 'I': cutLow=4.5; cutMid=7.5; break;
	     case 'C': case 'N': case 'O': case 'S':
		break;
	     default: fprintf(stderr,"typ = '%c'\n",typ);
		print_error("tri_typ::AddValue() input error");
		break;
           } 
	   if(nType[typ]==0) NEW(nType[typ],nChn +3,Int4);
	   nType[typ][i]++; nType[0][i]++;
	}
	
	if(d < cutLow){	
	   tot[i][0]++;
	   if(0) fprintf(stderr,"%d.%d: totalHB=%d; Sum=%.3lf\n",
				i,n,tot[i][0],Sum[i]);
	   if(State[i]=='h'){	// H-bond or pi interaction
		run[i][0]++;	// how long of a run is this.
		if(run[i][0] > max[i][0]) max[i][0]=run[i][0];
	   } else { run[i][0]=1; State[i]='h'; }
	} else if(d < cutMid){
	   tot[i][1]++;
	   if(State[i]=='i'){	// intermediate distance
		run[i][1]++;
		if(run[i][1] > max[i][1]) max[i][1]=run[i][1];
   	   } else { run[i][1]=1; State[i]='i'; }
	} else {	// d >= 4.0
	   tot[i][2]++;
	   if(State[i]=='d'){	// distant pair
		run[i][2]++;
		if(run[i][2] > max[i][2]) max[i][2]=run[i][2];
	   } else { run[i][2]=1; State[i]='d'; }
	}
}

char	**spc_typ::tri_typ::RtnResults(keytyp &key,char *Msg,char *cmd,
			BooLean trans,double **value,char mode,char **typ)
/**************************************************************************
      types: 
	T = aromatic-aromatic T-shaped.
	F = aromatic-aromatic face-to-face.
	O = aromatic-aromatic parallel offset.
      modes:
	P = CH-pi interaction
	F = Flipped interaction
	
 **************************************************************************/
{
	char	**Rtn=0,c=0,X='C',s[40],str[150]; str[0]=0;
	Int4	i,j,x,indx=0;
	double	dd;

	if(trans) X = 'T';
	if(strchr("FSBPA",mode) == NULL) 
		print_error("tri_typ::RtnResults() input error 1");
	if(mode=='P'){	// CH-pi interactions
	   BooLean okay=FALSE;
	   for(i=0; i < nChn; i++){	// at least one chain where > 50% conserved...
		Int4 sum=tot[i][0]+tot[i][1]+tot[i][2];
		dd=(double)tot[i][0]/(double)sum;
// fprintf(stderr,"%c: --> dd = %d/%d = %.2lf\n",mode,nType[0][i],nPDB,dd);
		if(dd > 0.50){ okay=TRUE; break; }
	   } if(!okay) return 0;
	}
	if(mode=='S' && typ != 0){
		Int4	nFailed=0;
		for(i=0; i < nChn; i++){
		  dd=(double)nType[0][i]/(double)nPDB;
// fprintf(stderr,"%c: --> dd = %d/%d = %.2lf\n",mode,nType[0][i],nPDB,dd);
		  if(dd < 0.50) nFailed++;
		} if(nFailed == nChn) return 0;
	}

	keytyp	maxkey=0.0,k;
	if(mode == 'B') maxkey=-9999999.0;
//fprintf(stderr,"key=%lf; mode=%c\ncmd=%s",key,mode,cmd);
	for(indx=i=0; i < nChn; i++){
	   if(mode=='F'){
	       if(typ==0  && !(max[i][0] > cutH && max[i][2] > cutD)) continue;
	       else if(typ != 0) { print_error("not yet implemented");
	       }
	   }
	   if(mode=='B'){
	      if(!(max[i][0] > 10 && max[i][2] > 10 && 
		   (value[0][i] <= 2.5 || value[1][i] <= 2.5 ||
			value[2][i] <= 2.5 || value[3][i] <= 2.5))) continue; 
	   }
	   if(mode=='S'){
		if(typ==0 && value[3][i] > 2.5) continue;
		else if(typ != 0){ // aromatic-aromatic interactions.
		  if(typ[0]==0) print_error("tri_typ::RtnResults() input error 2");
#if 0
		  if(typ[0][i] != typ[1][i] || typ[0][i] != typ[2][i]) continue;
#endif
		  char t=typ[0][i];
		  if(t == 'T' && value[3][i] > 6.5) continue; 
		  if(t == 'D' && value[3][i] > 5.5) continue; 
		  if(t == 'F' && value[3][i] > 4.5) continue; 
		  if(t == 'I') continue; 
		  dd=(double)nType[0][i]/(double)nPDB;
		  if(dd < 0.50) continue;
		  if(strchr("TDFI",t) == NULL && strchr("CNOS",t) == NULL)
		       print_error("tri_typ::RtnResults() input error 3");
		}
	   }
	   if(mode=='P'){
		if(nType[0] != 0){
#if 0
		  if(typ==0 && value[3][i] > 2.5) continue;
		  if(typ[0][i] != typ[1][i] || typ[0][i] != typ[2][i]) continue;
		  if(value[3][i] > 2.5) continue;
#endif
		  dd=(double)nType[0][i]/(double)nPDB;
		  if(dd < 0.50) continue;
		  char t=typ[0][i];
		  if(strchr("CNOS",t) == NULL)
		       print_error("tri_typ::RtnResults() input error 4");
		} else if(value[3][i] > 2.5) continue;
	   }

	   if(mode=='F') k=(keytyp)(max[i][0]+ max[i][2]);
	   else if(mode=='B') k=-(keytyp)abs(tot[i][0]- tot[i][2]);
	   else k=(keytyp) value[3][i];	// == average value
	   if(k > maxkey) maxkey=k;
	   if(c==0 && Msg != 0){
		 c='x'; 
		 if(Rtn==0) NEWP(Rtn,nChn*20, char);
		 sprintf(str,"(%c): %s",X,Msg);
// fprintf(stderr,"%d%s",i,str);
		 Rtn[indx]=AllocString(str); indx++;

		 sprintf(str,"Chn:");
		 for(j=0; chain[j]; j++){
		     if(trans){
			x=(j+1)%nChn; sprintf(s,"\t%c%c",chain[j],chain[x]); 
		     } else {
			sprintf(s,"\t%c",chain[j]); 
		     } strcat(str,s);
		 } sprintf(s,"\n"); strcat(str,s);
		 Rtn[indx]=AllocString(str); indx++;
		 for(j=0; j <=3; j++){
		    if(j==0) sprintf(str,"Fst:");
		    else if(j==1) sprintf(str,"Mid:");
		    else if(j==2) sprintf(str,"Lst:");
		    else sprintf(str,"Avg:");
		    // for(Int4 z=0; value[j][z] > 0.0; z++)
		    for(Int4 z=0; chain[z]; z++)
		    {
// if(j==3) fprintf(stderr,"%d=%.2lf\n",z,value[j][z]);
			if(value[j][z] < big_dist) sprintf(s,"\t%.2lf",value[j][z]); 
			else sprintf(s,"\t-");
			strcat(str,s);
// if(j==3)fprintf(stderr,"%s\n",str);
		    } sprintf(s,"\n"); strcat(str,s);
		    Rtn[indx]=AllocString(str); indx++;
		 }
		 if(typ){
		   for(j=0; chain[j]; j++){
		    if(nType[0][j] > 0){
		      if(trans){ x=(j+1)%nChn; sprintf(s,"%c%c",chain[j],chain[x]); }
		      else { sprintf(s,"%c",chain[j]); } 
		      Int4 x,y,z;
		      
#if 1
		      char cc,consTyp=' '; x=0;
		      dh_type dH=dheap(252,4);
		      for(cc='A'; cc <= 'z'; cc++){
			  if(nType[cc] == 0) continue;
			  if((y=nType[cc][j]) > x){ x=y; consTyp=cc;}
			  if(y > 0) insrtHeap((Int4)cc,-(keytyp)y,dH);
		      }
		      sprintf(str,"%s(%c): ",s,consTyp);
#if 0
		      for(char cc='A'; cc <= 'z'; cc++){
			  if(nType[cc] == 0) continue;
		          if(nType[cc][j] > 0){
			     if(strchr("CNOS",cc)) sprintf(s,"-%cH=%d ",cc,nType[cc][j]);
			     else sprintf(s,"%c=%d ",cc,nType[cc][j]);
			     strcat(str,s);
			  }
		      }
#else
		      while(!emptyHeap(dH)){
			  assert((z=delminHeap(dH)) != 0);
			  cc=(char)z;
			  if(strchr("CNOS",cc)) sprintf(s,"-%cH=%d ",cc,nType[cc][j]);
			  else sprintf(s,"%c=%d ",cc,nType[cc][j]);
			  strcat(str,s);
		      } Nildheap(dH);
#endif
#else
		      char consTyp='T';
		      x=nType['T'][j]; 
		      y =nType['D'][j]; if(x < y){ x=y; consTyp='D';}
		      y =nType['F'][j]; if(x < y){ x=y; consTyp='F';}
		      y =nType['I'][j]; if(x < y){ x=y; consTyp='I';}
		      if(x==0) consTyp=' ';
		      sprintf(str,"%s(%c): ",s,consTyp);
		      if(nType['T'][j] > 0){ sprintf(s,"T=%d ",nType['T'][j]); 
				strcat(str,s);}
		      if(nType['D'][j] > 0){ sprintf(s,"S=%d ",nType['D'][j]); 
				strcat(str,s);}
		      if(nType['F'][j] > 0){ sprintf(s,"F=%d ",nType['F'][j]); 
				strcat(str,s);}
		      if(nType['I'][j] > 0){ sprintf(s,"I=%d ",nType['I'][j]); 
				strcat(str,s);}
#endif
		      sprintf(s,"\n"); strcat(str,s);
// fprintf(stderr,"%s\n",str);
		      Rtn[indx]=AllocString(str); indx++;
		    }
		   }
		 }
	   } Rtn[indx]=this->StrMaxRuns(i,trans); indx++;
	} if(Rtn) Rtn[indx]=AllocString(cmd);
// fprintf(stderr,"indx=%d; nChn=%d\n",indx,nChn);
	key=maxkey; return Rtn;
}

char	*spc_typ::tri_typ::StrMaxRuns(Int4 i,BooLean trans)
{
	Int4 j;
	if(trans) j=(i+1)%nChn; else j=i;
	assert(i < nChn && i >= 0);
	char s[40],str[150]; str[0]=0;
	if(trans) sprintf(s,"%c%c: ",chain[i],chain[j]);
	else sprintf(s,"%c: ",chain[i]); strcat(str,s);
	sprintf(s,"bond=%d(%d); ",tot[i][0],max[i][0]); strcat(str,s);
	sprintf(s,"close=%d(%d); ",tot[i][1],max[i][1]); strcat(str,s);
	sprintf(s,"dist=%d(%d)\n",tot[i][2],max[i][2]); strcat(str,s);
	return AllocString(str);
}

void	spc_typ::tri_typ::PutFlipped(FILE *fp,
			char *Msg,BooLean trans,double *value)
{
	char c=0,X='C';
	if(trans) X = 'T';
	for(Int4 i=0; i < nChn; i++){
	   if(max[i][0] > cutH && max[i][2] > cutD){
		if(c==0 && Msg != 0){
		  fprintf(fp,"%c: %s",X,Msg); c='x'; 
		  fprintf(fp,"avg.");
		  for(Int4 i=0; value[i] > 0.0; i++){
			fprintf(fp,"\t%.3lf",value[i]);
		  } fprintf(fp,"\n");
		} PutMaxRuns(fp,i);
	   }
	}
	if(c) fprintf(fp,"\n");
}

void	spc_typ::tri_typ::PutMaxRuns(FILE *fp,Int4 i)
{
	assert(i < nChn && i >= 0);
	fprintf(fp,"%d: H-bond=%d/%d; ",i,max[i][0],tot[i][0]);
	fprintf(fp,"close=%d/%d; ",max[i][1],tot[i][1]);
	fprintf(fp,"dist=%d/%d\n",max[i][2],tot[i][2]);
}

void	spc_typ::tri_typ::Reset()
{
	for(Int4 i=0; i < nChn; i++){
	  for(Int4 w=0; w < window; w++){ array[i][w]=0.0; }
	  run[i][0]=run[i][1]=run[i][2]=0;
	  max[i][0]=max[i][1]=max[i][2]=0;
	  tot[i][0]=tot[i][1]=tot[i][2]=0;
	  State[i]=0; Sum[i]=0.0; 
	}
}

void	spc_typ::tri_typ::Put(FILE *fp,Int4 n)
{
	if(n < window) return;
	fprintf(fp,"%d: ",n);
	for(Int4 i=0; i < nChn; i++){
	  fprintf(fp,"\t%.3lf",Sum[i]/(double)window);
	} fprintf(fp,"\n");
}

void	spc_typ::tri_typ::Init(char *chn, Int4 w) 
{
	Int4	i;
	cutH=cutD=40;
	for(nChn=0,i=0; chn[i]; i++) nChn++;
	window=w;
	NEW(chain,nChn +5,char);
	for(i=0; chn[i]; i++) chain[i]=chn[i];
	NEW(Sum,nChn+3,double); NEW(State,nChn+3,char); 
	NEWP(run,nChn+3,Int4); NEWP(max,nChn+3,Int4);
	NEWP(tot,nChn+3,Int4); NEWP(array,nChn+3,double); 
	for(i=0; i < nChn; i++){
	  NEW(array[i],w+3,double);
	  NEW(run[i],5,Int4); NEW(max[i],5,Int4);
	  NEW(tot[i],5,Int4);
	}
	for(i=0; i < 250; i++) nType[i]=0;
	NEW(nType[0], nChn+3,Int4);
}
void	spc_typ::tri_typ::Free()
{ 
	free(State); free(Sum);
	for(Int4 i=0; i < nChn; i++){
	   free(array[i]); free(run[i]); free(max[i]); free(tot[i]);
	} free(array); free(run); free(max); free(tot); free(chain);
	for(Int4 j=0; j < 250; j++) if(nType[j]) free(nType[j]);
	// free(nType['T']); free(nType['D']); free(nType['F']); free(nType['I']);
}


