/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "scc_typ.h"

wdg_typ scc_typ::RmBadInternalNode(Int4 Root, Int4 LeafTrimCutoff,wdg_typ T,BooLean &IsRemoved)
{
        Int4	n,e,h,t,c,p,gp,rm_node=0;
	set_typ	Children=0;
        wdg_typ newT=MkWdgraph(WdgraphN(T), WdgraphM(T));
	IsRemoved=FALSE;
	for(n=1; n <= nWdgraph(T); n++){
	  if(n==Root) continue;
	  // if(FirstOutWdgraph(n,T) == 0){ assert(FirstInWdgraph(n,T) == 0); continue; } // orphan node.
	  if(FirstInWdgraph(n,T) == 0){ assert(FirstOutWdgraph(n,T) == 0); continue; } // orphan node.
	  if(IsLeafNode(n,T)){ continue; } 
	  e=FirstInWdgraph(n,T); assert(NextInWdgraph(e,T) == 0); // i.e., this is a tree.
          p=TailWdgraph(e,T);     // arrow points from parent (tail) to child (head)
	  set_typ CSet=RtnSubTreeSeqSet(n,Set,T); 
	  set_typ PSet=RtnSubTreeSeqSet(p,Set,T); 
	  IntersectNotSet(PSet,CSet); // P := P intersect not N.
	  if(CardSet(PSet) < LeafTrimCutoff){ // this internal node is too small.
	     // This triggers a modification of the Tree...
	     rm_node=n; Children=MakeSet(nWdgraph(T)+2);  ClearSet(Children);
	     UnionSet(Set[p],Set[n]);	// SetP = SetP U SetC.
	     ClearSet(Set[n]);
	     IsRemoved=TRUE;
	     for(e=FirstOutWdgraph(n,T);e!=0;e=NextOutWdgraph(e,T)){
                c=HeadWdgraph(e,T);
		JoinWdgraph(p,c,1,newT); // attach all children of n to p.
		AddSet(c,Children);
	     } break;	// stop here and return a new tree.
	  }
	}
        for(e=1; e <= mWdgraph(T); e++){
                p=TailWdgraph(e,T);  c=HeadWdgraph(e,T);
		if(Children==0) JoinWdgraph(p,c,1,newT); 
		else if(!MemberSet(c,Children) && p != rm_node && c != rm_node) JoinWdgraph(p,c,1,newT);
        } if(Children) NilSet(Children); return newT;
}

wdg_typ scc_typ::RewireMisfitNodes(Int4 Root, Int4 LeafTrimCutoff,wdg_typ T,BooLean &IsRewired)
// Find nodes that (after making sets disjoint) don't match parent node patterns.
// move these nodes up the tree until they match (possibly to the Root node.
{
        Int4	n,e,h,t,c,p,gp,rm_child=0;
	double	lpr,Lpr;
	char	Type;
        wdg_typ newT=MkWdgraph(WdgraphN(T), WdgraphM(T));
	IsRewired=FALSE;
#if 1
	for(n=1; n <= nWdgraph(T); n++){
	  if(n==Root) continue;
	  if(FirstInWdgraph(n,T) == 0){ assert(FirstOutWdgraph(n,T) == 0); continue; } // orphan node.
	  if(IsLeafNode(n,T)){ continue; } 
	  e=FirstInWdgraph(n,T); assert(NextInWdgraph(e,T) == 0);
	  gp=TailWdgraph(e,T);
	  set_typ PSet=RtnSubTreeSeqSet(n,Set,T); 
	  set_typ GPSet=RtnSubTreeSeqSet(gp,Set,T); 
	  IntersectNotSet(GPSet,PSet); // GP := GP intersect not P.
	  // assert(CardSet(GPSet) >= LeafTrimCutoff);  // should have been dealt in prior routine.
	  // internal nodes may have less nodes due to P making up the difference.

	  sst_typ *xsst=GetOptPttrnLPR(0,PSet,GPSet,FALSE,Lpr,20,'M');
	  double WtCntsParentFG,dummy; WtCardFG_BG_Sets(WtCntsParentFG,dummy); 
	
	  for(e=FirstOutWdgraph(n,T);e!=0;e=NextOutWdgraph(e,T)){
		c = HeadWdgraph(e,T);	// a child of n.
		assert(TailWdgraph(e,T)==n);
		set_typ TmpSet=RtnSubTreeSeqSet(c,Set,T); // rtn union subtree sets.
		lpr=CalcSetvsPttrnLPR(0,TmpSet,GPSet,xsst,FALSE,'M');  // Does child match parent pattern?
		double WtCntsFG; WtCardFG_BG_Sets(WtCntsFG,dummy);
		// double min_lpr = Lpr*0.80*((double) WtCntsFG/(double) WtCntsParentFG);
		double min_lpr = Lpr*0.66*((double) WtCntsFG/(double) WtCntsParentFG);
		// if(lpr < min_lpr || lpr < MinLPRforEdge)
		if(lpr < MinLPRforEdge)
		{
			rm_child=c; IsRewired=TRUE;
			fprintf(stderr,"Bad child node: %d (lpr=%.2f; target=%.2f)\n",c,lpr,min_lpr);
			fprintf(stderr,"connecting node %d to grandparent (%d)\n",c,gp);
                	JoinWdgraph(gp,c,1,newT);
			break;
		} NilSet(TmpSet);
	  } NilSet(GPSet); NilSet(PSet); 
	  if(IsRewired) break;
	}
        for(e=1; e <= mWdgraph(T); e++){
                p=TailWdgraph(e,T);  c=HeadWdgraph(e,T);
		if(c != rm_child) JoinWdgraph(p,c,1,newT); 
        } return newT;
#else
        for(e=1; e <= mWdgraph(T); e++){
                p=TailWdgraph(e,T);     // arrow points from parent (tail) to child (head)
                c=HeadWdgraph(e,T);
		if(IsRootNode(p,T)){
			JoinWdgraph(p,c,1,newT);
			continue;   // no need to check these.
		}
		gp=ParentNode(p,T);	// get grandparent of c.

		set_typ PSet=RtnSubTreeSeqSet(p,Set,T); 
		set_typ GPSet=RtnSubTreeSeqSet(gp,Set,T); 
	   	sst_typ *xsst=GetOptPttrnLPR(0,PSet,GPSet,FALSE,Lpr,20,'M');
		double WtCntsParentFG,dummy; WtCardFG_BG_Sets(WtCntsParentFG,dummy); 

		IntersectNotSet(GPSet,PSet); // GP := GP intersect not P.
  		Int4 crd=CardSet(GPSet);
		if(crd < LeafTrimCutoff){
			 JoinWdgraph(p,c,1,newT);
			 continue; // this should be removed later...
		}
		set_typ TmpSet=RtnSubTreeSeqSet(c,Set,T); // rtn union subtree sets.
		lpr=CalcSetvsPttrnLPR(0,TmpSet,GPSet,xsst,FALSE,'M');  // Does child match parent pattern?
		double WtCntsFG; WtCardFG_BG_Sets(WtCntsFG,dummy);
		double min_lpr = Lpr*0.80*((double) WtCntsFG/(double) WtCntsParentFG);
		if(lpr < min_lpr){
			IsRewired=TRUE;
			fprintf(stderr,"Bad child node: %d (lpr=%.2f; target=%.2f)\n",c,lpr,min_lpr);
			fprintf(stderr,"connecting node to grandparent (%d)\n",gp);
                	JoinWdgraph(gp,c,1,newT);
			
		} else JoinWdgraph(p,c,1,newT);
		NilSet(GPSet); NilSet(PSet); NilSet(TmpSet);
        } return newT;
#endif
}

wdg_typ	scc_typ::TrimLeaves(Int4 Root, Int4 LeafTrimCutoff,wdg_typ T)
{
	Int4 n,e,h,t,c,p;
	wdg_typ newT=MkWdgraph(WdgraphN(T), WdgraphM(T));
	for(e=1; e <= mWdgraph(T); e++){
		p=TailWdgraph(e,T); 	// arrow points from parent (tail) to child (head)
		c=HeadWdgraph(e,T);
		if(NumEdgesOut(c,T)==0 && (CardSet(Set[c]) < LeafTrimCutoff)) {
			// DeleteSet(c,leaves); // this might make p a single child node!
			UnionSet(Set[p],Set[c]);	// SetP = SetP U SetC.
			// ClearSet(OptimizedSet[child]);
			// DeleteOptimizedSet(child); // No need to delete it... 
			// NumOptimizedSets--;
			// edge e is deleted from newT by default.
		} else JoinWdgraph(p,c,1,newT); 
	} return newT;
}

set_typ	scc_typ::RtnLeafSet(wdg_typ T)
// WARNING: for shortest paths arrows are reversed!!!
{	// return a set containing only leaf nodes.
		set_typ IsLeaf=MakeSet(WdgraphN(T)+1); ClearSet(IsLeaf);
		for(Int4 n=1; n <= nWdgraph(T); n++){
		   if(NumEdgesIn(n,T) > 0 && NumEdgesOut(n,T)==0) AddSet(n,IsLeaf); 
		} return IsLeaf;
}

#if 1// removing bad edges...

set_typ	scc_typ::GetRewiredSet(Int4 parent, wdg_typ T)
// return a set containing those nodes affected by rewiring parent to a child node.
{
	set_typ Rewired=MakeSet(WdgraphN(T)+1); ClearSet(Rewired);
	Int4 c,e;
	AddSet(parent,Rewired);
	for(e=FirstOutWdgraph(parent,T); e!= 0; e=NextOutWdgraph(e,T)){
	     c=HeadWdgraph(e,T); 	// arrow from parent (tail) to child (head)
	     assert(parent==TailWdgraph(e,T));
	     AddSet(c,Rewired);
	} return Rewired;
}

Int4	scc_typ::FindEdge(Int4 tail, Int4 head, wdg_typ G)
{
	Int4 h,t,e,rtn=0;
	for(e=1; e <= mWdgraph(G); e++){
	     t=TailWdgraph(e,G); h=HeadWdgraph(e,G); 	// arrow from parent (tail) to child (head)
	     if(h==head && t==tail){ assert(rtn==0);  rtn=e; }	// make sure only one such edge...
	} return rtn;
}

wdg_typ	scc_typ::RmBadEdge(Int4 bad_edge, Int4 &ParentNode, wdg_typ T)
// WARNING: assumes that T is a tree.
{
	Int4	  e,f,n,c,gc,p,i;
	set_typ Skip=MakeSet(mWdgraph(T)+1); ClearSet(Skip);
	wdg_typ newT=MkWdgraph(WdgraphN(T), WdgraphM(T)); ParentNode=0;
	// Find the bad edge and label it and all sub-edges; connect subedges to the parent node.
	for(e=1; e <= mWdgraph(T); e++){
	     if(e == bad_edge){
	        p=TailWdgraph(e,T); c=HeadWdgraph(e,T); 	// arrow from parent (tail) to child (head)
		ParentNode=p; AddSet(e,Skip);
	        UnionSet(Set[p],Set[c]); // merge child node with parent.
		// ClearSet(Set[c]); // clear these as well?
		for(f=FirstOutWdgraph(c,T); f!= 0; f=NextOutWdgraph(f,T)){
		    gc=HeadWdgraph(f,T);	// a child of c and grandchild of p.
		    JoinWdgraph(p,gc,1,newT);	// join these directly to grandparent.
		    AddSet(f,Skip);	// don't add these later...
		}
	     }
	}
	for(e=1; e <= mWdgraph(T); e++){
		if(!MemberSet(e,Skip)){
	            p=TailWdgraph(e,T); c=HeadWdgraph(e,T); 	// arrow from parent (tail) to child (head)
		    JoinWdgraph(p,c,1,newT);	// join these directly to grandparent.
		}
	} NilSet(Skip);
	return newT;
}
#endif

wdg_typ	scc_typ::RmSingleChildNodes(wdg_typ T,Int4 LeafTrimCutoff)
{
	BooLean *skip;
	Int4	  e,f,n,c,p,i,I=0,*Parent,*Child,*Node;
	NEW(Parent,mWdgraph(T)+3,Int4); NEW(Child,mWdgraph(T)+3,Int4);
	NEW(Node,mWdgraph(T)+3,Int4); NEW(skip,mWdgraph(T)+3, BooLean);
	wdg_typ newT=MkWdgraph(WdgraphN(T), WdgraphM(T));
	for(e=1; e <= mWdgraph(T); e++){
	     if(skip[e]) continue;
	     n=TailWdgraph(e,T); c=HeadWdgraph(e,T); 	// arrow from parent (tail) to child (head)
	     if(NumEdgesIn(n,T) == 1 && NumEdgesOut(n,T)==1   // is n a single child & parent?
			&& NumEdgesIn(c,T) == 1 && NumEdgesOut(c,T) == 0){  // is child a leaf?
	       f=FirstInWdgraph(n,T);	// and only edge in...
	       p=TailWdgraph(f,T); assert(n==HeadWdgraph(f,T));
#if 1
	       Int4 cardP=CardSet(Set[p]);
	       Int4 cardC=CardSet(Set[c]);
	       if((cardP - cardC) < LeafTrimCutoff) continue; // Misc part is too small, so continue...
	       double d=(double) (cardP-cardC)/cardP;
	       if(d <= 0.33) continue;	// Misc part is relatively too small; so continue;
#endif
	       skip[e]=TRUE; skip[f]=TRUE;  // don't include p -> n and n -> c edges...
	       Parent[I]=p; Child[I]=c; Node[I]=n; I++;	// edge to add...etc.
	     }
	}
	for(e=1; e <= mWdgraph(T); e++){
		n=TailWdgraph(e,T); c=HeadWdgraph(e,T); 
		if(!skip[e]){ JoinWdgraph(n,c,1,newT); }
	}
	for(i=0; i < I; i++){
		p=Parent[i]; c=Child[i]; n=Node[i]; 
	        JoinWdgraph(p,c,1,newT);	// join p to c directly without n. 
	        UnionSet(Set[p],Set[n]); // merge with parent.
	        // DeleteOptimizedSet(node);	// delect node n.
		// NumOptimizedSets--; 
	} free(skip); free(Parent); free(Node); free(Child);
	return newT;
}

#if 0
BooLean	HasSiblingNodes(Int4 v, wdg_typ T)
	{	// If node v is a single child then return TRUE. 
		Int4 w=ParentNode(v,T);
		if(w==0) return FALSE;
		if(NumEdgesOut(w,T) == 1) return TRUE; else return FALSE;
	}
#endif

BooLean	scc_typ::IsLeafNode(Int4 v, wdg_typ T)
{	// return the parent node of v in T or 0 if v lacks a parent.
	Int4 e,n=NumEdgesOut(v,T); 
	if(n==0) return TRUE; else return FALSE;
} 

Int4	scc_typ::ParentNode(Int4 v, wdg_typ T)
{	// return the parent node of v in T or 0 if v lacks a parent.
	Int4 e,n=NumEdgesIn(v,T); assert(n <= 1);   // make sure that this is a tree.
	if(n==0) return 0;
	else { e=FirstInWdgraph(v,T); return TailWdgraph(e,T); }
} 

BooLean	scc_typ::IsRootNode(Int4 v, wdg_typ G)
{
	if(NumEdgesIn(v,G) == 0) return TRUE; else return FALSE;
}

void	scc_typ::PrintNewickTree(FILE *fp,Int4 root, char mode, wdg_typ T)
{ // print tree in Newick format.
	// fprintf(fp,"("); TreeDFS(fp, root, T); fprintf(fp,")%d;\n",root);
	// PutWdgraph(stderr,T);
	TreeDFS(fp, root, mode, T); fprintf(fp,";\n");
}

void	scc_typ::TreeDFS(FILE *fp, Int4 v, char mode, wdg_typ T)
{
	// start from root, recursively print tree in Newick format.
	Int4	w,e,i,n=NumEdgesOut(v,T);
	if(n > 0){
	   fprintf(fp,"(");
	   for(i=0,e=FirstOutWdgraph(v,T);e!=0;e=NextOutWdgraph(e,T)){
		i++; w = HeadWdgraph(e,T);	// a child of v.
		TreeDFS(fp,w,mode,T);
		if(i < n) fprintf(fp,",");
	   } fprintf(fp,")");
	}
	if(mode == 'I'){			// output to count seqs in trees.
	  Int4 index=InSetToOptimizedSet[v];
	  // if(IsRootNode(v,T)) fprintf(fp,"%d_Root%d",v,SetSize-1); else 
	  fprintf(fp,"%d_Seq%d",v,CardSet(OptimizedSet[index]));
	} else fprintf(fp,"%d_Set%d",v,v);	// output for HyperPartition.
}

Int4	scc_typ::TreeDFS(Int4 *SubTreeNodes, Int4 &number, Int4 v, wdg_typ T)
{
	// start from root, recursively print tree in Newick format.
	Int4	w,e,i,n=NumEdgesOut(v,T);
	number++; assert(number <= nWdgraph(T)); SubTreeNodes[number]=v;
	if(n > 0){
	   for(i=0,e=FirstOutWdgraph(v,T);e!=0;e=NextOutWdgraph(e,T)){
		i++; w = HeadWdgraph(e,T);	// a child of v.
		TreeDFS(SubTreeNodes,number,w,T);
	   }
	}
	return 0;
}

set_typ	scc_typ::RtnSubTreeSeqSet(Int4 subroot, set_typ *set,wdg_typ T)
// return the nodes in the subtree rooted at subroot. 
{
	Int4	number=0,*subTree; 
	NEW(subTree, nWdgraph(T) +3, Int4);
	set_typ SetU=MakeSet(SetSize); ClearSet(SetU);
	TreeDFS(subTree,number,subroot,T); 
	for(Int4 j=1; j<= number; j++) UnionSet(SetU,set[subTree[j]]);
#if 0
        fprintf(stderr,"Set%d subtree: ",subroot);
        for(Int4 j=1; j<= number; j++){
	    set_typ SetX=set[subTree[j]];
	    fprintf(stderr,"%d(%d|%d) ",subTree[j],CardSet(SetX),CardInterSet(SetX,RandomSet));
	    // fprintf(stderr,"%d(%d) ",subTree[j],CardInterSet(SetX,RandomSet));
	} fprintf(stderr,"\n");
#endif
	free(subTree); return SetU;
}

Int4	*scc_typ::RtnSubTreeNodes(Int4 subroot, Int4 &size, wdg_typ T)
{
	Int4	number=0,*SubTreeNodes; 
	NEW(SubTreeNodes, nWdgraph(T) +3, Int4);
	TreeDFS(SubTreeNodes,number,subroot,T); 
	size=number;
	return SubTreeNodes;
}

Int4	scc_typ::LabelMoreNodes(set_typ Labeled, wdg_typ T)
{	// return a set 
	Int4 N=0,e,n,m,x;
	set_typ NewLabeled=MakeSet(WdgraphN(T)+1); ClearSet(NewLabeled);
	for(n=1; n <= nWdgraph(T); n++){
	  BooLean bad=FALSE;  // Are all children of n labeled?
	  if(MemberSet(n,Labeled)) continue;
	  if(FirstOutWdgraph(n,T) == 0){	// this is an orphan node.
		assert(FirstInWdgraph(n,T) == 0); continue;
		// AddSet(n,NewLabeled); N++; continue; 
	  }
	  // ^ ignore leaves and orphan nodes; these should be labeled.
	  for(e=FirstOutWdgraph(n,T);e!=0;e=NextOutWdgraph(e,T)){
		m = HeadWdgraph(e,T);	// a child of n.
		assert(TailWdgraph(e,T)==n);
		if(!MemberSet(m,Labeled)){ bad=TRUE; break; }
	  } if(!bad){ AddSet(n,NewLabeled); N++; }
	} if(N > 0) UnionSet(Labeled,NewLabeled); 
	NilSet(NewLabeled); return N;
}

//****************** Make nodes disjoint ************************************
BooLean	scc_typ::MakeNodeSetsDisjoint(wdg_typ &Tree)
// returns TRUE if sets were changed, else returns FALSE.
{
   BooLean IsChanged=FALSE;
   Int4	NumNodesInList,*ListOfTreeNodes=RtnSubTreeNodes(Root,NumNodesInList,Tree);
   // 1. Make all leaf nodes disjoint from each other.
   set_typ Leaves=RtnLeafSet(Tree);
   set_typ Labeled=CopySet(Leaves);
   set_typ FixedSet=CopySet(Leaves); ClearSet(FixedSet); AddSet(Root,FixedSet);
   set_typ SetU=MakeSet(SetSize);
   Int4	   I,J,i,j,iter=0,N=CardSet(Labeled);
   h_type HG=Histogram("difference in pattern match scores", 0,100,1.0);
   do {			//  for leaves only, put ambiguous sequences into best matching set.
     iter++; fprintf(stderr,"Disjoint routine iteration %d (N=%d)\n",iter,N);
     for(I=1; I <= NumNodesInList; I++){
	i=ListOfTreeNodes[I]; 
	if(MemberSet(i,FixedSet)) continue;	  // Root skipped as well...
	if(!MemberSet(i,Labeled)) continue; 
	for(J=1; J <= NumNodesInList; J++){
	     if(I == J) continue;
	     j =ListOfTreeNodes[J]; 
	     if(j == Root) continue;
	     if(!MemberSet(j,Labeled)) continue;
#if 0	     // Should I check for child-parent overlaps?  These should probably be treated differently.
	     // This should only occur after the first iteration.
	     // other checks?
#endif
	     if(MemberSet(j,FixedSet)){		 // setJ is fixed whereas setI is not.
	        IntersectNotSet(Set[i],Set[j]);  //  SetI = SetI && Not SetJ.
// IsChanged=TRUE;  // calling this another time will not change this...
	     } else {
	       IntersectSet1(Set[i],Set[j],SetU); 	// SetU = SetI & SetJ; no need to clear SetU.
	       for(Int4 sq=1; sq <= NumSeqsCMSA(cma); sq++){
		 if(!MemberSet(sq,SetU)) continue;
		 Int4    sI=GetPatternScore(SST[i],sq),sJ=GetPatternScore(SST[j],sq);
		 if(sI > sJ) DeleteSet(sq,Set[j]); else DeleteSet(sq,Set[i]);
		 IncdHist((double)abs(sI-sJ),HG); // int abs(int j);
		 IsChanged=TRUE;
	       }
	     }
	}
     } UnionSet(FixedSet,Labeled);	// FixedSet = FixedSet U Labeled.
     N=LabelMoreNodes(Labeled,Tree);	// label nodes whose subtree is labeled.
   } while(N > 0);
   PutHist(stderr,50,HG); NilHist(HG);

   ClearSet(SetU);
   for(I=1; I <= NumNodesInList; I++){
	i=ListOfTreeNodes[I]; 
	if(i  !=  Root) UnionSet(SetU,Set[i]);
   } IntersectNotSet(Set[Root],SetU);	// Put all sequences not in other sets into the Root set.
   NilSet(SetU); NilSet(FixedSet);

   if(IsChanged && CardSet(Labeled) != NumNodesInList) {
	PutSet(stderr,Labeled);
	fprintf(stderr,"Labeled = %d; NumOptSets = %d\n",CardSet(Labeled),NumNodesInList);
	fprintf(stderr,"OptSet: ");
        for(i=1; i <= NumNodesInList; i++){
		fprintf(stderr," %d",ListOfTreeNodes[i]);
	} fprintf(stderr,"\n\n");
	assert(CardSet(Labeled) == NumNodesInList);
   } fprintf(stderr,"\n\nLabeled nodes:\n"); PutSet(stderr,Labeled); NilSet(Labeled);
   free(ListOfTreeNodes); NilSet(Leaves);
   return IsChanged;
}

