/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "scc_typ.h"

scc_typ::scc_typ(char *argv1,cma_typ in_cma,swt_typ *SWT)
{
	SCC_MODE='A'; OwnInput=TRUE; cma=in_cma; 
	InitializeParameters( );

	FILE *fp=open_file(argv1,".amc","r");
	Int4 i,Len,NumIters=1003,*nsets=0; NEW(nsets,NumIters+3,Int4);
	set_typ **set;  NEWP(set,NumIters+3,set_typ);
        sst_typ ***sst; NEWPP(sst,NumIters+3,sst_typ);

	RandomSet=0; NumRandom=1+(NumSeqsCMSA(cma)/3);
	for(Int4 iter=1; iter <= NumIters; iter++){
		if(iter > 1000) print_error("Too many input files for processing");
		nsets[iter]=ReadPttrnSets(fp,set[iter], sst[iter],Len);
		if(iter == 1 && nsets[iter] < 1){
                   fprintf(stderr,"\n\tInput '*.amc' file contains no sets!\n");
                   print_error("\t***** Fatal error in pmcBPPS procedure *****\n\n");
                } 
                if(RandomSet==0){
                        RandomSet=MakeSet(SetN(set[iter][1])); ClearSet(RandomSet);
                        for(i=NumSeqsCMSA(cma)+1; i < SetN(RandomSet); i++) AddSet(i,RandomSet);
                }
                if(nsets[iter]==0){ NumIters = iter-1; break; }
                else if(iter==1){
                        Int4 n1=NumSeqsCMSA(cma);
                        Int4 n2=SetN(set[iter][1])-1;
                        if((n2 - n1) != NumRandom){
                                fprintf(stderr,"%d seqs + %d random != %d card_sets\n",
                                        n1,NumRandom,n2);
                                fprintf(stderr,"Need %d random\n",n2-n1);
                                print_error("scc_typ: mma & amc input files are inconsistent");
                        }
                } CheckSets(nsets[iter],set[iter],iter);
	} fclose(fp);

	InitA(NumIters,nsets,set,sst);	// initializes MaxSetID;

	assert(NumRandom==SetSize - NumSeqsCMSA(cma)-1);
	// RandomSet=MakeSet(SetSize); ClearSet(RandomSet);
	// for(Int4 i=NumSeqsCMSA(cma)+1; i < SetSize; i++) AddSet(i,RandomSet);
	CheckSets(NumSets,Set);

        swt=SWT; // InitSWT( );
	lpr = new lpr_typ(cma,swt);
}

Int4    scc_typ::ReadPttrnSets(FILE *fp,set_typ *&set, sst_typ **&sst, Int4 &Len)
{
        Int4    k,i,N,rtn;
        rtn=fread(&N,sizeof(Int4),1,fp);
        if(rtn != 1) return 0;
        // if(rtn != 1) print_error("ReadPttrnSets() input error 0");
        rtn=fread(&Len,sizeof(Int4),1,fp);
        if(rtn != 1) print_error("ReadPttrnSets() input error 1");
        NEW(set,N+3,set_typ); NEWP(sst,N+3,sst_typ);
        for(i=1; i <= N; i++){
           set[i]=ReadSet(fp);
           NEW(sst[i],Len+4,sst_typ);
           rtn=fread(sst[i],sizeof(sst_typ),Len+2,fp);
           if(rtn != Len+2) print_error("ReadPttrnSets() input error 2");
        } return N;
}

void    scc_typ::InitA(Int4 NumIters, Int4 *nsets, set_typ **set, sst_typ ***sst)
{
	Int4 I,J,i;
	NumSets=0;
        SetSize=0;
        for(I=1; I <= NumIters; I++){
	    for(i=2; i <= nsets[I]; i++){		// skip MainSet = set[I][1]; 
              if(set[I][i])
	      {
		NumSets++; 
                if(SetSize == 0) SetSize=SetN(set[I][i]);       // Size of first set encountered.
                else if(SetSize != SetN(set[I][i])) print_error("InitA(): Inconsistent set sizes!");
	      }
	    }
        } // NumRandom=SetSize - NumSeqsCMSA(cma)-1;

	NEWP(SST, NumSets+3,sst_typ); 
	NEW(Set,NumSets+3,set_typ); NEW(SetID,NumSets+3,Int4);
        for(I=1,J=0; I <= NumIters; I++){
           for(i=2; i <= nsets[I]; i++){              // skip 1st set = MainSet...
             if(set[I][i])
	     {
		J++; SetID[J] = (I * 10) + i-2;  // set identifier; maps back to rtf files, etc.
		Set[J]=set[I][i]; SST[J]=sst[I][i]; 
             }
           }
        } MaxSetID=SetID[J];
	NEW(SetID2index,MaxSetID+3,Int4);
	for(i=1; i<= NumSets; i++){ SetID2index[SetID[i]] = i; }
#if 0	// Clean up sets...
	double NumStdev=2.5;
	for(i=1; i <= NumSets; i++){
	     if(Set[i]){
		set_typ TmpSet=RmWeakMatches(0,i,NumStdev);
		Int4 c1=CardSet(TmpSet),c2=CardSet(Set[i]);
		double d=(double) c1/(double)c2;
		if(d >= 0.90){
			fprintf(stderr,"Set%d: %d/%d sequences removed\n",SetID[i],c2-c1,c2);
			NilSet(Set[i]); Set[i] = TmpSet; 
		} else NilSet(TmpSet);
     	     }
	}
#endif
}

scc_typ::scc_typ(Int4 NumIters, Int4 *nsets, set_typ **set,sst_typ ***sst,cma_typ in_cma,swt_typ *SWT)
{
	SCC_MODE='A'; OwnInput=TRUE; cma=in_cma; 
	InitializeParameters( );
	InitA(NumIters,nsets,set,sst);	// initializes MaxSetID;

	NumRandom=SetSize - NumSeqsCMSA(cma)-1;
	RandomSet=MakeSet(SetSize); ClearSet(RandomSet);
	for(Int4 i=NumSeqsCMSA(cma)+1; i < SetSize; i++) AddSet(i,RandomSet);
	CheckSets(NumSets,Set);

        swt=SWT; // InitSWT( );
	lpr = new lpr_typ(cma,swt);
}

scc_typ::scc_typ(Int4 nsets,set_typ *set, sst_typ **sst, cma_typ in_cma, Int4 *set_id,swt_typ *SWT)
{
	SCC_MODE='B'; OwnInput=FALSE; cma=in_cma; 
	InitializeParameters( );
	NumSets=nsets; Set=set; SetID=set_id; SST=sst; 
	NEWP(CSQ,nsets+5,unsigned char);
	MaxSetID=0; 
	for(Int4 i=1; i<= nsets; i++){ MaxSetID=MAXIMUM(Int4,SetID[i],MaxSetID); }
	NEW(SetID2index,MaxSetID+3,Int4);
       	for(Int4 i=1; i<= NumSets; i++){ SetID2index[SetID[i]] = i; }
	InitB();

	NumRandom=SetSize - NumSeqsCMSA(cma)-1;
	RandomSet=MakeSet(SetSize); ClearSet(RandomSet);
	for(Int4 i=NumSeqsCMSA(cma)+1; i < SetSize; i++) AddSet(i,RandomSet);
	CheckSets(NumSets,Set);

        swt=SWT; // InitSWT( );
	lpr= new lpr_typ(cma,swt);
}

void	scc_typ::CheckSets(Int4 nsets,set_typ *set,Int4 iter)
{
   BooLean okay=TRUE;
   Int4	y,failed=0;
   if(iter > 0) fprintf(stderr,"================== iter %d (%d sets) ===================\n",iter,nsets);
   for(Int4 x=1; x <= nsets; x++){
        if(set[x] == 0) continue;
	y = CardInterSet(set[x],RandomSet);
	if(iter > 0 || y > 0){
            fprintf(stderr,"Card Set[%d] = %d [%d].\n", x,CardSet(set[x]),y);
	    if(y > 0) failed++;
	}
#if 0	// Debug...
        fprintf(stderr,"Card Set[%d] = %d; Random = %d.\n",
                x,CardSet(set[x]),CardInterSet(set[x],RandomSet));
        assert(CardInterSet(set[x],RandomSet) == 0);
#endif
   } if(failed > 0) print_error("pmcBPPS input error: set file (*.stb) fails to check out.\n");
}


void    scc_typ::InitB( )
{
	SetSize=0;
        for(Int4 i=1; i <= NumSets; i++){
            if(Set[i]){	// some sets may be skipped.
                // fprintf(stderr,"Set%d_%d/%d = %d\n",I,i,NumSets[I],CardSet(Set[I][i]));
                if(SetSize == 0) SetSize=SetN(Set[i]);       // Size of first set.
                else if(SetSize != SetN(Set[i])){
                        print_error("scc_typ( ): Fatal! Input set size inconsistent!");
                }
            }
        } 
}

void	scc_typ::InitializeParameters( )
{
	swt=0; OptimizedSet=0; InSetToOptimizedSet=0; NumOptimizedSets=0;
        LPR=0; CSQ=0; RtnOptSets=0; NumRtnOptSet=0; RtnOptSetID=0;
        OptTree=0; UCSet=0; UCSST=0; UCSetID=0; NumClust=0;
        OptimizedSetToInSet=0;
        Length=LengthCMSA(1,cma); assert(nBlksCMSA(cma) ==1);
        AB=AlphabetCMSA(cma);
	MaxSetID=0;

	// MinSetSizeRatio=0.75;		// in order to merge sets...
	MinSetSizeRatio=0.80;		// in order to merge sets...
	MaxDistinctRatio=0.1;
	// MaxDistinctRatio=0.4;
	MinIntersect=0.80;
	SuperSetMinRatio=1.33;
	SubSetMaxRatio=1.0/SuperSetMinRatio;
	MinSubIntersectSuperRatio=0.50;	 // What fraction of subset must be in superset.

	MinFractPttrnIntersect=0.33;
	MinPttrnIntersect=5;

#if 0	// old settings
	MinLPRtoMerge=500.0;
	MinLPRforEdge=100.0;
#else	// new settings
	MinLPRtoMerge=250.0;
	MinLPRforEdge=50.0;
#endif
#if 0// For looking at distributions.  allows everything through heuristic filters.
	MinIntersect=0.0;
	MinSetSizeRatio=0.0;
	MaxDistinctRatio=0.0;
	MinPttrnIntersect=0;
	MinFractPttrnIntersect=0.0;
#endif
#if 0	// moved to lpr_typ
	RST=new rst_typ('L');  // residue sets
	LegalSST=RST->LegalResSets();
#endif
}

void	scc_typ::SetParameters(double d, char mode )
{
	switch(mode){
	     case 'S': 
		  if(d <= 1.0 || d >= 100000.0) print_error("SuperSetMinRatio out of range");
		  SuperSetMinRatio=d; SubSetMaxRatio=1.0/d; break;
	     case 'M': 
		if(d <= 0.20 || d >= 1.0) print_error("minimum ratio out of range");
		MinSetSizeRatio=d; break;
	     case 'i': 	// minimum smaller set fractional intersection.
		if(d <= 0.2 || d >= 1.0) print_error("minimum intersection out of range");
		MinIntersect=d; break;
	     case 'D': 
		if(d <= 0.0 || d >= 0.25) print_error("MaxDistinctRatio out of range");
		MaxDistinctRatio=d; break;
	     default: print_error("SetParameters( ) mode input error"); break;
	}
}

void    scc_typ::Free( )
{
	Int4 i,j;
	if(OwnInput){ 
		if(SST) free(SST);
		if(CSQ){
		    for(i=1; i<= NumSets; i++) if(CSQ[i]) free(CSQ[i]); 
		    free(CSQ);
		}
		if(Set) free(Set); 
		if(SetID) free(SetID); 
	} if(SetID2index) free(SetID2index);
#if 0	// moved to lpr_typ
        for(Int4 s=1; s <= Length; s++){ free(CntFG[s]); free(CntBG[s]); free(SqWt[s]); }
	free(CntFG); free(CntBG); free(SqWt); free(AveSqIWt);
	delete RST;
#endif

	if(UCSet) free(UCSet); 
	if(UCSST){ free(UCSST); }
	if(UCSetID) free(UCSetID);
	if(OptTree) NilWdgraph(OptTree);

	NilSet(RandomSet);
	delete lpr;
}

