/*************************************************************************************
    Copyright (C) 1997-2019 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in the
    Software without restriction, including without limitation the rights to use, copy
,
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software
,
    and to permit persons to whom the Software is furnished to do so, subject to the
    following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED
,
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PART
ICULAR
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS B
E
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT
, TORT
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE O
R
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ***********************************************************************************/
#if !defined (_DCA_TYP_H_)
#define _DCA_TYP_H_

#include <assert.h>
#include "conjugrad.h"
#include <ctime>

#ifdef CUDA
#include "evaluate_cuda_kernels.h"
#include <cuda.h>
#include <cuda_runtime_api.h>
// #include "evaluate_cuda.h"
#endif

#define N_ALPHA 21
#ifdef PADDING
#define N_ALPHA_PAD 32
#else
#define N_ALPHA_PAD 21
#endif

#define X1_index(nc,j,a) ((a) * (nc) + j)
#define VV(x1,j,a,nc) (x1[X1_index(nc,j,a)])
#define GG1(g1,j,a,nc) (g1[X1_index(nc,j,a)])
// #define L1(j,a) l1[X1_index(j,a)]

#define X2_index(nc,b,k,a,j) ((((b) * nc + (k)) * (N_ALPHA_PAD) + (a)) * nc + j)
#define WW(x2,b,k,a,j,nc) (x2[X2_index(nc,b,k,a,j)])
#define GG2(g2,b,k,a,j,nc) (g2[X2_index(nc,b,k,a,j)])
// #define L2(b,k,a,j) l2[X2_index(b,k,a,j)]

#define MSA_index(nc,i,j) ((i) * (nc) + j)
#define XX(ud,i,j,nc) ((ud)->msa[MSA_index(nc,i,j)])

#define PC_index(nc,a,s) ((a) * (nc) + (s))
#define PrC(nc,a,s) (precomp[PC_index(nc,a,s)])
#define PrCN(nc,a,s) (precomp_norm[PC_index(nc,a,s)])

#define OMP_PC_index(nc,i,a,s) (((i) * N_ALPHA + (a)) * nc + (s))
#define OMP_PrC(nc,i,a,s) (precomp[OMP_PC_index(nc,i,a,s)])
#define OMP_PrCN(nc,i,a,s) (precomp_norm[OMP_PC_index(nc,i,a,s)])

//================== evaluate_cpu ================
//================== evaluate_omp ================
#define GG2L(g2l,b,k,a,j,nc) (g2l[X2_index(nc,b,k,a,j)])
// #define G2L(b,k,a,j) g2l[X2_index(b,k,a,j)]

typedef struct parse_options {
          char option;
          int index;
          char *argument;
          struct parse_options *next;
} parse_options;

typedef struct usrdata {
	   // User data passed to the LBFGS optimizer and available by 'evaluate'
           unsigned char* msa;
           conjugrad_float_t *weights; // sequence weights
           int ncol;       // Number of columns in the MSA (i.e. L)
           int nrow;       // Number of rows in the MSA (i.e. N)
           int nsingle;    // Number of single emission potentials
           int nvar;       // Number of variables
           void *extra;    // Extra data for the actual evaluate implementation
           // Single emission regularization coefficients
           conjugrad_float_t lambda_single;
           // Pairwise emisssion regularization coefficients
           conjugrad_float_t lambda_pair;
           conjugrad_float_t reweighting_threshold;
} usrdata;

class dca_typ { // Direct coupling analysis type.
public:
        dca_typ( ) { fprintf(stderr,"Illegal constructor"); exit(1);  }
        dca_typ(int ac, char *av[]){ argc=ac; argv=av; Init( ); }
        ~dca_typ( ) { Free(); }
	// ============== dca_run.cc ======================
        int     Run( );
private:
	void    Init(){ return; }
        void    Free(){ return; }
	//=============== input data ====================
	int argc;
	char **argv;
	// ============== dca_run.cc ======================
	void	init_bias(conjugrad_float_t *x, usrdata *ud);
	void	logo(bool color);
	char	*concatenate(const char *s1, const char *s2);
	void	usage(char* exename, int long_usage);

	//============== dca_seq.cc ======================
	unsigned char *read_msa(FILE *f, int *ncol, int *nrow);
	unsigned char itoaa(unsigned char i);
	unsigned char aatoi(unsigned char aa);
	void trim_right(char *str);

	//============== dca_wts.cc ======================
	void calculate_weights(conjugrad_float_t *w, unsigned char *msa, uint64_t ncol,
			uint64_t nrow, conjugrad_float_t threshold);
	void uniform_weights(conjugrad_float_t *w, uint64_t nrow);
	//============== dca_opt.cc ======================
	parse_options *new_parse_options(char option, char* argument, int index);
	int findopt(char opt, const char* options);
	parse_options *parseopt(int argc, char *argv[], const char *options);

	//============== dca_io.cc ======================
	void write_matrix(FILE *out, conjugrad_float_t *mat, int ncol, int nrow);
	void write_raw(FILE *out, conjugrad_float_t *x, int ncol);
	void read_raw(char *filename, usrdata *ud, conjugrad_float_t *x);

	//============== dca_util.cc ======================
	void sum_submatrices(conjugrad_float_t *x, conjugrad_float_t *out, int ncol);
	void die(const char* message);
	void apc(conjugrad_float_t *mat, int ncol);
	void normalize(conjugrad_float_t *mat, int ncol);
	//============== dca_cpu.cc ======================
	int init_cpu( void *instance );
	int destroy_cpu( void *instance );

	//============== dca_omp.cc ======================
#ifdef OPENMP
	int init_cpu_omp( void *instance );
	int destroy_cpu_omp( void *instance );
#endif

	//============== dca_cuda.cc ======================
#ifdef CUDA
	int init_cuda( void *instance );
	int destroy_cuda( void *instance );
#endif
};

typedef struct extra_usrdata { // Extra memory for symmetric access
       	conjugrad_float_t *g2;
} extra_usrdata;
conjugrad_float_t evaluate_cpu (void *instance,const conjugrad_float_t *x,
        	conjugrad_float_t *g, const int nvar);

typedef struct omp_extra_usrdata { // Extra memory for symmetric access
       	conjugrad_float_t *g2;
} omp_extra_usrdata;
conjugrad_float_t evaluate_cpu_omp (void *instance,const conjugrad_float_t *x,
			conjugrad_float_t *g, const int nvar);

conjugrad_float_t evaluate_cuda (void *instance,const conjugrad_float_t *x,
        	conjugrad_float_t *g, const int nvar);

#endif

