/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "gpsi_typ.h"
#include "alphabet.h"
#include "residues.h"
#include "smatrix.h"
#include "cmsa.h"

#define USAGE "usage: unrelatedseq fafile cmafile [options]\n\
      options:\n\
        -X<int>  X dropoff for gapxdrop\n\
        -H<int>  heapsize (default 5000)\n\
        -T<int>  blast word hit threshold (default 11)\n\
        -m<int>  print option m=0..6 (default 4)\n\
        -L       do low complexity masking of database sequences\n\
        -v       verbose output\n\
        -D       Don't SEG query sequence\n\
        -d       input sequenct set instead of cmafile\n\
        -w<int>  alignment width (default 60)\n\
        -S<int>  Use score mode with cutoff score of <int>\n\
        -I<int>  Use score mode with cutoff score of <int> percent identities\n\
        -x       dummy\n\n"

int	main(int argc, char *argv[])
{
	Int4    T=11,gap_open=11,gap_extend=1,time1=time(NULL),timeX;
        double	x_parameter=25.0;
	Int4	width=60,cutscore=0;
	UInt4	hpsz=5000,printopt=4;
	double	Ethresh=0.001,Ecutoff=0.001;
	BooLean	verbose=FALSE;
	BooLean	seg_seq=TRUE,success=FALSE,cmafile=TRUE;
	char	mask_dbs=' ',mode='I';
	double	percentID=0.33;
	BooLean top_only=TRUE;

	if(argc < 3) print_error(USAGE);
        for(Int4 arg = 3; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(USAGE);
           switch(argv[arg][1]) {
	     case 'X': x_parameter=IntOption(argv[arg],'X',1,1000,USAGE); break;
	     case 'H': hpsz=IntOption(argv[arg],'H',1,1000000,USAGE); break;
	     case 'T': T=IntOption(argv[arg],'T',1,100,USAGE); break;
	     case 'm': printopt=IntOption(argv[arg],'m',0,6,USAGE); break;
             case 'e': Ecutoff=RealOption(argv[arg],'e',0.0,10000,USAGE); break;
             case 'h': Ethresh=RealOption(argv[arg],'h',0.0,10000,USAGE); break;
	     case 'w': width=IntOption(argv[arg],'w',5,50000,USAGE); break;
             case 'v': verbose=TRUE; break;
             case 'D': seg_seq=FALSE; break;
             case 'd': cmafile=FALSE; break;
	     case 'S': cutscore=IntOption(argv[arg],'S',0,5000,USAGE); mode = 'S'; break;
	     case 'I': cutscore=IntOption(argv[arg],'I',0,100,USAGE); mode = 'I'; break;
             case 'L': mask_dbs='x'; break;
             default: print_error(USAGE);
           }
        }

	// a_type		A=MkAlpha(AMINO_ACIDS,PROT_BLOSUM62);
	a_type	  A=MkAlpha(AMINO_ACIDS,GBLAST_BLOSUM62); // IMPORTANT!!
	char str[100];
	cma_typ	cma = 0;
	Int4	*alnlen=0;
	ss_type data = 0;
	if(cmafile) {
		cma = ReadCMSA2(argv[2],A);
		data= TrueDataCMSA(cma);
	} else data=MakeSeqSet(argv[2],200000,A);
	if(mode == 'I') {
		assert(cmafile); assert(nBlksCMSA(cma) == 1); 
		NEW(alnlen,NSeqsSeqSet(data)+3,Int4);
	}
	ss_type qdata=MkSeqSet(argv[1],A);

  double neglogE=-log10(Ecutoff);
  Int4 N=NSeqsSeqSet(data);
  for(Int4 sq =1; sq <= NSeqsSeqSet(qdata); sq++){
	fprintf(stderr,".");
	e_type qE = SeqSetE(sq,qdata);
	FILE *null_fp=0;
	gpsi_type gpsi(qE,data,Ethresh,Ecutoff,x_parameter,hpsz,1,null_fp);
	if(seg_seq) ProcessSeqPSeg(17,2.2,2.5,100,qE,A);
	double *score= gpsi.FastSearch(top_only,N,T,mode,alnlen);
	success=FALSE;
	double	max=0;
	for(Int4 n=1; n <= N; n++){
	  switch (mode) {
	     case 'E': if(score[n] >= neglogE) success=TRUE; break;
	     case 'S': if(score[n] >= (double) cutscore) success=TRUE; break;
	     case 'I': 
		if(alnlen[n] < LengthCMSA(1,cma)){
		  score[n] = 100.0*score[n]/(double)LengthCMSA(1,cma);
		} else score[n] = 100.0*score[n]/(double)alnlen[n];
		if(score[n] >= (double) cutscore) success=TRUE; break;
	     default: break;
	  }
	  max = MAXIMUM(double,max,score[n]);
	  // if(success) break;
	} free(score);
	fprintf(stderr,"score(%d) = %.1f\n",sq,max);
	if(!success) PutSeq(stdout,qE,A);
   } fprintf(stderr,"\n");
   if(cmafile){
	TotalNilCMSA(cma);
   } else {
	NilSeqSet(data);
   }
   if(alnlen) free(alnlen);
   NilSeqSet(qdata); NilAlpha(A);
   fprintf(stderr,"time = %d seconds\n",time(NULL)-time1);
   if(success) return 0;
   else return 1;
}

