/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "pdb.h"
#include "swaln.h"
#include "cmsa.h"

void    ResidueToName(char r, char *str)
{
        switch (r){
          case 'A': strncpy(str,"ALA",4); break;
          case 'C': strncpy(str,"CYS",4); break;
          case 'D': strncpy(str,"ASP",4); break;
          case 'E': strncpy(str,"GLU",4); break;
          case 'F': strncpy(str,"PHE",4); break;
          case 'G': strncpy(str,"GLY",4); break;
          case 'H': strncpy(str,"HIS",4); break;
          case 'I': strncpy(str,"ILE",4); break;
          case 'K': strncpy(str,"LYS",4); break;
          case 'L': strncpy(str,"LEU",4); break;
          case 'M': strncpy(str,"MET",4); break;
          case 'N': strncpy(str,"ASN",4); break;
          case 'P': strncpy(str,"PRO",4); break;
          case 'Q': strncpy(str,"GLN",4); break;
          case 'R': strncpy(str,"ARG",4); break;
          case 'S': strncpy(str,"SER",4); break;
          case 'T': strncpy(str,"THR",4); break;
          case 'V': strncpy(str,"VAL",4); break;
          case 'W': strncpy(str,"TRP",4); break;
          case 'Y': strncpy(str,"TYR",4); break;
          default: strncpy(str,"XXX",4); break;
        }
}

void	CapitalizeAtom(char *name)
{
   char str[20];

   for(Int4 i=0; name[i]!='\0'; i++){
      if(islower(name[i])) name[i] = toupper(name[i]);
   }
   Int4 len=strlen(name);
   if(len == 4) return;		// e.g., "1HG1"
   else if(isdigit(name[0])){
	if(len == 3){		// e.g., "1HB "
		name[3]=' '; 
	} else if(len == 2){	// e.g.  "1X  "
		name[2] = name[3]=' '; 
	} else print_error("CapitalizeAtom( ) input error 1"); 
   } else if(isalpha(name[0])){			// e.g., "CD1" or "CA" or "C"
	switch (len){
	  case 1:		// e.g., " C  "
	     name[1] = name[0];
	     name[3]=name[2]=name[0]=' '; 
	   break;
	  case 2:		// e.g., " CA "
	     name[2] = name[1];
	     name[1] = name[0];
	     name[3]=name[0]=' '; 
	   break;
	  case 3:		// e.g., " CD1"
	     name[3] = name[2];
	     name[2] = name[1];
	     name[1] = name[0];
	     name[0]=' '; 
	   break;
	  default: print_error("CapitalizeAtom( ) input error 2"); break;
	}
   } else print_error("CapitalizeAtom( ) input error 3"); 
   name[4]=0; return;
}

BooLean	FindAtomPDB(FILE *fp,char AA, Int4 Site, char Chn, char* atm,  pdb_typ P)
// search through pdb file P for atm of residue AA at position Site in chain Chn.
{
	Int4 chain;
	char atm_str[10];
	strncpy(atm_str,atm,5);
	char amino_acid[10];
	ResidueToName(AA,amino_acid);
        Int4    a;
        BooLean success=FALSE;
        atm_typ atom;
	char str[10];
	

	Int4 *natoms=NumAtomsPDB(P);
        for(chain = 1; chain <=  nChainsPDB(P); chain++){
	     char chn=ChainCharPDB(chain,P);
	     if(chn != Chn) continue;
             for(a=1; a<= natoms[chain]; a++){
                atom = AtomPDB(chain, a, P);
		if(ResAtom(atom) == Site && IsResAtom(amino_acid,atom)){
		 // fprintf(fp,"%s; %s\n",amino_acid,atm_str);
		 if(IsAtom(atm_str, atom)){
			PutAtom(fp,atom);
                	success=TRUE;
		 }
		}
             } 
        } 
	return success;
}

#define WUSAGE "Usage: superimpose source_pdb target_pdb source_Triad target_Triad\n\
       Superimposes source_pdb onto target_pdb based on atom triads\n\
       e.g., \"formA.pdb rab27_CDH.pdb W74A.ca:K22A.ca:V24A.ca W73D.ca:K11C.ca:L13C.ca\"\n\
       \n\n"


int	main(int argc,char *argv[])
{ 
	Int4	i,start,arg,n,chain_num=0;
	char	c=' ',chain=' ';
	BooLean	silent = FALSE;
	UInt4	seed=18364592;
	pdb_typ	P;
	Int4	C=1,C1=1,C2=0;

	if(argc < 5) print_error(WUSAGE);
	start = 5; 
        for(arg = start; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(WUSAGE);
           switch(argv[arg][1]) {
                case 'x': c = 'x'; break;
                default: print_error(WUSAGE);
            }
        }
	if(!silent) {
          for(arg=0; arg < argc; arg++) fprintf(stderr,"%s ",argv[arg]); 
	  fprintf(stderr,"\n");
	}
	time_t	time1=time(NULL);
        if(seed == 18364592) seed = (unsigned int) (time(NULL)/2);
	if(!silent) fprintf(stderr,"seed = %d\n",seed);
	sRandom(seed);
	pdb_typ	sourcePDB = MakePDB(argv[1]);
	pdb_typ targetPDB = MakePDB(argv[2]);

	char	to_aa[5],from_aa[5];
	Int4	to_site[5],from_site[5];
	char	to_chn[5],from_chn[5];
	char	to_atm[5][10],from_atm[5][10];

	// source
	if(sscanf(argv[3],"%c%d%c.%[^:]:%c%d%c.%[^:]:%c%d%c.%[^:]",
		&from_aa[1], &from_site[1], &from_chn[1],from_atm[1],
		&from_aa[2], &from_site[2], &from_chn[2],from_atm[2],
		&from_aa[3], &from_site[3], &from_chn[3],from_atm[3]) != 12) print_error(WUSAGE);

	// target
	if(sscanf(argv[4],"%c%d%c.%[^:]:%c%d%c.%[^:]:%c%d%c.%[^:]",
		&to_aa[1], &to_site[1], &to_chn[1],to_atm[1],
		&to_aa[2], &to_site[2], &to_chn[2],to_atm[2],
		&to_aa[3], &to_site[3], &to_chn[3],to_atm[3]) != 12) print_error(WUSAGE);

	CapitalizeAtom(from_atm[1]); CapitalizeAtom(from_atm[2]); CapitalizeAtom(from_atm[3]);
	CapitalizeAtom(to_atm[1]); CapitalizeAtom(to_atm[2]); CapitalizeAtom(to_atm[3]);

	FILE *fp = tmpfile(); 
	for(i=1; i<= 3; i++){
	  if(!FindAtomPDB(fp,from_aa[i], from_site[i],from_chn[i], from_atm[i], sourcePDB)){
		print_error("superimpose input error 1");
	  }
	} rewind(fp);
	pdb_typ FromP = MakePDB(fp); fclose(fp);

	fp = tmpfile(); 
	for(i=1; i<= 3; i++){
	  if(!FindAtomPDB(fp,to_aa[i], to_site[i],to_chn[i], to_atm[i], targetPDB)){
		print_error("superimpose input error 2");
	  }
	} rewind(fp);
	pdb_typ	ToP = MakePDB(fp); fclose(fp);

	fp = tmpfile(); 
	TriadSuperImposePDB(fp,ToP,FromP,sourcePDB);
	PutPDB(fp,targetPDB);
	rewind(fp);
	pdb_typ newP = MakePDB(fp); fclose(fp);
	LabelNullChainsPDB(newP); // relabel chains to be unique.
	PutPDB(stdout,newP);
	
	NilPDB(ToP); NilPDB(FromP);
	NilPDB(newP);
	NilPDB(sourcePDB); NilPDB(targetPDB);
	if(!silent){
	  double runtime=difftime(time(NULL),time1);
	  fprintf(stderr,"\ttime: %0.1f seconds (%0.2f minutes)\n",runtime,runtime/60.0);
	}
	return 0;
}

