/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "residues.h"
#include "alphabet.h"
#include "blosum62.h"
#include "blosum45.h"
#include "blosum35.h"
#include "haussler.h"
#include "sset.h"
#include "dheap.h"
#include "afnio.h"
#include "random.h"
#include "rst_typ.h"
// #include "wgraph.h" // this needs to be converted over for use...

//***************** from ncbi code
static char NCBI_AMINO_ACIDS[] = "ARNDCQEGHILKMFPSTWYVBZX";  

#define PROTEIN_ALPHABET 26

/*underlying frequency ratios for BLOSUM62 as determined by Stephen Altschul;
  Stephen and Jorja Henikoff used different number for B,Z,X*/
static double BLOSUM62_FREQRATIOS[PROTEIN_ALPHABET][PROTEIN_ALPHABET] = {
 {0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
  0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
  0.000, 0.000, 0.000, 0.250},
 {0.000, 3.903, 0.565, 0.868, 0.545, 0.741, 0.465, 1.057, 0.569, 0.632, 0.775,
  0.602, 0.723, 0.588, 0.754, 0.757, 0.613, 1.472, 0.984, 0.936, 0.416, 0.750,
  0.543, 0.747, 0.000, 0.250},
 {0.000, 0.565, 4.438, 0.345, 4.743, 1.335, 0.324, 0.739, 0.925, 0.334, 0.855,
  0.297, 0.405, 4.071, 0.554, 0.944, 0.703, 1.058, 0.826, 0.351, 0.253, 0.750,
  0.409, 1.184, 0.000, 0.250},
 {0.000, 0.868, 0.345, 19.577, 0.301, 0.286, 0.439, 0.420, 0.355, 0.653, 0.349,
  0.642, 0.611, 0.398, 0.380, 0.366, 0.309, 0.738, 0.741, 0.756, 0.450, 0.750,
  0.434, 0.317, 0.000, 0.250},
 {0.000, 0.545, 4.743, 0.301, 7.398, 1.688, 0.299, 0.634, 0.679, 0.339, 0.784,
  0.287, 0.346, 1.554, 0.599, 0.897, 0.573, 0.913, 0.695, 0.337, 0.232, 0.750,
  0.346, 1.382, 0.000, 0.250},
 {0.000, 0.741, 1.335, 0.286, 1.688, 5.470, 0.331, 0.481, 0.960, 0.331, 1.308,
  0.373, 0.500, 0.911, 0.679, 1.902, 0.961, 0.950, 0.741, 0.429, 0.374, 0.750,
  0.496, 4.090, 0.000, 0.250},
 {0.000, 0.465, 0.324, 0.439, 0.299, 0.331, 8.129, 0.341, 0.652, 0.946, 0.344,
  1.155, 1.004, 0.354, 0.287, 0.334, 0.381, 0.440, 0.482, 0.745, 1.374, 0.750,
  2.769, 0.332, 0.000, 0.250},
 {0.000, 1.057, 0.739, 0.420, 0.634, 0.481, 0.341, 6.876, 0.493, 0.275, 0.589,
  0.284, 0.396, 0.864, 0.477, 0.539, 0.450, 0.904, 0.579, 0.337, 0.422, 0.750,
  0.349, 0.503, 0.000, 0.250},
 {0.000, 0.569, 0.925, 0.355, 0.679, 0.960, 0.652, 0.493, 13.506, 0.326, 0.779,
  0.381, 0.584, 1.222, 0.473, 1.168, 0.917, 0.737, 0.557, 0.339, 0.444, 0.750,
  1.798, 1.040, 0.000, 0.250},
 {0.000, 0.632, 0.334, 0.653, 0.339, 0.331, 0.946, 0.275, 0.326, 3.998, 0.396,
  1.694, 1.478, 0.328, 0.385, 0.383, 0.355, 0.443, 0.780, 2.417, 0.409, 0.750,
  0.630, 0.351, 0.000, 0.250},
 {0.000, 0.775, 0.855, 0.349, 0.784, 1.308, 0.344, 0.589, 0.779, 0.396, 4.764,
  0.428, 0.625, 0.940, 0.704, 1.554, 2.077, 0.932, 0.793, 0.457, 0.359, 0.750,
  0.532, 1.403, 0.000, 0.250},
 {0.000, 0.602, 0.297, 0.642, 0.287, 0.373, 1.155, 0.284, 0.381, 1.694, 0.428,
  3.797, 1.994, 0.310, 0.371, 0.477, 0.474, 0.429, 0.660, 1.314, 0.568, 0.750,
  0.692, 0.413, 0.000, 0.250},
 {0.000, 0.723, 0.405, 0.611, 0.346, 0.500, 1.004, 0.396, 0.584, 1.478, 0.625,
  1.994, 6.481, 0.474, 0.424, 0.864, 0.623, 0.599, 0.794, 1.269, 0.610, 0.750,
  0.708, 0.641, 0.000, 0.250},
 {0.000, 0.588, 4.071, 0.398, 1.554, 0.911, 0.354, 0.864, 1.222, 0.328, 0.940,
  0.310, 0.474, 7.094, 0.500, 1.001, 0.859, 1.232, 0.984, 0.369, 0.278, 0.750,
  0.486, 0.946, 0.000, 0.250},
 {0.000, 0.754, 0.554, 0.380, 0.599, 0.679, 0.287, 0.477, 0.473, 0.385, 0.704,
  0.371, 0.424, 0.500, 12.838, 0.641, 0.481, 0.755, 0.689, 0.443, 0.282, 0.750,
  0.363, 0.664, 0.000, 0.250},
 {0.000, 0.757, 0.944, 0.366, 0.897, 1.902, 0.334, 0.539, 1.168, 0.383, 1.554,
  0.477, 0.864, 1.001, 0.641, 6.244, 1.406, 0.966, 0.791, 0.467, 0.509, 0.750,
  0.611, 3.582, 0.000, 0.250},
 {0.000, 0.613, 0.703, 0.309, 0.573, 0.961, 0.381, 0.450, 0.917, 0.355, 2.077,
  0.474, 0.623, 0.859, 0.481, 1.406, 6.666, 0.767, 0.678, 0.420, 0.395, 0.750,
  0.556, 1.133, 0.000, 0.250},
 {0.000, 1.472, 1.058, 0.738, 0.913, 0.950, 0.440, 0.904, 0.737, 0.443, 0.932,
  0.429, 0.599, 1.232, 0.755, 0.966, 0.767, 3.843, 1.614, 0.565, 0.385, 0.750,
  0.557, 0.956, 0.000, 0.250},
 {0.000, 0.984, 0.826, 0.741, 0.695, 0.741, 0.482, 0.579, 0.557, 0.780, 0.793,
  0.660, 0.794, 0.984, 0.689, 0.791, 0.678, 1.614, 4.832, 0.981, 0.431, 0.750,
  0.573, 0.761, 0.000, 0.250},
 {0.000, 0.936, 0.351, 0.756, 0.337, 0.429, 0.745, 0.337, 0.339, 2.417, 0.457,
  1.314, 1.269, 0.369, 0.443, 0.467, 0.420, 0.565, 0.981, 3.692, 0.374, 0.750,
  0.658, 0.444, 0.000, 0.250},
 {0.000, 0.416, 0.253, 0.450, 0.232, 0.374, 1.374, 0.422, 0.444, 0.409, 0.359,
  0.568, 0.610, 0.278, 0.282, 0.509, 0.395, 0.385, 0.431, 0.374, 38.108, 0.750,
  2.110, 0.426, 0.000, 0.250},
 {0.000, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750,
  0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750, 0.750,
  0.750, 0.750, 0.000, 0.250},
 {0.000, 0.543, 0.409, 0.434, 0.346, 0.496, 2.769, 0.349, 1.798, 0.630, 0.532,
  0.692, 0.708, 0.486, 0.363, 0.611, 0.556, 0.557, 0.573, 0.658, 2.110, 0.750,
  9.832, 0.541, 0.000, 0.250},
 {0.000, 0.747, 1.184, 0.317, 1.382, 4.090, 0.332, 0.503, 1.040, 0.351, 1.403,
  0.413, 0.641, 0.946, 0.664, 3.582, 1.133, 0.956, 0.761, 0.444, 0.426, 0.750,
  0.541, 3.893, 0.000, 0.250},
 {0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
  0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
  0.000, 0.000, 0.000, 0.250},
 {0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250,
  0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250, 0.250,
  0.250, 0.250, 0.250, 1.333},
};

const char *UInt4_to_binary(UInt4 x)
{
    // char *b; NEW(b,36,char); 
    static char b[36]; b[0]=0;

    int z;
    // for (z = 2097152; z > 0; z >>= 1) 
    for (z=1; z < 2097152; z <<= 1) 
    {
        strcat(b, ((x & z) == z) ? "1" : "0");
    } return b;
}

void	PutDMPriors(FILE *fp, a_type AB, dmp_typ D)
{
	Int4	i,j,x;
	static char    aa[22] = {"ACDEFGHIKLMNPQRSTVWY"};
	fprintf(fp,"static float Dirichlet30P[%d][21] = {\n",D->L+1);
	fprintf(fp," { 0.0, 0.0500, 0.0500, 0.0500, 0.0500, 0.0500,\n");
        fprintf(fp,"   0.0500, 0.0500, 0.0500, 0.0500, 0.0500,\n");
        fprintf(fp,"   0.0500, 0.0500, 0.0500, 0.0500, 0.0500,\n");
        fprintf(fp,"   0.0500, 0.0500, 0.0500, 0.0500, 0.0500 },\n");

	for(i=0; i < D->L; i++){
	   double d=0.0;
	   fprintf(fp,"{ 0.0, ");
	   for(j=1; j <= nAlpha(AB); j++){
		x = D->let2code[AlphaChar(j,AB)];
		fprintf(fp,"%.6f",D->Distr[i][x]/D->Distr[i][0]);
		// fprintf(fp,"%c=%.2f ",AlphaChar(j,AB),D->Distr[i][x]);
		if(j > 0) d+=D->Distr[i][x];
		if(j==nAlpha(AB)) fprintf(fp,"},\n");
		else if(j%5==0) fprintf(fp,",\n   ");
		else fprintf(fp,", ");
	   }
	   // fprintf(fp," (total=%.3f)\n",d);
	} fprintf(fp," };\n\n\n");
	fprintf(fp,"static float Dirichlet30Mix[%d] = { 0.0, ",D->L+1);
	for(i=0; i < D->L; i++){
	   if(i < D->L-1) fprintf(fp,"%.6f, ",D->Mix[i]); else fprintf(fp,"%.6f };\n",D->Mix[i]);
	} fprintf(fp,"\n\n");
	fprintf(fp,"static float Dirichlet30Conc[%d] = { 0.0, ",D->L+1);
	for(i=0; i < D->L; i++){
	   if(i < D->L-1) fprintf(fp,"%.6f, ",D->Distr[i][0]); else fprintf(fp,"%.6f };\n",D->Distr[i][0]);
	} fprintf(fp,"\n\n");
}

#if 0	// moved to rst_typ.cc; checked on initialization.
void	CheckRST(FILE *fp,char mode,a_type AB)
// check for programing errors...
{
	Int4	r,x,s,t,C,c,n,i,j;
	rst_typ *rst=new rst_typ(mode); // rst->Put(stderr);
	sst_typ **RST=rst->LegalResSets( );
	sst_typ	*xsst,*rsst,sst,usst;
	for(r =1; r < nAlpha(AB); r++) {
	   rsst=RST[r];
	   for(usst=0,C=1,s=1; rsst[s] != 0; s++){
	      sst=rsst[s]; usst=UnionSset(usst,sst);  n=CardSST(sst,AB);
	      if(n < C){		// Check to make sure sets get larger only.
	      	PutSST(stderr,sst,AB);
		fprintf(stderr," current size = %d; last size = %d\n",n,C);
		print_error("FATAL: rst_typ - residues set sizes out of order");
	      } C = n;
	      for(t=s+1; rsst[t] != 0; t++){	// check to ensure no redundant sets.
		if(sst == rsst[t]){
			print_error("FATAL: rst_typ - redundant residue sets");
		} 
	      }
#if 1	// check for same set in
	      PutSST(stderr,sst,AB); fprintf(stderr,"\n");
	      for(x =1; x <= nAlpha(AB); x++){	// Check for symmetry of sets.
		if(x==r) continue;
		if(!MemSset(x,sst)) continue;
	        xsst=RST[x];
	        for(t=1; xsst[t] != 0; t++){ if(xsst[t] == sst) break; }   // should find a match.
		if(xsst[t] == 0){
	      	  PutSST(stderr,xsst[1],AB);
		  fprintf(stderr," (r = %c; s = %d; x = %c; t = %d\n",
			AlphaChar(r,AB),s,AlphaChar(x,AB),t);
	          for(i=0; xsst[i] != 0; i++){ PutSST(stderr,xsst[i],AB); fprintf(stderr," "); } fprintf(stderr,"\n");
		  print_error("FATAL: rst_typ - residue sets asymmetric");
		}
	      }
#endif
	   } if(usst != rsst[0]){	// check the universal set for residue r.
	      	PutSST(stderr,usst,AB); fprintf(stderr," != "); PutSST(stderr,rsst[0],AB); fprintf(stderr,"\n");
		print_error("FATAL: rst_typ - invalid universal residue set");
	     }
	}
}
#endif

void	PutRST(FILE *fp,char mode,double rho,a_type AB)
{
	Int4	s;
	rst_typ *rst=new rst_typ(mode); // rst->Put(stderr);
	//************ Show the prior probabilites, p(A), for pattern sets given rho.
	sst_typ **RST=rst->LegalResSets( );
	Int4 Length=nAlpha(AB);
	sst_typ **SST; NEWP(SST,Length+3,sst_typ);
	for(s=1; s <= nAlpha(AB); s++){ SST[s]=RST[s]; }
	double **Rho=GetRhoCategoricalPriors(fp, Length, rho, SST, AB);
	for(s=1; s <= nAlpha(AB); s++){ free(Rho[s]);	} free(Rho);
	return;

	fprintf(fp,"\n");
	unsigned char r,c;
	for(r =1; r <= nAlpha(AB); r++) {
	     // fprintf(fp,"%c: ",AlphaChar(r,AB));
	     for(Int4 rset=1; RST[r][rset] != 0; rset++){
		sst_typ sst=RST[r][rset];
		BooLean	found=0;
                // for(c = 1; c < r; c++){ if(MemSset(c,sst)) found=1; }
                for(c = nAlpha(AB); c > r; c--){ if(MemSset(c,sst)) found=1; }
		// if(found) continue;
	        fprintf(fp,",{");
                for(c = 1; c <= nAlpha(AB); c++){
                        if(MemSset(c,sst)){ fprintf(fp,"%c",AlphaChar(c,AB)); }
                } fprintf(fp,"}"); 
	     } fprintf(fp,"}\n");
	} fprintf(fp,"\n");fprintf(fp,"\n");
}

//*******************************************************************

#define	USAGE_START	"USAGE: alphabet [options]\n\
   options:\n\
     -t         - test mode\n\
     -p=<char>  - output blosum62P[<char>] array\n\
     -P=<char>  - compute blosum62P[<char>] array\n\
     -g         - output substitition matrix with greater precision\n\
     -x         - dummy\n\n"

/**************************** Global Variables ******************************/
int	main(Int4 argc,char *argv[])
{ 
	Int4	arg,i,s,time1,block1,block2,a=5,b=2;
        static double blosum62freq[21] = {  0.0, /*  X
            C     G     A     S     T     N     D     E     Q     K */
        0.025,0.074,0.074,0.057,0.051,0.045,0.054,0.054,0.034,0.058,
        /*  R     H     W     Y     F     V     I     L     M     P */
        0.052,0.026,0.013,0.032,0.047,0.073,0.068,0.099,0.025,0.039 };
	char	mode=' ',res=' ';

	time1=time(NULL); 
	if(argc < 1) print_error(USAGE_START);
	for(arg = 1; arg < argc; arg++){
	   if(argv[arg][0] != '-') print_error(USAGE_START);
	   switch(argv[arg][1]) {
#if 0
	     case 'P': minprob=RealOption(argv[arg],'P',-5000,5000,USAGE_START);
		break;
             case 'I': 
                  if(sscanf(argv[arg],"-I%d:%d",&left_flank,&right_flank) != 2)
                        print_error(USAGE_START);
                  break;
	     case 'm': if(!isalpha(method=argv[arg][2])) print_error(USAGE_START);
		  break;
	     case 'u': if(!isalpha(mode=argv[arg][2])) print_error(USAGE_START);
		  break;
#endif
	     case 'x': break;
	     case 'P': 
                if(sscanf(argv[arg],"-P=%c",&res) != 1)
                        print_error(USAGE_START);
		mode='P'; 
		break;
	     case 'p': 
                if(sscanf(argv[arg],"-p=%c",&res) != 1)
                        print_error(USAGE_START);
		mode='p'; 
		break;
	     case 'g': mode='g'; break;
	     default: print_error(USAGE_START);
	   }
	}
	a_type	A=0; 

	A = MkAlpha(AMINO_ACIDS,PROT_BLOSUM45,blsm62_ins_emit_nats);
	// A = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62,blsm62_ins_emit_nats);
#if 0
	A = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62,0);
	A = MkAlpha(AMINO_ACIDS,PROT_PAM250,0);
	A = MkAlpha(AMINO_ACIDS,PROT_PAM120,0);
#elif 0
	time1=time(NULL); 
	long double d=1.0;
	UInt4 seed= (UInt4) time1,iter=200000;
	sRandom(seed);
	for(Int4 i=1; i <= iter; i++){
	    d=1.0;
	    for(Int4 j=1; j <= 1000; j++){ d *= (double)Random(	); }
	} fprintf(stderr,"d=%g; log10(d) = %g\n",d,log10(d));
	fprintf(stderr,"\ttime: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	time1=time(NULL); sRandom(seed);
	for(Int4 i=1; i <= iter; i++){
	    d=0.0;
	    for(Int4 j=1; j <= 1000; j++){ d += log10((double)Random(	)); }
	} fprintf(stderr,"d=%g; log10(d) = %g\n",pow(10,d),d);
	fprintf(stderr,"\ttime: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
exit(1);
#endif
	switch(mode){
	 case 'P': // sort
	{
	  Int4	r,r0=AlphaCode(res,A),R=((r0+10) % 20) + 1;
	  double ex=3.0;
	  double P,D,d,d0,total_prob=0.0;
	  for(r=1; r <= nAlpha(A); r++){
		d=blosum62freq[r]; d0=blosum62freq[r0];
		// D=blosum62L[r0][r]*d*d0;
		D=blosum62L[r0][r]*d;
		printf("%d.P(%c,%c) = %.4g; ",r,AlphaChar(r0,A),AlphaChar(r,A),D);
		D=blosum62L[r][r0]*d*d0;
		printf("P(%c,%c) = %.4g\n",AlphaChar(r,A),AlphaChar(r0,A),D);
		total_prob += D;
	  } printf("total prob = %.4f\n",total_prob);
	  for(r=1; r <= nAlpha(A); r++){
		// d=blosum62freq[r]; 
		P=blosum62L[r][r0]*blosum62freq[r]; 	// == L(r/r0)*p(r) == p(r,r0)/p(r0).
		printf("P(%c|%c) = %.4g; ",AlphaChar(r,A),AlphaChar(r0,A),P/total_prob);
		// d=blosum62freq[r]; 
		D=blosum62L[r][R]*blosum62freq[r];		// == L(r/R)*p(r) == p(r,R)/p(R).
		printf("LR(%c/%c) = %.5g; ",AlphaChar(r0,A),AlphaChar(R,A),P/D);
		printf(" LR(%c/%c) = %.5g\n",AlphaChar(r0,A),AlphaChar(R,A),blosum62L[r0][r]/blosum62L[R][r]);
	  }
	} break;
	 case 'p': // sort
	{
	  Int4 r0=AlphaCode(res,A);
	  double ex=3.0;
	  double total_prob=0.0;
	  for(Int4 r=1; r <= nAlpha(A); r++){
		if(r != r0) total_prob += pow(blosum62P[r0][r],ex);
		double d=blosum62freq[r],d0=blosum62freq[r0];
		double D=blosum62L[r0][r]*d*d0;
		printf("%d.P(%c,%c) = %.4g; ",r,AlphaChar(r0,A),AlphaChar(r,A),D);
		D=blosum62L[r][r0]*d*d0;
		printf("P(%c,%c) = %.4g\n",AlphaChar(r,A),AlphaChar(r0,A),D);
	  }
	  printf("total prob = %.4f\n",total_prob);

	  for(Int4 r=1; r <= nAlpha(A); r++){
	     total_prob=0.0;
	     for(Int4 rx=1; rx <= nAlpha(A); rx++){
		total_prob += blosum62P[r][rx];
	     } printf("total prob = %.4f\n",total_prob);
	  }
	  for(Int4 r=1; r <= nAlpha(A); r++){
		printf("%c vs %c = %.4f (%.4f)\n", res,
			AlphaChar(r,A),
			// blosum62P[r0][r],
			blosum62[r0][r],blosum62L[r0][r]);
			// pow(blosum62P[r0][r],ex),
			// pow(blosum62P[r0][r],ex)/total_prob); // (blosum62P[r0][r0]));
			// (blosum62P[r0][r])/total_prob); // (blosum62P[r0][r0]));
	  }
	  printf("\n");
#if 0
	  for(Int4 r=1; r <= nAlpha(A); r++){
		printf("%.4f,", res,pow(blosum62P[r0][r],ex)/total_prob);
	  }
	  printf("\n");
#endif
	} break;
	 case 'g': // sort
	{
	  double CutOff=0.0;
	  //double CutOff=-0.2;
	  Int4	size_dH=500,item,r,r2,r3,r4;
	  char	tmp_str[50],**ResSets;
	  double Score,ave,ave2,sum,sum2;
	  dh_type dH = dheap(size_dH + 2,4);
	  NEWP(ResSets, size_dH + 3,char);
	
	  for(item=0,r=1; r <= nAlpha(A); r++){
	    printf("%c: ", AlphaChar(r,A));
	    for(r2=r+1; r2 <= nAlpha(A); r2++){
		if(r != r2 && (Score=blosum62[r][r2]) >= CutOff){
		   item++;
		   if(item > size_dH) print_error("size_dH overflow error");
		   printf("\"%c%c\" %5.3f ",
			AlphaChar(r,A), AlphaChar(r2,A),Score);
		   sprintf(tmp_str,"%c%c",AlphaChar(r,A), AlphaChar(r2,A));
		   insrtHeap(item,-(keytyp)Score,dH); 
		   // caution: eventually need to free up strings if exceed size...
		   ResSets[item]=NewString(tmp_str);
	           for(r3=r2+1; r3 <= nAlpha(A); r3++){
		      if(blosum62[r][r3] >= CutOff && blosum62[r2][r3] >= CutOff){
			item++;
			if(item > size_dH) print_error("size_dH overflow error");
			sum=blosum62[r][r2] + blosum62[r][r3] + blosum62[r2][r3]; ave= sum/3.0;
			insrtHeap(item,-(keytyp)ave,dH);
			sprintf(tmp_str,"%c%c%c",AlphaChar(r,A), AlphaChar(r2,A), AlphaChar(r3,A));
		   	ResSets[item]=NewString(tmp_str);
			printf("(\"%c%c%c\" [%.3f]) ", 
				AlphaChar(r,A), AlphaChar(r2,A), AlphaChar(r3,A),ave);
	                for(r4=r3+1; r4 <= nAlpha(A); r4++){
		          if(blosum62[r][r4] >= CutOff && blosum62[r2][r4] >= CutOff && 
					blosum62[r3][r4] >= CutOff){
				item++;
				if(item > size_dH) print_error("size_dH overflow error");
				sum2= sum + blosum62[r][r4] + blosum62[r2][r4] + blosum62[r3][r4];
				Score= sum2/6.0;
				insrtHeap(item,-(keytyp)Score,dH);
				sprintf(tmp_str,"%c%c%c%c",
					AlphaChar(r,A),AlphaChar(r2,A),
					AlphaChar(r3,A),AlphaChar(r4,A));
		   		ResSets[item]=NewString(tmp_str);
				printf("(\"%c%c%c%c\" [%.3f]) ", 
					AlphaChar(r,A),AlphaChar(r2,A),
					AlphaChar(r3,A),AlphaChar(r4,A),Score);
			  }
		        }
		      }
		   }
		}
	    }
	    printf("\n");
	  }
	  printf("\n");
	  for(Int4 I=1; !emptyHeap(dH); I++){
		Score = -minkeyHeap(dH);
		item=delminHeap(dH);
		printf("\"%s\" %.3f\n",ResSets[item],Score);
	  }
	  free(ResSets);
	  Nildheap(dH);
	}
	  break;
	 default:
	  for(char c='!'; c <= '~'; c++) fprintf(stdout,"%c = %d\n",c,AlphaCode(c,A));
	  for(Int4 r=0; r <= nAlpha(A)+1; r++)
		fprintf(stdout,"%d = %c = %c\n",r,AlphaChar(r,A),AlphaCharLow(r,A));
	  fprintf(stdout,"\n\n\t%d = %c\n",nAlpha(A)+1,AlphaChar(nAlpha(A)+1,A));
	  fprintf(stdout,"\t%c = %d\n",'-',AlphaCode('-',A));
	  PutAlpha(stdout,A);
#if 1	  // 
	  sst_typ ust[25];
	  for(Int4 r=1; r <= nAlpha(A); r++){
	     ust[r]=0;
	     for(Int4 s=1; s <= nAlpha(A); s++){
		// double d=log(blosum45L[r][s]);
		// double d=log(blosum62L[r][s]);
		double d=log(blosum35L[r][s]);
		if(d >= 0.0){
			fprintf(stdout,"%d.%c vs %c: %.3f\n",r,AlphaChar(r,A),AlphaChar(s,A),d);
			if(r != s) { sst_typ xst = SsetLet(s); ust[r]=UnionSset(ust[r],xst); }
		}
	     } fprintf(stdout,"\n\"%c",AlphaChar(r,A)); PutSST(stdout,ust[r],A); fprintf(stdout,"\"\n\n");
	     { sst_typ xst = SsetLet(r); ust[r]=UnionSset(ust[r],xst); }
	  } fprintf(stdout,"\n");
	  for(Int4 r=1; r <= nAlpha(A); r++){
	     sst_typ ist=ust[r];
	     fprintf(stdout,"\n%c:",AlphaChar(r,A));
	     for(Int4 s=1; s <= nAlpha(A); s++){
		if(r == s) continue;
		ist=IntersectSset(ust[r],ust[s]);
		if(ist != 0){
		  for(Int4 t=s+1; t <= nAlpha(A); t++){
			if(r == t) continue;
	     		ist=IntersectSset(ust[r],IntersectSset(ust[s],ust[t]));
			if(CardSST(ist,A) == 3 && MemSset(r,ist) && MemSset(s,ist) && MemSset(t,ist)){
			   fprintf(stdout," \"",AlphaChar(r,A)); PutSST(stdout,ist,A); fprintf(stdout,"\" "); 
			}
			if(ist != 0) for(Int4 u=t+1; u <= nAlpha(A); u++){
			   if(r == u) continue;
	     		   ist=IntersectSset(IntersectSset(ust[r],ust[u]),IntersectSset(ust[s],ust[t]));
			   if(CardSST(ist,A) >= 4 && MemSset(r,ist) && MemSset(s,ist) && MemSset(t,ist) && MemSset(u,ist)){
			       fprintf(stdout," \"",AlphaChar(r,A)); PutSST(stdout,ist,A); fprintf(stdout,"\" "); 
			   }
			}
		  }
		}
	     } fprintf(stdout,"\n");
	  }
#if 1	  // minimum set...
	// PutRST(stderr,'M',1e-5,A);  // not sure that I want this mode.
	// PutRST(stderr,'L',1e-5,A);
	PutRST(stderr,'R',1e-5,A);
	// CheckRST(stderr,'G',A);
	fprintf(stderr,"\n\n");
	PutRST(stderr,'G',1e-5,A);
	fprintf(stderr,"\n\n");
	PutRST(stderr,'G',2e-5,A);
	fprintf(stderr,"\n\n");
	PutRST(stderr,'G',1e-3,A);
	fprintf(stderr,"\n\n");
	PutRST(stderr,'G',0.25,A);
#else
	PutRST(stderr,'L',0.25,A);
	PutRST(stderr,'L',0.02,A);
	PutRST(stderr,'L',0.01,A);
	PutRST(stderr,'L',0.001,A);
	PutRST(stderr,'L',0.0001,A);
	PutRST(stderr,'L',1e-5,A);
	PutRST(stderr,'L',1e-6,A);
	PutRST(stderr,'L',1e-7,A);
	PutRST(stderr,'L',1e-8,A);
	PutRST(stderr,'L',1e-9,A);
	PutRST(stderr,'L',1e-10,A);
#endif
#endif
	 break;
	}
	// blosum62[21][21] = 
#if 0
	for(unsigned char r=0; r <= nAlpha(A); r++){
	  for(unsigned char s=0; s <= nAlpha(A); s++){
	  }
	}
#endif
#if 0	// print out sset type ...
	fprintf(stderr,"FOPEN_MAX = %d\n",FOPEN_MAX);
	for(unsigned char r=0; r <= nAlpha(A); r++){
		sst_typ sst[30];
		sst[r]=SsetLet(r);
#if 0
		fprintf(stderr,"%c = %d\n",AlphaChar(r,A),sst[r]);
#else
		fprintf(stderr,"%c = ",AlphaChar(r,A));
		fprintf(stderr,"%s\n",UInt4_to_binary(sst[r]));
#endif
		for(unsigned char x=0; x <= nAlpha(A); x++){
		   sst_typ xst = SsetLet(x);
		   sst_typ ust=UnionSset(sst[r],xst);
#if 0
		   fprintf(stderr,"%c + %c = %d\n",AlphaChar(r,A),AlphaChar(x,A),ust);
#else
		   fprintf(stderr,"%c + %c = ",AlphaChar(r,A),AlphaChar(x,A));
		   fprintf(stderr,"%s\n",UInt4_to_binary(ust));
#endif
		}
	}
#endif
	// assert(nAlpha(A) == 50);
#if 0	// dmp_typ
	dmp_typ dmp=MkDMPriors(TRUE);
	PutDMPriors(stderr,A,dmp);
	NilDMPriors(dmp);
#endif
#if 0
	for(Int4 x=1; x <= 100000; x++){
	  rst_typ *rst=new rst_typ('R',A); // rst->Put(stderr);
	  delete rst;
	}
#else
	rst_typ *rst=new rst_typ('G',A); rst->Put(stderr); delete rst;
#endif
	NilAlpha(A);
	fprintf(stderr,"\ttime: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	return 0;
}



