/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "afnio.h"
#include "seqset.h"
#include "residues.h"
#include "histogram.h"
#include "tax_typ.h"

#define USAGE_START "usage: phylum_seq fasta_file [options]\n\
      options:\n\
        -F       Print the first sequence found from each phylum.\n\
        -ids     Print short sequence ids.\n\
        -K<str>  Show info for sequences from kingdoms in <str>(MFEVBA).\n\
        -k<str>  Print sequences from kingdoms <str>(MFEVBA).\n\
        -L       Print the labeled sequences.\n\
        -N       Print the number of kingdoms represented by the sequences.\n\
        -N<int>  Print the number of kingdoms represented and\n\
                  return 1 if the number of phyla is >= <int>\n\
        -n<char> Print sequences NOT from kingdom <char>(MFEVBA).\n\
        -P       Print the number of phyla.\n\
        -p<str>  Print sequences from phylum <str>.\n\
        -r       Print sequences without phylogenetic information in defline.\n\
        -R<str>  Print sequences NOT from phylum <str>.\n\
	-status  Check the status of the first sequences in file.\n\
	-t       Print sequences as a fake taxonomy file with fake taxonomy data.\n\
	-T<str>:<char>  Print seqs with phyum <str> & kingdom <char>.\n\
        -u       Print the unlabeled sequences.\n\
        -x       dummy.\n\n"

//         -P<str>  Print sequences from phylum <string>(e.g. CHORDATA).\n


int	main(int argc,char *argv[])
{ 
	Int4	i,arg,min_num=0;
	char	c=' ',Kingdom=' ',*Phylum=0,phylum_str[100],k;
	char	*Kingdoms=0;
	a_type	A;
	ss_type data = NULL;
	e_type	Seq;
	UInt4	*iskingdom,*isphylum;

	if(argc < 2) print_error(USAGE_START);
        for(arg = 2; arg < argc; arg++){
           if(argv[arg][0] != '-') print_error(USAGE_START);
           switch(argv[arg][1]) {
		case 'F': c = 'F'; break;
		case 'i': c = 'i'; break;
		case 't': c = 't'; break;
		case 'T': c = 'T';
			if(sscanf(argv[arg],"-T%[^:]:%c",phylum_str,&k)!=2) {
				print_error(USAGE_START);
			}
			Phylum=phylum_str; Kingdom=k;
			break;
		case 'r': c = 'r'; break;
		case 'N': {
		     c = 'N'; 
		     if(isdigit(argv[arg][2])) min_num=IntOption(argv[arg],'N',1,100,USAGE_START); 
		   } break;
		// case 'n': c = 'n'; 
		case 'P': c = 'P'; break;
		case 'L': c = 'L'; break;
		case 'u': c = 'u'; break;
		case 'p': c = 'p'; Phylum=argv[arg]+2; break;
		case 'R': c = 'R'; Phylum=argv[arg]+2; break;
		case 'K': c = 'K'; Kingdoms=argv[arg]+2; break;
		case 'k': c = 'k'; Kingdoms=argv[arg]+2; break;
		case 'n': c = 'n'; Kingdom=argv[arg][2]; break;
	  	case 's': 
		   {
              		if(strcmp("-status",argv[arg]) != 0) print_error(USAGE_START);
	      		else c='s'; // == default...
		   }
	    	   break;
                default: print_error(USAGE_START);
            }
        }
	A = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62); /****/
#if 0
FILE *fptr=open_file(argv[1],"","r");
e_type Sq=ReadSeq(fptr, 1, 353,A); fclose(fptr);
PutSeq(stderr,Sq,A); 
NilSeq(Sq); NilAlpha(A); return 0;
#endif
	data = MakeSeqSet(argv[1],200000, A);
	// PutSeqSetEs(stderr,data);
	// data = SeqSet(argv[1],A);
	char str[MAX_SEQ_DEFLINE_LENG+100],*buffer=0;
	switch(c){
	  case 'R': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
	        char	*phylum=PhylumSeq(Seq);
		if(phylum && strcmp(phylum,Phylum)!=0){
 		  	PutSeq(stdout,Seq,A);
		}
	    } break;
	  case 'i': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		if(i > 1) printf(",");
		PutShortSeqID(stdout,Seq);
		
	    } printf("\n"); break;
	  case 'u': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		if(!LabeledSeq(Seq)) PutSeq(stdout,Seq,A);
	    } break;
	  case 'L': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		if(LabeledSeq(Seq)) PutSeq(stdout,Seq,A);
	    } break;
	  case 'p': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
	        char	*phylum=PhylumSeq(Seq);
		if(phylum && strcmp(phylum,Phylum)==0){
 		  	PutSeq(stdout,Seq,A);
		}
	    } break;
	  case 'T': 
	    {
	      for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		TaxAssignSeq(Phylum,Kingdom,Seq);
 		PutSeq(stdout,Seq,A);
	      }
	    } break;
	  case 't': 
	   {
	    fprintf(stdout,"<TaxSeqs(%d;%d)={",NSeqsSeqSet(data),NSeqsSeqSet(data));
	    for(i=1; i < NSeqsSeqSet(data); i++){
		    Seq = SeqSetE(i,data);
		    if(KingdomSeq(Seq)) fprintf(stdout,"%s(%c1)=%d;",PhylumSeq(Seq),
			KingdomSeq(Seq),LenSeq(Seq));
		    else fprintf(stdout,"NotKnown(X1)=%d;",LenSeq(Seq));
	    } 
	    Seq = SeqSetE(i,data);
	    if(KingdomSeq(Seq)) fprintf(stdout,"%s(%c1)=%d};\n\n",
			PhylumSeq(Seq),KingdomSeq(Seq),LenSeq(Seq));
	    else fprintf(stdout,"NotKnown(X1)=%d};\n\n",LenSeq(Seq));
	    NEW(buffer,MaxSeqSeqSet(data)+3,char);
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		StrSeqID(str,MAX_SEQ_DEFLINE_LENG,Seq);
		fprintf(stdout,">%s {|%d(%d)|<%s(%c)>}",str,OffSetSeq(Seq),TaxIdentSeq(Seq),
						PhylumSeq(Seq),KingdomSeq(Seq));
		StrSeqDescript(str,MAX_SEQ_DEFLINE_LENG,Seq);
		fprintf(stdout,"%s\n",str);
		SeqToString(buffer, Seq, A);
		for(Int4 j=0; j < LenSeq(Seq); j++){
			if(j%60 == 59) fprintf(stdout,"\n");
	       		fprintf(stdout,"%c",buffer[j]);
		} fprintf(stdout,"\n");

	    } fprintf(stdout,">.\n"); } break;
	  case 'r': 
	    NEW(buffer,MaxSeqSeqSet(data),char);
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		StrSeqID(str,MAX_SEQ_DEFLINE_LENG,Seq);
		fprintf(stdout,">%s ",str);
		StrSeqDescript(str,MAX_SEQ_DEFLINE_LENG,Seq);
		fprintf(stdout,"%s\n",str);
		SeqToString(buffer, Seq, A);
		for(Int4 j=0; j< LenSeq(Seq); j++){
			if(j%60 == 59) fprintf(stdout,"\n");
	       		fprintf(stdout,"%c",buffer[j]);
		} fprintf(stdout,"\n");

	    } break;
	  case 'n': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		char kingdom=KingdomSeq(Seq);
		if(kingdom != Kingdom){
 		  PutSeq(stdout,Seq,A);
		}
	    } break;
	  case 'K': case 'k': 
	    for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		char kingdom=KingdomSeq(Seq);
		// if(kingdom == Kingdom)
		if(memchr(Kingdoms,kingdom,strlen(Kingdoms))){
		  if(c=='K'){
			printf("%d: ",i);
			PutSeqInfo(stdout,Seq);
			// fprintf(stdout,"\n");
 		  } else PutSeq(stdout,Seq,A);
		}
	    } break;
	  case 'P': 
	    { PutPhylaSeqSet(stdout,0,data); }
	    break;
	  case 'F': 
	    { PutPhylaSeqSet(stderr,stdout,data); }
	    break;
	  case 'N': 
	    {
	      NEW(iskingdom,255,UInt4); NEW(isphylum,256,UInt4);
	      Int4 num_kingdoms=0,num_phyla=0;
	      for(i=1; i <= NSeqsSeqSet(data); i++){
		Seq = SeqSetE(i,data);
		char kingdom=KingdomSeq(Seq);
		if(iskingdom[kingdom] == 0){ num_kingdoms++; }
		iskingdom[kingdom]++;
	      } 
	      fprintf(stdout,"%s(%d seqs): num_kingdoms = %d (",argv[1],NSeqsSeqSet(data),num_kingdoms);
	      for(c = 'A'; c <= 'Z'; c++) if(iskingdom[c]) fprintf(stdout,"%c%d",c,iskingdom[c]);
	      if(iskingdom[0]) fprintf(stdout,"U%d",iskingdom[0]);
	      fprintf(stdout,")\n");
	      if(min_num <= num_kingdoms) return 1; 
	    } break;
	  default:
	  for(i=1; i <= NSeqsSeqSet(data); i++){
	    e_type	QuerySeq=SeqSetE(i,data);
	    char	*phylum=PhylumSeq(QuerySeq);
	    if(phylum==0){ return 0; }
	    else {
		char kingdom=KingdomSeq(QuerySeq);
		fprintf(stdout,"Seq %d: phylum = %s(%c)\n",i,phylum,kingdom);
		return 1;
	    }
	  }
	}
	if(buffer) free(buffer);
	NilSeqSet(data); NilAlpha(A); 
	return 0;
}

