/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "cma_gmb.h"
#include "gmb_typ.h"
#include "gpr_typ.h"
#include "gsm_typ.h"
#include "gibbs.h"

#define	USAGE_START	"USAGE: gambit cmafile [options]\n\
   options:\n\
     -adj=<real>  - adjust sequence weights (range: 0.01 .. 1; default: 1.0)\n\
     -aln=<str>   - align sequences in the database <str> against the gambit hmm\n\
     -C           - sample on closely-related clusters\n\
     -C<int>      - sample on Clusters with \% similarity cutoff = <int>\n\
     -dbs=<str>   - search database using gambit hmm\n\
     -dms=<char>  - Choose the number of Dirichlet mixture components (default: 20)\n\
                     'T': 20 components\n\
                     'f': 52 components\n\
                     'F': 58 components\n\
                     'S': 72 components\n\
                     'M': 110 components\n\
                     'L': 134 components\n\
     -lpr         - print out LLR for a msa\n\
     -p<int1>:<int2>:<int3>:<int4>  - priors for computing transition probabilities\n\
                  int1 = average residue spacing between insertions (default: 1000)\n\
                  int2 = average length of insertion extensions (default: 3)\n\
                  int3 = average residue spacing between deletions (default: 1000)\n\
                  int4 = average length of deletion extensions (default: 1)\n\
     -psi         - use psi-blast scoring method instead of Dirichlet mixture priors\n\
     -seed=<int> - random generator seed\n\
     -temp=<real> - sampling temperature for -dbs option (default: 0.0)\n\
     -u<mode>    - alignment mode: 'S' = sample; 'M' = MLE; 'C' = check (default 'S')\n\
                                  'P' = progressive clustered sampling\n\
     -wt=<real>  - relative weight on indel penalty priors (default: 1.0)\n\
     -x          - dummy\n\n"

/**************************** Global Variables ******************************/
inline	double  betadev(Int4 a, Int4 b, Int4 *seed)
{ double  x = gamdev(a,seed); return x/(x+gamdev(b,seed)); }

int	main(Int4 argc,char *argv[])
{ 
	Int4	arg,i,j,s,time1,similarity=0;
	// Int4	aa_per_io=1000,aa_per_do=1000,exp_ie=3,exp_de=1;
	Int4	aa_per_io=75,aa_per_do=100,exp_ie=1,exp_de=1;
	char 	method='S',mode='S',str[200],*dbs=0,*aln=0; 
     	// method = alignment method: 'E' = gapped blast; 'S' = \% similarity (default S).
	a_type	AB;
	cma_typ	cma;
	char	dms_mode=' ';	// 20 original components.
	FILE	*fp;
	UInt4 seed=18364592;
	BooLean	input_priors=FALSE,print_lpr=FALSE;
	double	temp=-1.0,prior_wt=0,AdjSqWt=0.0;

// 	std::cerr << "DEBUG START" << std::endl;
	time1=time(NULL); 
	if(argc < 2) print_error(USAGE_START);
	for(arg = 2; arg < argc; arg++){
	   if(argv[arg][0] != '-') print_error(USAGE_START);
	   switch(argv[arg][1]) {
	     case 'a': 
		if(sscanf(argv[arg],"-aln=%s",str) == 1) aln=AllocString(str); 
		else if(sscanf(argv[arg],"-adj=%lf",&AdjSqWt) == 1){
			if(AdjSqWt < 0.01 || AdjSqWt >= 1.0) print_error(USAGE_START);
		} else  print_error(USAGE_START);
		break;
	     case 'C': 
		if(argv[arg][2] != 0) similarity=IntOption(argv[arg],'C',10,100,USAGE_START); 
		else similarity=-1; break;
	     case 'd': 
		if(sscanf(argv[arg],"-dbs=%s",str) == 1) dbs=AllocString(str); 
		else if(sscanf(argv[arg],"-dms=%c",&dms_mode) == 1){
		   if(strchr("TfFSML",dms_mode) == NULL) print_error(USAGE_START);
		} else  print_error(USAGE_START);
		break;
             case 'l': 
		if(strcmp("-lpr",argv[arg])==0) { print_lpr=TRUE; }
		break;
             case 'p': 
		if(strcmp("-psi",argv[arg])==0) { dms_mode='P'; }
		else {

		  input_priors=TRUE;
		  if(sscanf(argv[arg],"-p%d:%d:%d:%d",&aa_per_io,&exp_ie,&aa_per_do,&exp_de) != 4)
                                        print_error(USAGE_START); 
                  if(aa_per_io <= 0) print_error(USAGE_START);
                  if(aa_per_do <= 0) print_error(USAGE_START);
                  if(exp_ie <= 0) print_error(USAGE_START);
                  if(exp_de <= 0) print_error(USAGE_START);
		} break;
	     case 'u': if(!isalpha(mode=argv[arg][2])) print_error(USAGE_START);
		  break;
             case 's': if(sscanf(argv[arg],"-seed=%d",&seed)!=1)
                        print_error(USAGE_START); break;
             case 't': if(sscanf(argv[arg],"-temp=%lf",&temp)!=1)
                        print_error(USAGE_START); break;
	     case 'w':
                  if(sscanf(argv[arg],"-wt=%lf",&prior_wt) == 1){
			fprintf(stderr,"prior_wt = %g\n",prior_wt);
                        if(prior_wt < 0.01 || prior_wt > 1000) print_error(USAGE_START);
                  } else print_error(USAGE_START);
	     case 'x': break;
	     default: print_error(USAGE_START);
	   }
	}
	fp = open_file(argv[1],".cmd","w");
	for(i = 0; i < argc; i++) { if(argv[i][1] != ' ') fprintf(fp,"%s ",argv[i]); }
	if(seed == 18364592) {  // not provided by user
        	seed=(UInt4) time(NULL)/2; // seed = (UInt4) time(NULL);
        	fprintf(fp,"-seed=%d\n",seed);
   	} else fprintf(fp,"\n"); fclose(fp);

        sRandom(seed);
//========================================================================
	AB = MkAlpha(AMINO_ACIDS,PROT_BLOSUM62);
	cma=ReadCMSA2(argv[1],AB); 
	if(cma == 0) print_error("input cmafile read failed");

    char Option=1;	// test subalignment sampling.
    // Option=2;
    // Option=6;
    // Option=10;
    Option=9;	// sample sticky and sample...
    // Option=0;
    // Option=3;
    Option='C';
    Option='D';
    Option='E';
    Option='B';
    Option='G';
    Option='A';
    Option='F';
    // Option=7;
    // Option=0;	// run default.
    if(aln || dbs) Option=0;
    if(print_lpr) Option='L';
    switch(Option){
//====================================================================
    case 'G':	
	 {
	      char	StMd=' ';
	      double	bild_cut=-3.2,bild_add=-3.0,afrq=0.30,rfrq=0.40;  
	      cma_typ	rcma;
	      double	MaxFrctn=0.01,Temp=0.0;
	      prior_wt=0.90;
              gpr_typ gpr("test",1,1);
              gpr.SetParameters(aa_per_io,aa_per_do,exp_ie,exp_de,prior_wt,dms_mode);
              gpr.SetAddRmCols(bild_cut,bild_add,Temp,afrq,rfrq); gpr.SetSticky(2,StMd);
              Int4 nAddRms=gpr.DoAddRmColumns(stdout,cma,1);        // need to first add columns (for 1 cycle).
   		sprintf(str,"%s_g",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
	} break;
    case 'F':	// test out subalignment sampling.
	{
		Int4	MinSticky=5;
		double	MaxFrctn=0.01,Temp=0.0;
		// char   strategy[]=" RORO ";
		// char   strategy[]=" RO ";
		// char   strategy[]=" RTNFSXV ";
		char   strategy[]=" RIV ";
		dms_mode='F';
		// cma_typ xcma=CopyCMSA(cma);
		aa_per_io=20,aa_per_do=150,exp_ie=1,exp_de=1;
		cma_typ xcma=cma,rcma;
		gss_typ	*gss=gssCMSA(cma);
		Int4	b,nblk=15,*len; NEW(len,nblk+3,Int4);
		for(b=1; b <= nblk; b++) len[b]=8;
		char	StMd=' ';
	      double	bild_cut=-3.2,bild_add=-3.0,afrq=0.30,rfrq=0.40;  
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
		if(prior_wt > 0) gmb->SetPriorWt(prior_wt); 
		gmb->SetMaxIter(1);
	   for(i=1; isalpha(strategy[i]); i++){
		gmb->SampleStickyTogether(stdout, MinSticky, MaxFrctn,str,Temp,strategy[i]);
		cma=gmb->RtnCMSA();
	   } 
   		// sprintf(str,"%s_%c",argv[1],strategy[i]); 
   		sprintf(str,"%s_s",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
		delete gmb;
		// NilCMSA(rcma);
	} break;
    case 'E':	// test out subalignment sampling.
	{
		char *Argv[5];
		int     Argc=2; Argv[0]=0; Argv[1]=argv[1]; Argv[2]=AllocString("junk_dbs");
#if 0
		gsm_typ *gsm= new gsm_typ(USAGE_START,Argc,Argv, AB);
		sprintf(str,"%s_gsmB.cma",argv[1]); WriteCMSA(str,cma);
		// gsm->RefineInputAln(cma);
		if(gsm->GappedAln(cma)) InitMAPCMSA(cma);
		   { sprintf(str,"%s_gsmA.cma",argv[1]); WriteCMSA(str,cma); }
		delete gsm;
#else
		Int4	timeI=time(NULL);
		double	T=300,WtSq,lpr0,lpr;
		dms_mode='T';
	aa_per_io=20;aa_per_do=100;exp_ie=1;exp_de=1;
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
		if(prior_wt > 0) gmb->SetPriorWt(prior_wt); 
		gmb->SetMaxIter(3);
		sprintf(str,"%s_gsmA.cma",argv[1]); 
#if 0
		cma_typ rcma=gmb->Sample(str,' ',0,T,T,30);
#else
		gmb->SampleByLayers(50,T,lpr0,lpr,0);
		cma=gmb->RtnCMSA(); // WARNING: cma has changed...
		cma_typ rcma=cma;
#endif
		WriteCMSA(str,rcma);
		Int4 dt=time(NULL)-timeI; WtSq=gmb->RtnWtNumSeqs();
	        fprintf(stdout," %.1f wt.sq. (%d secs)(%.1f K)\n", WtSq,dt,T);
		delete gmb; // NilCMSA(xcma);
#endif
		
	} break;
    case 'D':	// test out subalignment sampling.
	{
		double	lpr0,lpr,d,pfrq=0.95,WtSq,T=150.0;
		dms_mode='f';
		Int4	timeI,stage=0;
	aa_per_io=20;aa_per_do=100;exp_ie=1;exp_de=1;
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
		gmb->SetMaxIter(3);
		timeI=time(NULL);
		gmb->SampleByLayers(50,T,lpr0,lpr,stage); // lpr=gmb->RtnMap();
		cma=gmb->RtnCMSA(); d=lpr-lpr0;
		fprintf(stdout,"%d: SamplePurged %d (%.2f)(%d cols): LLR = %.2f (%.2f);",
                        i,stage,pfrq,LengthCMSA(1,cma),lpr,d);
   		sprintf(str,"%s_p",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
		Int4 dt=time(NULL)-timeI; WtSq=gmb->RtnWtNumSeqs();
	        fprintf(stdout," %.1f wt.sq. (%d secs)(%.1f K)\n", WtSq,dt,T);
		delete gmb; // NilCMSA(xcma);
	   
	} break;
    case 'C':	// test out subalignment sampling.
	{
		double	lpr0,lpr,d,pfrq=0.95,WtSq,T=200;
		dms_mode='T';
		Int4	timeI,stage=0;
          for(i=1; i <= 50; i++){
	    for(stage=0; stage <= 1; stage++){
		cma_typ xcma=cma; // CopyCMSA(cma);
		timeI=time(NULL);
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
		gmb->SetMaxIter(3);
		d=gmb->SamplePurged(30,pfrq,T,lpr0,lpr,stage); // lpr=gmb->RtnMap();
		fprintf(stdout,"%d: SamplePurged %d (%.2f)(%d cols): LLR = %.2f (%.2f);",
                        i,stage,pfrq,LengthCMSA(1,cma),lpr,d);
		Int4 dt=time(NULL)-timeI; WtSq=gmb->RtnWtNumSeqs();
	        fprintf(stdout," %.1f wt.sq. (%d secs)(%.1f K)\n", WtSq,dt,T);
		delete gmb; // NilCMSA(xcma);
		// if(d < 0.0) exit(1);
break;
	    }
break;
	  }
	   	// exit(1);
	} break;
    case 'B':	// test out subalignment sampling.
	{
		Int4	MinSticky=2;
		// double	MaxFrctn=0.50,Temp=150.0;
		// double	MaxFrctn=0.10,Temp=0.0;
		double	MaxFrctn=0.01,Temp=0.0;
		// char   strategy[]=" RSCTNXVIWcOPD ";
		// char   strategy[]=" PTNRFXD ";
		// char   strategy[]=" WCO ";
		// char   strategy[]=" D ";
		char   strategy[]=" RTNFSXV ";
		// char   strategy[]=" PS "; MaxFrctn=0.10; Temp=100.0;
		// char   strategy[]=" W "; MaxFrctn=0.15; Temp=0.0;
		// char   strategy[]=" TNRFD ";
		// char   strategy[]=" CTNXVIWD ";
		dms_mode='F';
		// dms_mode='S';
		// cma_typ xcma=CopyCMSA(cma);
		aa_per_io=20,aa_per_do=120,exp_ie=1,exp_de=1;
		cma_typ xcma=cma,rcma;
		gss_typ	*gss=gssCMSA(cma);
		Int4	b,nblk=15,*len; NEW(len,nblk+3,Int4);
		for(b=1; b <= nblk; b++) len[b]=8;
#if 0
		ss_type	data=TrueDataCMSA(cma);
#if 0
		rcma=EmptyCMSA(nblk,len,data,gss->GapOpen(),gss->GapExtend(),PerNatsCMSA(cma),0,0);
		Int4 t,ln,sq,n = MaxSeqSeqSet(data);
		dh_type dH=dheap(n+2,3);
		for(sq=1; sq <= NSeqsSeqSet(data); sq++) {
		    ln=SqLenSeqSet(sq,data);
	            for(b=1; b <= nblk; b++){
	                ln -= len[b]; insrtHeap(b,(keytyp)Random(),dH);
	            }
            	    for(s=b; ln > 0 ; s++,ln--) insrtHeap(s,(keytyp)Random(),dH);
	            for(b=s=1; (i=delminHeap(dH)) != NULL; ){
	                // WARNING: need to generalize beyond one colinear!
	                if(i<=nblk){
				AddSiteCMSA(b,sq,s,rcma); s+=len[b]; b++; 
			} else s++;
	            }
	        } Nildheap(dH);
#else
		xcma=CreateRandomCMA(nblk,len,data,PerNatsCMSA(cma));
#endif

#else
		rcma=RandomCMSA(nblk,len,*gss); InitCMSA(rcma);
		xcma=OneBlockCMSA(rcma); ExtendFakeToRealCMSA(xcma); 
#endif
	{
   		sprintf(str,"%s_r",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,xcma); fclose(wfp);
	}

		char	StMd=' ';
	      double	bild_cut=-3.2,bild_add=-3.0,afrq=0.30,rfrq=0.40;  
	 {
	      prior_wt=0.90;
              gpr_typ gpr("test",1,1);
              gpr.SetParameters(aa_per_io,aa_per_do,exp_ie,exp_de,prior_wt,dms_mode);
              gpr.SetAddRmCols(bild_cut,bild_add,Temp,afrq,rfrq); gpr.SetSticky(MinSticky,StMd);
                Int4 nAddRms=gpr.DoAddRmColumns(stdout,xcma,1);        // need to first add columns (for 1 cycle).
                if(nAddRms==0){
                   gpr.DoStickySample(stdout,xcma);
                   nAddRms=gpr.DoAddRmColumns(stdout,xcma,1);
                   // if(nAddRms > 0) nAddRmMv += nAddRms;
                } // else nAddRmMv += nAddRms;
                gpr.DoPurgeSampling(stdout,xcma);
	 }
		


                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
		if(prior_wt > 0) gmb->SetPriorWt(prior_wt); 
#if 1
	   for(i=1; isalpha(strategy[i]); i++){
		gmb->SampleStickyTogether(stdout, MinSticky, MaxFrctn,str,Temp,strategy[i]);
		cma=gmb->RtnCMSA();
	   } 
#else
	   double map;
	   Int4 sq=91,N=NumSeqsCMSA(xcma);
	   for(i=1; i <= 10; i++){
	   for(sq=1; sq <= N; sq++){
#if 0
	        if(i==1){ 
		  aa_per_io=1,aa_per_do=120,exp_ie=1,exp_de=1; 
		  prior_wt=2.00; gmb->SetPriorWt(prior_wt); 
		  gmb->ChangePriors(aa_per_io,aa_per_do,exp_ie,exp_de);
	        } else {
		  aa_per_io=20,aa_per_do=200,exp_ie=1,exp_de=1; 
		  prior_wt=0.30; gmb->SetPriorWt(prior_wt); 
		  gmb->ChangePriors(aa_per_io,aa_per_do,exp_ie,exp_de);
	        } gmb->SampleSingle(sq,Temp,map);
#else
		  aa_per_io=20,aa_per_do=200,exp_ie=4,exp_de=1; 
		  prior_wt=0.10; gmb->SetPriorWt(prior_wt); 
		  gmb->ChangePriors(aa_per_io,aa_per_do,exp_ie,exp_de);
		gmb->SampleSingle(sq,Temp,map);
		// break;
#endif
	      }
	   }
#endif
   		// sprintf(str,"%s_%c",argv[1],strategy[i]); 
   		sprintf(str,"%s_s",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
		delete gmb;
		// NilCMSA(rcma);
	} break;
    case 'A':	// test out subalignment sampling.
	{
		dms_mode='F';
		ssx_typ *issx;
		cma_typ	rcma=0,xcma=cma;
		Int4	i=0,N=NumSeqsCMSA(cma),n,sq;
		double	d,dd,Temp=100;
		h_type	HG=Histogram("deletions per sequences",0,LengthCMSA(1,cma),5.0);;
#if 1
		// 1. find column residue freqs.
		// double d,**frq = ColResFreqsCMSA(see_blk,&obs, cma);

		aa_per_io=75,aa_per_do=200,exp_ie=1,exp_de=1;
		aa_per_io=200,aa_per_do=20,exp_ie=1,exp_de=1;
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
#if 1
	{
double temp=0.0;
Int4	similarity=0; // not used, but if < 0 signals to sample in clusters...
gmb->Sample(0,'S',similarity,temp,30,30,stderr);
do {
		double bildcut=0.0;
		// rcma=gmb->AddColumns(bildcut,TRUE,0,0.50);
		rcma=gmb->AddColumns(bildcut,FALSE,0,0.75);
   		sprintf(str,"%s_ac",argv[1]); 
		if(rcma==0) break;
		else {
			delete gmb;
			NilCMSA(xcma); xcma=rcma;
                	gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
		}
#if 1
similarity=-2; // Sample in clusters...
gmb->Sample(0,'S',similarity,temp,30,30,stderr);
#else
	char    Strategy[20]=" IVXFD    ";  // 'D' ruins the alignment?
	double  sfrq[20]={ 0.0,0.25,0.25,0.25,0.25,0.25,0.0,0.0,0.0,0.0,0.0,0,0,0.0};
	Int4	MinSticky=5;
	gmb->SetMaxIter(1);
	rcma=gmb->SampleStickyTogether(stderr,MinSticky,Strategy,sfrq,temp);
#endif
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,rcma); fclose(wfp);
} while(rcma);
		exit(1);
	}
#endif

                // gmb_typ *gmb=new gmb_typ(75,50,1,5,xcma,dms_mode);
	{
		Int4 MinSticky=2;
	for(i=1; i <= 3; i++){
		double d,dd;
		time1=time(NULL); 
		// gmb->ConservedTogether(0.20,300);
		// gmb->SamplePurged(50,0.20,300);
		gmb->SamplePurged(50,0.20,0.0,d,dd);
		fprintf(stderr,"time: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	}
#if 0
char   strategy[]=" RRRR ";
double frq[]={0.0,0.15,0.15,0.15,0.15,0.0 };
		for(Int4 z=1; z <=2; z++){
                gmb->SampleStickyTogether(stderr,MinSticky,strategy,frq,150,'O');
                // if(rcma){ FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,rcma); fclose(wfp); }
                }
		gmb->ConservedTogether(0.20,100);
#endif
#if 0
		delete gmb;
                gmb=new gmb_typ(75,100,1,1,xcma,dms_mode);
		gmb->ConservedTogether(0.20,150);
#endif
   		sprintf(str,"%s_corr",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
#if 0
	     	char   strategy[]=" C  ";
	     	double frq[]={0.0,0.15,0.15,0.15,0.15,0.0 };
		for(Int4 z=1; z <=2; z++){
		   for(i=1; i <= 3; i++){
		     rcma=gmb->SampleStickyTogether(stderr,MinSticky,frq[i],str,Temp,strategy[i]);
		   }
		   if(rcma){
   			  FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,rcma); fclose(wfp);
		   }
		}
#endif
	}
		delete gmb;
		exit(1);
	
		// 2. find column bild scores...
		// 3. find 
		// Look at -r twkcma option..
#elif 0
		class foo_typ {
		   public:
			foo_typ(){ }
			double prob(Int4 strt,Int4 end,Int4 **FG,Int4 **BG,a_type AB)
			{
			  double d,p,q,P=0,Q=0,sum=0,sumFG=0,sumBG=0;
			  double tiny=0.01;
			  Int4	i,r;
			  for(P=0,i=strt; i <= end; i++){
			     sumFG=sumBG=(double)nAlpha(AB)*tiny;
		   	     for(r=1; r <= nAlpha(AB); r++){
				sumFG += (double) FG[r][i];
				sumBG += (double) BG[r][i];
			     }
	// fprintf(stderr,"%d: SumFG=%g; SumBG=%g\n",i,sumFG,sumFG);
		   	     for(d=0,r=1; r <= nAlpha(AB); r++){
				p = ((double)FG[r][i] + tiny)/sumFG;
				q = ((double)BG[r][i] + tiny)/sumBG;
				d += p*log(p/q);
	// fprintf(stderr,"  %c: FG=%d; BG=%d;",AlphaChar(r,AB),FG[r][i],BG[r][i]);
	// fprintf(stderr," p=%g; q=%g\n",p,q);
			     } P+=d;
	// fprintf(stderr," P=%g; d=%g\n",P,d);
			  } return P;
			}
		};
		{
		   Int4	  st,pos[5],r,*FG[30],*BG[30];	// number of each residue.
		   Int4	strt=15,end=36;
		   // strt=50,end=70;
		   // strt=70,end=90;
		   // strt=1,end=15;
		   strt=23,end=27;
		   char   *Label; NEW(Label,N+5,char);
		   Int4	nFG=0,nBG=0;
		   for(r=0; r <= nAlpha(AB); r++){
		      NEW(FG[r],105,Int4); NEW(BG[r],105,Int4);
		   }
		   double d,dd;
		   unsigned char   *isq=0;
		   foo_typ *foo = new foo_typ();
		   for(sq=1; sq <= N; sq++){
		      isq=SeqPtrCMSA(sq,cma); assert(PosSiteCMSA(1,sq,pos,cma) > 0); st=pos[1];
		      if(sq%3==0){ Label[sq]='F'; nFG++; } else { Label[sq]='B'; nBG++; }
		      for(i=strt; i <=end; i++){
			r=isq[st+i-1];
			if(Label[sq]=='F'){ FG[r][i]+=1; } else { BG[r][i]+=1; }
		      }
		   } d=foo->prob(strt,end,FG,BG,AB); fprintf(stderr,"prob=%g\n",d);
		   BooLean  improved=FALSE;
		   do { 
		      improved=FALSE;
		      for(sq=1; sq <= N; sq++){
		        d=foo->prob(strt,end,FG,BG,AB); // fprintf(stderr,"prob=%g\n",d);
		        isq=SeqPtrCMSA(sq,cma); assert(PosSiteCMSA(1,sq,pos,cma) > 0); st=pos[1];
			if(Label[sq]=='F'){ Label[sq]='B'; nBG++; nFG--; } else { Label[sq]='F'; nFG++; nBG--; }
			for(i=strt; i <=end; i++){
			    r=isq[st+i-1];
			    if(Label[sq]=='B'){ FG[r][i]-=1; BG[r][i]+=1; }
			    else { BG[r][i]-=1; FG[r][i]+=1; }
			} 
		        dd=foo->prob(strt,end,FG,BG,AB); // fprintf(stderr,"prob=%g\n",d);
			if(dd < d){	// then  revert back;
			  if(Label[sq]=='F'){ Label[sq]='B'; nBG++; nFG--; }
			  else { Label[sq]='F'; nFG++; nBG--; }
			  for(i=strt; i <=end; i++){
			    r=isq[st+i-1];
			    if(Label[sq]=='B'){ FG[r][i]-=1; BG[r][i]+=1; }
			    else { BG[r][i]-=1; FG[r][i]+=1; }
			  } 
			} else {
		          improved=TRUE;
			  fprintf(stderr,"%d: prob=%g --> %g (fg=%d,bg=%d)\n",sq,d,dd,nFG,nBG);
			}
		      }
		   } while(improved);
		delete foo;
		} exit(1);
#elif 1
		{ 
		   gmb_typ *gmb=0;
		   Int4	MinSticky=2;
		   char       Strategy[20]=" SRXVWTS ";
     		   double     sfrq[20]={ 0.0,0.15,0.15,0.15,0.15,0.05,0.10,0.10,0.0,0.0};
                                          //   S    R    X    V    W    T    S

		   for(i=1; i <= 7; i++){
			xcma=CopyCMSA(cma);
			switch(i){
#if 1
			  case 1: aa_per_io=1000; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 2: aa_per_io=500; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 3: aa_per_io=250; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 4: aa_per_io=125; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 5: aa_per_io=80; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 6: aa_per_io=40; aa_per_do=100; exp_ie=1; exp_de=1; break;
			  case 7: aa_per_io=20; aa_per_do=100; exp_ie=1; exp_de=1; break;
#else
			  case 1: aa_per_io=100; aa_per_do=1000; exp_ie=1; exp_de=1; break;
			  case 2: aa_per_io=100; aa_per_do=500; exp_ie=1; exp_de=1; break;
			  case 3: aa_per_io=100; aa_per_do=250; exp_ie=1; exp_de=1; break;
			  case 4: aa_per_io=100; aa_per_do=125; exp_ie=1; exp_de=1; break;
			  case 5: aa_per_io=100; aa_per_do=80; exp_ie=1; exp_de=1; break;
			  case 6: aa_per_io=100; aa_per_do=40; exp_ie=1; exp_de=1; break;
			  case 7: aa_per_io=100; aa_per_do=20; exp_ie=1; exp_de=1; break;
#endif
			  default: print_error("input error"); break;
			}
                	gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,xcma,dms_mode);
			// rcma=gmb->AddColumns(-4.0,TRUE,0,0.50);
			rcma=gmb->SampleStickyTogether(stdout,MinSticky,Strategy,sfrq,100,'O');
   			sprintf(str,"%s_tst%d",argv[1],i); 
			if(rcma){
   			  FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,rcma); fclose(wfp);
			}
			NilCMSA(xcma);
		   } exit(1);
			
		}
#elif 1
		Int4	x,numSets,percent_id=40;
		double	max_fract=0.15;
		set_typ *SetSq=RtnTargetSizeSetsCMSA(numSets,percent_id,cma,max_fract);
		dd=max_fract;
                gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
		for(i=1; i <= numSets; i++){
fprintf(stderr,"Set %d: %d\n",i,(x=CardSet(SetSq[i]))); 
	if(x < 4) continue;
			dh_type dH=dheap(NumSeqsCMSA(cma)+5,4);
			for(Int4 sq=1; sq <= NumSeqsCMSA(cma); sq++){
			   if(MemberSet(sq,SetSq[i]))
			     { gsq_typ *gsq=gsqCMSA(sq,cma); insrtHeap(sq,(keytyp)gsq->NumDel(),dH); }
			} gmb->SampleTogether(SetSq[i],Temp,dd,dH); Nildheap(dH); NilSet(SetSq[i]);
		} free(SetSq);
		delete gmb;
   		sprintf(str,"%s_sim",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
#elif 1
	        Int4 nMvCycles=0,MinSticky=2;
	        do {
                  ssx_typ *issx = new ssx_typ(aa_per_io,aa_per_do,exp_ie,exp_de,1000,cma,dms_mode);
		  set_typ SetSq;
                  rcma=MvColumnsCMSA(issx,SetSq); delete issx;
		  gmb_typ *gmb=0;
                  if(rcma) {
		        if(gmb) delete gmb; gmb=0; nMvCycles++; 
			if(rcma != cma){ NilCMSA(cma); cma=rcma; } rcma=0; 
                        gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
			// gmb->SetPriorWt(prior_wt); 
			gmb->SetMaxIter(3);
                        // sprintf(str,"%s_gmb%d",argv[1],i);  RenameCMSA(str,cma); 
#if 1
			dd=(double)CardSet(SetSq)/(double)NumSeqsCMSA(cma);
		 	if(dd < 0.15){
			  dh_type dH=dheap(NumSeqsCMSA(cma)+5,4);
			  for(Int4 sq=1; sq <= NumSeqsCMSA(cma); sq++){
				if(MemberSet(sq,SetSq)) insrtHeap(sq,(keytyp)Random(),dH);
			  } gmb->SampleTogether(SetSq,Temp,dd,dH); Nildheap(dH); NilSet(SetSq);
			} else {
	     		  char   strategy[]=" TRF  ";
	     		  double frq[]={0.0,0.15,0.15,0.15,0.15,0.0 };
			  for(Int4 z=1; z <=2; z++){
			    for(i=1; i <= 3; i++){
			      rcma=gmb->SampleStickyTogether(stderr,MinSticky,frq[i],str,Temp,strategy[i]);
			      assert(cma==rcma); // fflush(rfp); 
			    }
			  }
			}
#else
                        rcma=gmb->Sample(str,'S',similarity,Temp,Temp,50,stderr); 
			assert(cma==rcma);
#endif
			delete gmb; gmb=0;
                  }
	       	} while(rcma); 
   		sprintf(str,"%s_del",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
#elif 1
		set_typ	SetSq=MakeSet(N+5); ClearSet(SetSq);
		dh_type dH=dheap(N+3,4);
		for(sq=1; sq <= N; sq++){
			gsq_typ *gsq=gsqCMSA(sq,cma);
			if((n=gsq->NumDel()) > 0){
				insrtHeap(sq,(keytyp)-n,dH);	// minkey = most deletions.
				IncdHist(n,HG);
			}
		} PutHist(stderr,60,HG); NilHist(HG);
		dh_type dH2=dheap(N+3,4);
		for(n=0; !emptyHeap(dH); ){
			n++; d=(double)n/(double)N;
			if(d > 0.15) break;
			sq=delminHeap(dH); assert(sq); d=minkeyHeap(dH);
			AddSet(sq,SetSq); insrtHeap(sq,(keytyp)-d,dH2); // minkey=least deletions.
		} Nildheap(dH);
   		gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode); 
   		sprintf(str,"%s_gmb%d",argv[1],i);  gmb->SetMaxIter(3);
		gmb->SampleTogether(SetSq,150,d,dH2); delete gmb; Nildheap(dH2);
   		sprintf(str,"%s_del",argv[1]); 
   		FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
#elif 1
		do {
		  issx = new ssx_typ(aa_per_io,aa_per_do,exp_ie,exp_de,1000,xcma,dms_mode);
		  set_typ SetSq;
		  rcma=MvColumnsCMSA(issx,SetSq); delete issx; i++;
		  NilSet(SetSq);
		  if(rcma) {
   			gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,rcma,dms_mode); 
   		  	sprintf(str,"%s_gmb%d",argv[1],i);  gmb->SetMaxIter(3);
			gmb->Sample(str,'S',similarity,100,0,50,stderr); delete gmb;
			if(xcma != cma){ NilCMSA(xcma);} xcma=rcma; 
		  }
		} while(rcma);
		if(xcma != cma){
   		  sprintf(str,"%s_mv",argv[1]); 
   		  FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,xcma); fclose(wfp);
		  NilCMSA(xcma); 
		}
	// exit(1);
#else
		cma_typ rcma=run_gambit(stdout,argv[1],1,issx,0,4,3);
		delete issx;
#endif
	} break;
    case 1:	// test out subalignment sampling.
	{
   Int4	MinSticky=2;
   double	MaxFrctn=0.0; // not used
#if 0
   gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,' ');  // Altschul dm-20.
#elif 0
   gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,'O'); 
#elif 1
   // gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,'T'); 
   gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,'O'); 

#if 0
   sprintf(str,"%s_gmb",argv[1]); 
   do { 
	ExtendFakeToRealCMSA(cma);
   	gmb->SampleStickyTogether(stderr,MinSticky,MaxFrctn,str);
   } while(TRUE);
delete gmb; break;
#endif

#else
   gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,'f'); 
#endif
   gmb->SetMaxIter(2);
   sprintf(str,"%s_gmb",argv[1]); gmb->Sample(str,'S',similarity);
   FILE *wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
delete gmb; break;
   // gmb->SampleStickyTogether(stderr,MinSticky,MaxFrctn,str);
   wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
   gmb->Sample(str,'S',similarity);
   wfp=open_file(str,".cma","w"); PutCMSA(wfp,cma); fclose(wfp);
   delete gmb;
   // WriteCMSA("junkF_debug.cma",cma); 
    } break;
//====================================================================
    case 2:	// test sub-realignment routine.
/******************************************************************
Note: need to create a blank gsq_typ for this sequence (doesn't exist for new cma 
because it was omitted when created.  This is influencing ndl_typ::AddIndels() 
routine...take a look.
Also look at AddSite(): somehow adding at position zero when no site is present...!
look at create_gsq_cmsa() in cmsa_io.cc; this needs to add a gsq when no sites are present.
Create void    gsq_typ::init(e_type E) with realE == fakeE?  test out...
 ******************************************************************/
    {
	// cma=gmb->Sample(argv[1],'S',similarity,temp); break;
	// sprintf(str,"%s_gmb.cma",argv[1]); WriteCMSA(str,cma);
	// cma=SampleSubAlignsCMSA(argv[1],cma,dms_mode);
    } break; 
//====================================================================
    case 3:	// test extend diagonals.
	  {
#if 0
	   Int4 score,offset,N=NumSeqsCMSA(cma);
	   for(i=1; i < N; i++){
	     e_type E1=TrueSeqCMSA(i,cma);
	     Int4 s1=TruePosCMSA(i,1,cma);
	     for(j=i+1; j <= N; j++){
	  	e_type E2=TrueSeqCMSA(j,cma);
	  	Int4 s2=TruePosCMSA(j,1,cma);
		offset=s1-s2;
		// FastExtendGBlastStr(E1, s1,LenSeq(E2),SeqPtr(E2),s2,AlphaR(AB),score);
		PutDiagonalSeq(stderr,offset,E1,E2,AB);
	     }
  	  }
#elif 0
   	   gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	   for(i=1; i < 20; i++){
		gmb->TestAddRemove(); 
	   } delete gmb;
#else
	   // Test adding and removing sequences...
	   for(i=1; i < 20; i++){
   	        gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
		gmb->TestAddRemove(); delete gmb;
	   }
#endif
    } break; 
//====================================================================
    case 4:	// test clustering of sequences into sets for sampling.
	  {
	time1=time(NULL); 
	set_typ OutSet,InSet=MakeSet(NumSeqsCMSA(cma)+4); FillSet(InSet);
	for(i=30; i <= 90; i+=5){
		fprintf(stderr,"*************** %d%c identity *******************\n",i,'%');
		OutSet=RtnFastRepSetCMSA(stderr,i,InSet,cma);
		NilSet(OutSet);
	}
	fprintf(stderr,"\tssx time: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	exit(1);
	  } break; 
//====================================================================
	 case 5:	// test clustering of sequences into sets for sampling.
	  {
	time1=time(NULL); 
	set_typ *OutSet,InSet=MakeSet(NumSeqsCMSA(cma)+4); FillSet(InSet);
	Int4	numSets=0,percent_ident=60;
	OutSet=RtnFastClustersCMSA(numSets,percent_ident,InSet,cma);
	for(i=1; i <= numSets; i++){
		PutSet(stderr,OutSet[i]);
		NilSet(OutSet[i]);
	} free(OutSet);
	fprintf(stderr,"\tssx time: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	exit(1);
	  } break; 
//====================================================================
	 case 6:	// test sampling....
     {
	FILE *rfp=open_file(argv[1],".log","w");
	gmb_typ	*gmb;
	// gmb_typ	*gmb = new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
	sprintf(str,"%s_gmb",argv[1]);
	cma_typ	rcma=0;
        double bild_cut=0.0,BildCut=0.0;
        // for(bild_cut=3.0, BildCut=-6.0, J=1; J <= 4; J++,bild_cut -= 1.0,BildCut += 2.0)
	BooLean improved;
	Int4 J=0;

	do {
	   J++; improved=FALSE;
#if 0
	if(J > 1){
           fprintf(rfp,"%d.rm %d columns\n",J,LengthCMSA(1,cma)); 
           do {
               RenameCMSA(str,cma);
               gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	       if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
               cma=gmb->Sample(str,'S',similarity);
               rcma=gmb->RmColumns(bild_cut);
               if(rcma){ delete gmb; NilCMSA(cma); cma=rcma; improved=TRUE; }
           } while(rcma);
	   // add columns...

           delete gmb;
 	}
#endif
           fprintf(rfp,"%d.add %d columns\n",J,LengthCMSA(1,cma)); 
           RenameCMSA(str,cma);
	   Int4 MinSticky=2;
	   double MaxFrctn=0.49;
           gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	   if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
	   gmb->SetMaxIter(3);
	   // gmb->SampleStickyTogether(rfp,MinSticky,MaxFrctn,str);
           gmb->Sample(str,'S',similarity);
           do {
               // rcma=gmb->AddColumns(BildCut);
	       BildCut=-1.0;
               rcma=gmb->AddColumns(BildCut,FALSE);
               if(rcma){
                        delete gmb; NilCMSA(cma); cma=rcma; improved=TRUE;
                        gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
			if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
	   		gmb->SetMaxIter(3);
                        rcma=gmb->Sample(str,'S',similarity);
// break;
               }
           } while(rcma);
	   delete gmb;

          gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	  // rcma=gmb->SampleStickyTogether(rfp,MinSticky,MaxFrctn,str);
          fprintf(rfp,"%d.done %d columns; LLR = %.2f\n",J,LengthCMSA(1,cma),gmb->RtnMap());
          delete gmb; 
	   fflush(rfp);
// improved=FALSE;
        } while(improved || J < 2);
	fclose(rfp);
	// exit(1);
	  } break; 
//====================================================================
	 case 'L':	//============= check LLR.... ==================
	  {
	gmb_typ *gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
        fprintf(stderr," %d columns; LLR = %.2f\n",LengthCMSA(1,cma),gmb->RtnMap());
	delete gmb;
	exit(1);
	  } break; 
//====================================================================
	 case 8:	//============= check sticky.... ==================
	{
#if 0
	Int4	mhpsz=1;
	char    **Argv; NEWP(Argv,argc+20,char);
	int     Argc=argc; Argv[1]=0; Argv[1]=argv[1];
	i=Argc; Argv[i]=AllocString("-DoNotSeed"); Argc++;
        sprintf(str,"-h%d",mhpsz); i++; Argv[i]=AllocString(str); Argc++;
	gsm_typ *gsm = new gsm_typ("gismo usage",Argc,Argv,AB,TrueDataCMSA(cma));
	Int4    ave_blk=10,avecol=7,minblk=3,maxblk=40;
	Int4 M=ModeLengthSeqSet(TrueDataCMSA(cma)); maxblk=MAX_NO_TYPESITES;
        cma_typ *rtn_cma=gsm->Align(ave_blk,avecol,minblk,maxblk,M);
	cma=rtn_cma[1];
#elif 0
	Int4	mhpsz=1;
	char    **Argv; NEWP(Argv,argc+20,char);
	int     Argc=argc; Argv[1]=0; Argv[1]=argv[1];
	i=Argc; Argv[i]=AllocString("-DoNotSeed"); Argc++;
        // sprintf(str,"-h%d",mhpsz); i++; Argv[i]=AllocString(str); Argc++;
	gsm_typ *gsm = new gsm_typ("gismo usage",Argc,Argv,AB,TrueDataCMSA(cma));
        double map=gsm->RunPublicGibbs(&cma);
#else
	char *options=AllocString("-t1 -g -l1 ");
	double map=SimAnnealGibbs(options,&cma,'S',100);
	free(options);
#endif
	cma_typ xcma=cma; cma=OneBlockCMSA(xcma); ExtendFakeToRealCMSA(cma);
	gmb_typ	*gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
        fprintf(stderr,"Start %d columns; LLR = %.2f\n",LengthCMSA(1,cma),gmb->RtnMap());
	sprintf(str,"%s_sticky",argv[1]);
	gmb->SetMaxIter(2);
	do { 
	   // gmb->SampleStickyTogether(stderr,2,0.49,str);
           gmb->Sample(str,'S',similarity);
	} while(TRUE);
	// ssx_typ *ssx=gmb->RtnSSX();
	gmb->RtnMap();
        fprintf(stderr,"Done %d columns; LLR = %.2f\n",LengthCMSA(1,cma),gmb->RtnMap());
        delete gmb; 
	  } break; 
//====================================================================
	 case 9:	//============= check sticky.... ==================
    {
	Int4 MinSticky=2;
	double MaxFrctn=0.49;
	similarity=0;
	sprintf(str,"%s_sticky",argv[1]);
	gmb_typ	*gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	double	f[3][6]={ {0}, {0.0,0.20,0.33,0.33,0.33,0.0},
			 {0.0,0.20,0.33,0.33,0.33,0.0}
		       };
gmb->foo(f[0]); exit(1);
	if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
        fprintf(stderr,"Start %d columns; LLR = %.2f\n",LengthCMSA(1,cma),gmb->RtnMap());
#if 0
	// gmb->SampleStickyTogether(stderr,MinSticky,0.25,str,150,'D');
	// gmb->SampleStickyTogether(stderr,MinSticky,0.25,str,'S');
	gmb->SampleStickyTogether(0,MinSticky,MaxFrctn,str,0.0);
        // cma_typ rcma=gmb->Sample(str,'S',similarity,150);
        cma_typ rcma=gmb->Sample(str,'S',similarity,0,0,30);
#else
	cma_typ rcma=0;
        double Temp,delta=0.90,lpr1,lpr0;
#if 0
	for(Temp=150; Temp > 0.0; Temp -= 25) {
             delete gmb;
             gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
             // gmb->InitHMM(); lpr0=gmb->RtnMap();
             char   strategy[]=" DTRF  ";
             for(i=1; i <= 4; i++){
                rcma=gmb->SampleStickyTogether(stderr,MinSticky,0.33,str,Temp,strategy[i]);
                assert(cma==rcma); 
             }
        }
#elif 1
        for(Temp=150; Temp > 0.0; Temp -= 25)
        // for(Temp=150,j=1; j <= 10; j++)
	{
           if(Temp < 40) Temp=0.0;
	   
	   lpr0=gmb->RtnMap(); // assert(cma==rcma);
	   gmb->InitHMM(); lpr0=gmb->RtnMap(); 
           char   action[]=" DTRF  ";
	   Int4   strt=1;
           if(Temp==0) strt=2;
           for(i=strt; i <= 4; i++){
	        cma_typ xcma = CopyCMSA(cma);
                rcma=gmb->SampleStickyTogether(stderr,MinSticky,0.33,str,Temp,action[i]);
                assert(cma==rcma); lpr1=gmb->RtnMap();
                fprintf(stderr,"LLR: %.2f --> %.2f (%2f)(%.1f K)\n",lpr0,lpr1,lpr1-lpr0,Temp);
		double dd=lpr1 - lpr0;
                if(i==1 && dd < -40){
                    delete gmb; NilCMSA(cma); cma=xcma; rcma=cma;
                    gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
                    gmb->InitHMM(); lpr0=gmb->RtnMap();
                } else { NilCMSA(xcma); lpr0=lpr1; }
           }
        }
#else
#if 0
           rcma=gmb->SampleStickyTogether(stderr,MinSticky,MaxFrctn,str,Temp); 
#else
	   gmb->SampleStickyTogether(stderr,MinSticky,0.33,str,Temp,'T');
           gmb->SampleStickyTogether(stderr,MinSticky,0.33,str,Temp,'R');
           rcma=gmb->SampleStickyTogether(stderr,MinSticky,0.33,str,Temp,'F');
#endif
	   assert(cma==rcma); 
           rcma=gmb->Sample(str,'S',similarity,Temp,Temp,30);
	   assert(cma==rcma); 
           lpr1=gmb->RtnMap(); // assert(cma==rcma);
           fprintf(stderr,"LLR: %.2f --> %.2f (%2f)(%.1f K)\n",lpr0,lpr1,lpr1-lpr0,Temp);
           delete gmb; 
	   gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
        }
#endif
#endif
        fprintf(stderr,"Done %d columns; LLR = %.2f\n",LengthCMSA(1,cma),gmb->RtnMap());
	fp = open_file(argv[1],"_gmb.cma","w"); PutCMSA(fp,rcma);  fclose(fp);
        delete gmb; 
    } break; 
//====================================================================
	 case 10:	//============= check SMatrix.... ==================
    {
	gmb_typ	*gmb=new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma); 
	ssx_typ *ssx=gmb->RtnSSX();
	ssx->InitNDL(0.0);
fprintf(stderr,"VrtlMap = %.2f; DirichletMap=%.2f\n",
		ssx->VirtualMap(),ssx->DirichletLLR()); exit(1);
		// ssx->GapMap(),ssx->DirichletLLR()); exit(1);

	smx_typ *smx=ssx->RtnSMX(); 
	PutSMatrix(stderr, smx[1]); 
	for(i=1; i <=10; i++){
		smx=ssx->SampleSMX(300); 
		PutSMatrix(stderr, smx[1]); NilSMatrix(smx[1]); free(smx);
	}
    } break; 
//====================================================================
	default:	// Run gambit as is.	
     {
	// std::cerr << "DEBUG 0" << std::endl;
	gmb_typ	*gmb = new gmb_typ(aa_per_io,aa_per_do,exp_ie,exp_de,cma,dms_mode);
	if(prior_wt > 0) gmb->SetPriorWt(prior_wt);
	if(AdjSqWt > 0.0) gmb->SetSqWtAdjust(AdjSqWt);
	// std::cerr << "DEBUG X" << std::endl;

	if(aln){
	   if(temp < 0.0) temp=0.0;
	   ss_type data=SeqSet(aln,AB);
	   // if(!input_priors) gmb->ChangePriors(200,200,2,1);
	   cma_typ rcma=gmb->CreateAlign(data);
	   fp = open_file(argv[1],"_aln.cma","w"); PutCMSA(fp,rcma);  fclose(fp);
	   TotalNilCMSA(rcma);	// deletes data as well.
	   free(aln); aln=0;
	} else if(dbs){
	   if(temp < 0.0) temp=0.0;
	   ss_type data=SeqSet(dbs,AB);
	   // if(!input_priors) gmb->ChangePriors(200,200,2,1);
	   for(Int4 sq=1; sq <= NSeqsSeqSet(data); sq++){
		e_type E=SeqSetE(sq,data);
   		// for(i=1; i <=3; i++) gmb->AlignSeq(stderr,method,E); 
		// gmb->AlignSeq(stderr,method,E,temp); 
		gmb->SampleAlignSeq(stderr,method,E,temp); 
	   } NilSeqSet(data); free(dbs); dbs=0;
	} else {
	 if(temp < 0.0) temp=300.0;
	 switch (mode){
	  case 'S': cma=gmb->Sample(argv[1],'S',similarity,temp); break;
#if 0	// These don't seem to work...
	  case 'P': cma=gmb->ProgressiveSampling(argv[1],15,70,1); break;
	  case 'B': 
		gmb->ChangePriors(5,5,2,1);
	  	cma=gmb->ProgressiveSampling(argv[1],20,70,1); 
		gmb->ChangePriors(200,200,2,1);
		cma=gmb->Sampling(argv[1],'S',0); // 0 --> don't sample in clusters.
		break;
#endif
	  case 'T': break; // test...
	  default: print_error("mode input error");
	 }
	 sprintf(str,"%s_gmb.cma",argv[1]); WriteCMSA(str,cma);
	} delete gmb;
	} break;
      } // end of switch(Options)...
//====================================================================
	TotalNilCMSA(cma); NilAlpha(AB);
	fprintf(stderr,"\ttime: %d seconds (%0.2f minutes)\n",
                        time(NULL)-time1,(float)(time(NULL)-time1)/60.0);
	return 0;
}



