/******************************************************************************************
    Copyright (C) 1997-2014 Andrew F. Neuwald, Cold Spring Harbor Laboratory
    and the University of Maryland School of Medicine.

    Permission is hereby granted, free of charge, to any person obtaining a copy of 
    this software and associated documentation files (the "Software"), to deal in the 
    Software without restriction, including without limitation the rights to use, copy, 
    modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
    and to permit persons to whom the Software is furnished to do so, subject to the 
    following conditions:

    The above copyright notice and this permission notice shall be included in all 
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
    INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
    PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE 
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
    OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
    OTHER DEALINGS IN THE SOFTWARE.

    For further information contact:
         Andrew F. Neuwald
         Institute for Genome Sciences and
         Department of Biochemistry & Molecular Biology
         University of Maryland School of Medicine
         801 West Baltimore St.
         BioPark II, Room 617
         Baltimore, MD 21201
         Tel: 410-706-6724; Fax: 410-706-1482; E-mail: aneuwald@som.umaryland.edu
 ******************************************************************************************/

#include "math.h"
#include "stdinc.h"
#include "afnio.h"
#include "dheap.h"
#include "histogram.h"

class wcx_typ {
public:
		wcx_typ(char *p,char *m,float mxD,double sc,Int4 el,double biu,Int4 du,Int4 x,Int4 dl){
			pdb=AllocString(p); mthd=AllocString(m);
			maxD=mxD;L=el; D=du; d=dl; X=x; 
			score=sc; nrm_scr=100*sc/(double)L; biu_scr=biu;
		}
		void	PutExcel(FILE *fp,float start, float end) {
		  if(maxD==start) fprintf(fp,"%s\t%s\t%.1lf\t%lf\t%lf\t%d\t%d\t%d\t%d\t%d\t%lf\t0\t",
				pdb,mthd,maxD,score,biu_scr,L,D,X,d,D-d,log(10)*nrm_scr);
		  else if(maxD==end) fprintf(fp,"%.1lf\t%lf\t%lf\t%d\t%d\t%d\t%d\t%d\t%lf\n",
				maxD,score,biu_scr,L,D,X,d,D-d,log(10)*nrm_scr);
		  else fprintf(fp,"%.1lf\t%lf\t%lf\t%d\t%d\t%d\t%d\t%d\t%lf\t0\t",
				maxD,score,biu_scr,L,D,X,d,D-d,log(10)*nrm_scr);
		}
		void	PutAll(FILE *fp) {
		  fprintf(fp,"%s\t%s\t%lf\t%lf\t%d\t%lf\t%d\t%d\t%d\n",pdb,mthd,maxD,score,L,biu_scr,D,X,d);
		}
		BooLean	IsDCA(char *m){ if(strcmp(m,mthd)==0) return TRUE; else return FALSE; }
		void	PutKey(FILE *fp) { fprintf(fp,"%s\t%lf\n",pdb,score); }
		double  BIUScore(){ return biu_scr; }
		char	*RtnPDB(){ return pdb; }
		char	*RtnMthd(){ return mthd; }
		// double	RtnScore(){ return score; }
		double	RtnScore(){ return nrm_scr; }	// use normalized score...
		double	RtnBIU(){ return biu_scr; }
private:
		char	*pdb,*mthd;
		float	maxD;
		double	score,biu_scr,nrm_scr;
		Int4	L,D,X,d;
};

double	RunWilcoxon(FILE *fp,Int4 N, char **Names, double *classA, double *classB,double &pval)
{
	Int4	i,j,k,R,x,Pos,Neg;
	double	diff[100],Sum,V,D,Z,T; 
	dh_type dH=dheap(N+5,4);
	for(i=1; i <= N; i++){
		diff[i]=classA[i]-classB[i]; 
		insrtHeap(i,(keytyp)fabs(diff[i]),dH);
	}
	for(Pos=Neg=0,Sum=0.0,R=1; !emptyHeap(dH); R++){
		assert((i=delminHeap(dH)) != 0); 
		if(diff[i] > 0){ Pos+=R; Sum += (double) R; }
		else if(diff[i] < 0){ Neg+=R; Sum -= (double) R; }
		else { R--; N--; }
		if(fp) fprintf(stderr,"%2d.%s\t%lf\t%lf\t%lf\t%.0lf\n",
			R,Names[i],classA[i],classB[i],diff[i],Sum);
	} D=(double) N; 
	V = (double)D*(D+1)*(2*D + 1)/6.0; Z = Sum/sqrt(V);
	pval=erfc(fabs(Z)/sqrt(2));
#if 0
	T = (double) MINIMUM(Int4,Pos,Neg);
	double ZZ = T-(double)(N*(N+1)/4)/sqrt((N*(N+1)*(2*N+1))/24);
	fprintf(stdout,"POS=%d; NEG=%d; ZZ=%.2lf\n",Pos,Neg,ZZ);
#endif
	return Z;
}

#define WUSAGE "   *** Wilcoxon signed rank test for Direct Coupling Analyses. ***\n\
   Usage: Wilcoxon <infile> <mthd_1> ... <mthd_n> [options]\n\
	    -A     Compute average score for each program.\n\
	    -B     Use the Ball-in-Urn scores instead of joint scores.\n\
	    -H     Output histogram of normalized scores for each method.\n\
	    -v     Verbose output.\n\
\n\n"


int     main(int argc, char *argv[])
{
	Int4	i,j,L,D,d,X,N,arg,R;
	char	c,pdb[10],mthd[10],str[302],chn[20];
	char	*Pdb[20][100];
	double	Key[20][100];
	Int4	Num[20],NumMthd=0,Npdb,start=2;

	double	pval,Z;
	char	*PDB[100];
	float	maxD,F;
	double	score,biu_scr;
	BooLean	Excel_format=FALSE,UseAve=FALSE,UseBIU=FALSE,PutHG=FALSE;
	FILE	*fp=0,*efp=0; 
	if(argc < 2) print_error(WUSAGE);
	char	*Mthd[20];
	for(NumMthd=0,start=2; start < argc; ){
	    if(argv[start][0] != '-'){
		if(!isprint(argv[start][0])) print_error(WUSAGE);
		else { NumMthd++; Mthd[NumMthd]=AllocString(argv[start]); start++; }
		if(NumMthd >= 20) print_error("Input error: too many categories");
	    } else break;
	} Mthd[NumMthd+1]=0;
	if(start < 4) print_error("Input error: too few categories");
	for(arg=start; arg < argc; arg++){
	    if(argv[arg][0] != '-') print_error(WUSAGE);
	    switch(argv[arg][1]) {
		case 'A': UseAve=TRUE; break;   
		case 'B': UseBIU=TRUE; break;   
		case 'e': Excel_format=TRUE; break;   
		case 'H': PutHG=TRUE; break;   
		case 'v': efp=stderr; break;   
		default: print_error(WUSAGE);
	    }
	}
	fp=open_file(argv[1],"","r");
	wcx_typ	**List; NEWP(List,1000,wcx_typ);
	if((c=fgetc(fp)) == 'p') fgets(str,300,fp); else ungetc(c,fp);
	for(N=0,biu_scr=0.0; fgets(str,300,fp) != NULL; ){
	   if(sscanf(str,"%5s\t%s\t%s\t%f\t%lf\tn.a.\t%d\t%d\t%d\t%d\tn.a.\t%f",
			pdb,mthd,chn,&maxD,&score,&L,&D,&X,&d,&F) == 10 ||
		sscanf(str,"%5s\t%s\t%s\t%f\t%lf\t%lf\t%d\t%d\t%d\t%d\t%d\t%f",
			pdb,mthd,chn,&maxD,&score,&biu_scr,&L,&D,&X,&d,&R,&F) == 12)
	   {
// fprintf(stderr,"%s\t%s\t%.2lf\n",pdb,mthd,score);
		N++; List[N] = new wcx_typ(pdb,mthd,maxD,score,L,biu_scr,D,X,d);
		biu_scr=0.0;
	   } else break;
		// List[i]->PutKey(stdout);
	} fclose(fp);
	if(N < 20) print_error("FATAL: too few data points\n");
#if 0 // For excel input...used for MinSqSepICAvsBIU analysis
	for(i=1,Npdb=0; i <= N; i++) List[i]->PutExcel(stdout,5,20); 
	fprintf(stdout,"\n"); exit(1);
#endif
	for(j=1,NumMthd=0; Mthd[j]; j++){ Num[j]=0; NumMthd++; }
	for(i=1,Npdb=0; i <= N; i++){
	  for(j=1; j <= NumMthd; j++){
	    if(List[i]->IsDCA(Mthd[j])){
		Num[j]++; assert(Num[j] < 100); 
		if(UseBIU) Key[j][Num[j]]=List[i]->BIUScore();
		else Key[j][Num[j]]=List[i]->RtnScore();
		Pdb[j][Num[j]]=List[i]->RtnPDB();
		if(j==1){ Npdb++; PDB[Npdb]=List[i]->RtnPDB(); }
		break;
	    }
	  } 
	}
#if 0	//=========== ranks for each protein =============
	dh_type	dH=dheap(NumMthd+5,4);
	char Rank[20];
	for(i=1; i <= Npdb; i++){
	   for(j=1; j <= NumMthd; j++) insrtHeap(j,(keytyp)-Key[j][i],dH);
	   fprintf(stdout,"%s\t",Pdb[1][i]);
	   Rank[NumMthd]=0;
	   double dd=0;
	   for(char x=1; !emptyHeap(dH); x++) {
		assert((j=delminHeap(dH)) != 0);
		Rank[j-1]='0'+ x;
		
		if(x==1) fprintf(stdout,"%c\t",Mthd[j][0]);
		else fprintf(stdout,"%.2f\t%c\t",dd-Key[j][i],Mthd[j][0]);
		dd=Key[j][i];
	   } fprintf(stdout,"%s\n",Rank);
	} Nildheap(dH);
#endif
	for(i=1; i <= NumMthd; i++){
		if(Num[i] != Npdb){ 
		  for(j=1; j <= NumMthd; j++) fprintf(stderr,"N[%d](%s)=%d\n",j,Mthd[j],Num[j]);
		  print_error("Input error: unequal number of trials");
		}
	}
	for(i=1; i <= Npdb; i++){
	    for(j=1; j <= NumMthd; j++){
		if(strcmp(PDB[i],Pdb[j][i]) != 0) print_error("Wilcoxon input error 2");
	    }
	}
#if 0	// print out in pairs
	if(1){
	   Int4 n,m;
	   for(i=1; i < NumMthd; i++){
	      for(j=i+1; j <= NumMthd; j++){
		fprintf(stdout,"\nscores for %s vs %s\n",Mthd[i],Mthd[j]);
		for(n=1; n <= N; n++){
		    if(strcmp(Mthd[i],List[n]->RtnMthd()) != 0) continue;
		    for(m=1; m <= N; m++){
		       if(strcmp(Mthd[j],List[m]->RtnMthd()) != 0) continue;
		       if(strcmp(List[n]->RtnPDB(),List[m]->RtnPDB()) !=0) continue;
		       fprintf(stdout,"%.2lf\t%.2lf\n",List[n]->RtnScore(),List[m]->RtnScore());
		    }
		}
	      }
	   }
	}
#endif
	if(PutHG){	// put histogram of results.
	   Int4 n;
	   for(i=1; i <= NumMthd; i++){
		if(UseBIU) sprintf(str,"ball-in-urn scores for %s",Mthd[i]);
		else sprintf(str,"normalized scores for %s",Mthd[i]);
		h_type	HG=0;
		if(UseBIU) HG=Histogram(str,0,100,1.0); else HG=Histogram(str,0,100,0.25);
// fprintf(stdout,"%s\n",Mthd[i]);
		for(n=1,j=1; j <= N; j++){
		    if(strcmp(Mthd[i],List[j]->RtnMthd()) == 0){
#if 0
			fprintf(stdout,"%.2lf\n",List[j]->RtnScore());
#else
			 List[j]->PutAll(stderr);
			 if(UseBIU) IncdHist(List[j]->RtnBIU(),HG); 
			 else IncdHist(List[j]->RtnScore(),HG); 
#endif
		    }
		} PutHist(stderr,60,HG); NilHist(HG);
	   }
	}
	if(UseAve){
	   double sum=0;
	   fprintf(stdout,"%s",Mthd[1]);    // this is the p-value.
	   for(i=2; i <= NumMthd; i++) fprintf(stdout,"\t%s",Mthd[i]); fprintf(stdout,"\n");
	   for(i=1; i <= NumMthd; i++){
		for(sum=0,j=1; j <= Npdb; j++) sum += Key[i][j]; 
		fprintf(stdout,"%lg\t",sum/(double)Npdb);    // this is the p-value.
	   } fprintf(stdout,"\n");
	} else {
	  fprintf(stdout,"\nMthd1\tMthd2\tZ\tp(2-tailed)\n");
	  for(i=1; i <= NumMthd; i++){
	    for(j=i+1; j <= NumMthd; j++){
		Z=RunWilcoxon(efp,Npdb,PDB,Key[i],Key[j],pval);
		fprintf(stdout,"%s\t%s\t%.3lf\t%lg\n",
			Mthd[i],Mthd[j],Z,pval);    // this is the p-value.
	    }
	  } fprintf(stdout,"N=%d\n",Npdb);
	}
	return 0;
}

